import { useState } from "react";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "./ui/dialog";
import { Button } from "./ui/button";
import { Textarea } from "./ui/textarea";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "./ui/card";
import { Badge } from "./ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "./ui/tabs";
import { Alert, AlertDescription } from "./ui/alert";
import { Progress } from "./ui/progress";
import { 
  Sparkles, 
  Copy, 
  RefreshCw, 
  CheckCircle, 
  AlertTriangle,
  Lightbulb,
  Target,
  Heart,
  Users,
  Zap,
  Wand2,
  BookOpen,
  ThumbsUp,
  Edit3,
  CheckCircle2,
  X
} from "lucide-react";
import { motion, AnimatePresence } from "motion/react";
import { toast } from "sonner";

interface AIWritingAssistantProps {
  isOpen: boolean;
  onClose: () => void;
  currentText: string;
  onApplyText: (text: string) => void;
  type: "bio" | "interests" | "looking-for";
  userInfo?: {
    name?: string;
    age?: number;
    occupation?: string;
    interests?: string[];
  };
}

interface Suggestion {
  id: string;
  title: string;
  text: string;
  type: "improvement" | "alternative" | "enhancement";
  score: number;
  reasons: string[];
}

interface GrammarIssue {
  id: string;
  text: string;
  suggestion: string;
  type: "spelling" | "grammar" | "style" | "clarity";
  position: { start: number; end: number };
}

export function AIWritingAssistant({ 
  isOpen, 
  onClose, 
  currentText, 
  onApplyText, 
  type,
  userInfo 
}: AIWritingAssistantProps) {
  const [activeTab, setActiveTab] = useState("improve");
  const [isGenerating, setIsGenerating] = useState(false);
  const [suggestions, setSuggestions] = useState<Suggestion[]>([]);
  const [grammarIssues, setGrammarIssues] = useState<GrammarIssue[]>([]);
  const [workingText, setWorkingText] = useState(currentText);
  const [appliedSuggestions, setAppliedSuggestions] = useState<string[]>([]);

  // AI Bio Generation Prompts
  const bioPrompts = [
    "Create a warm and approachable bio",
    "Write something fun and playful", 
    "Make it mysterious and intriguing",
    "Focus on adventure and travel",
    "Highlight creativity and arts",
    "Emphasize career and ambition",
    "Show humor and wit",
    "Be romantic and sincere"
  ];

  // Interest Categories
  const interestCategories = {
    "Adventure & Outdoors": ["Hiking", "Rock Climbing", "Surfing", "Camping", "Skiing", "Kayaking"],
    "Arts & Culture": ["Photography", "Painting", "Museums", "Theatre", "Dancing", "Music"],
    "Food & Drinks": ["Cooking", "Wine Tasting", "Coffee", "Foodie Adventures", "Baking", "Craft Beer"],
    "Fitness & Wellness": ["Yoga", "Running", "Gym", "Meditation", "Cycling", "Swimming"],
    "Technology & Gaming": ["Gaming", "Programming", "VR", "Tech Gadgets", "AI", "Robotics"],
    "Learning & Growth": ["Reading", "Languages", "Podcasts", "Documentaries", "Philosophy", "History"]
  };

  // Simulate AI analysis
  const analyzeText = async () => {
    setIsGenerating(true);
    
    // Simulate API delay
    await new Promise(resolve => setTimeout(resolve, 2000));

    if (type === "bio") {
      // Generate bio suggestions
      const bioSuggestions: Suggestion[] = [
        {
          id: "1",
          title: "Enhanced Storytelling",
          text: `Adventure-seeking photographer with a passion for capturing life's beautiful moments 📸 When I'm not behind the lens, you'll find me exploring hidden gems around the city or planning my next travel escape. I believe the best conversations happen over good food and great wine. Looking for someone who shares my love for spontaneous adventures and lazy Sunday mornings.`,
          type: "improvement",
          score: 92,
          reasons: ["Better flow", "More engaging", "Shows personality", "Includes emojis"]
        },
        {
          id: "2", 
          title: "Conversational Approach",
          text: `Hey there! I'm Emma, and I'm all about living life to the fullest. By day, I'm capturing the world through my camera lens, and by night, I'm either trying a new restaurant or getting lost in a good book. I'm looking for someone who can match my energy for adventure but also appreciates quiet moments together. Coffee dates or hiking trails - I'm up for both!`,
          type: "alternative",
          score: 88,
          reasons: ["Friendly tone", "Clear personality", "Balanced interests", "Great conversation starter"]
        },
        {
          id: "3",
          title: "Optimized for Connection",
          text: `Life's too short for ordinary moments! As a photographer, I see beauty everywhere - from golden hour sunsets to the perfect espresso foam art ☕ I'm passionate about travel, love discovering new cuisines, and believe laughter is the best medicine. Seeking a genuine connection with someone who values experiences over possessions and isn't afraid to be authentically themselves.`,
          type: "enhancement", 
          score: 95,
          reasons: ["Memorable opening", "Authentic voice", "Clear values", "Optimized for matches"]
        }
      ];

      // Generate grammar suggestions
      const grammarSuggestions: GrammarIssue[] = [
        {
          id: "g1",
          text: "traveling",
          suggestion: "travelling",
          type: "spelling",
          position: { start: 5, end: 14 }
        },
        {
          id: "g2", 
          text: "Looking for someone to share adventures with!",
          suggestion: "I'm looking for someone special to share life's adventures with.",
          type: "clarity",
          position: { start: 45, end: 89 }
        }
      ];

      setSuggestions(bioSuggestions);
      setGrammarIssues(grammarSuggestions);
    } else if (type === "interests") {
      // Generate interest suggestions based on current interests
      const currentInterests = userInfo?.interests || [];
      const suggestedInterests = [
        "Street Photography", "Coffee Roasting", "Urban Exploration", "Vintage Markets",
        "Food Styling", "Portrait Photography", "Documentary Films", "Art Galleries"
      ].filter(interest => !currentInterests.includes(interest));

      const interestSuggestions: Suggestion[] = [
        {
          id: "i1",
          title: "Photography Enthusiast Package",
          text: suggestedInterests.slice(0, 4).join(", "),
          type: "enhancement",
          score: 90,
          reasons: ["Matches your profession", "Shows depth", "Attractive to like-minded people"]
        },
        {
          id: "i2",
          title: "Creative & Cultural Mix", 
          text: ["Art Galleries", "Vintage Markets", "Documentary Films", "Coffee Roasting"].join(", "),
          type: "improvement",
          score: 85,
          reasons: ["Well-rounded interests", "Great conversation starters", "Shows sophistication"]
        }
      ];

      setSuggestions(interestSuggestions);
      setGrammarIssues([]);
    } else if (type === "looking-for") {
      // Generate "looking for" suggestions
      const lookingForSuggestions: Suggestion[] = [
        {
          id: "lf1",
          title: "Authentic & Positive",
          text: `I'm drawn to genuine connections with someone who values honesty, laughter, and growth. Looking for a partner who's emotionally mature, shares my enthusiasm for life's adventures, and believes in building something beautiful together. Someone who can be my best friend and biggest supporter.`,
          type: "improvement",
          score: 94,
          reasons: ["Positive framing", "Clear values", "Emotional maturity focus", "Partnership oriented"]
        },
        {
          id: "lf2",
          title: "Lifestyle Compatibility",
          text: `Seeking someone who appreciates both spontaneous weekend getaways and cozy nights in. I value ambition, kindness, and a good sense of humor. Looking for a genuine partnership with someone who's ready to explore life together and create our own adventure story.`,
          type: "alternative",
          score: 89,
          reasons: ["Lifestyle balance", "Clear expectations", "Adventure focus", "Partnership mindset"]
        },
        {
          id: "lf3",
          title: "Values-Focused",
          text: `I believe in deep conversations, shared laughter, and authentic connections. Looking for someone who values personal growth, treats others with respect, and isn't afraid to be vulnerable. Age is just a number - what matters is emotional intelligence, integrity, and a zest for life.`,
          type: "enhancement",
          score: 92,
          reasons: ["Values-driven", "Emotional depth", "Authentic connection", "Character focus"]
        }
      ];

      // Generate grammar suggestions for looking-for
      const lookingForGrammar: GrammarIssue[] = [
        {
          id: "lg1",
          text: "someone who has",
          suggestion: "someone with",
          type: "style",
          position: { start: 20, end: 35 }
        },
        {
          id: "lg2",
          text: "I want someone that",
          suggestion: "I'm looking for someone who",
          type: "clarity",
          position: { start: 50, end: 70 }
        }
      ];

      setSuggestions(lookingForSuggestions);
      setGrammarIssues(lookingForGrammar);
    }

    setIsGenerating(false);
  };

  // Handle applying suggestion
  const applySuggestion = (suggestion: Suggestion) => {
    setWorkingText(suggestion.text);
    setAppliedSuggestions([...appliedSuggestions, suggestion.id]);
    toast.success("Suggestion applied successfully!");
  };

  // Copy to clipboard
  const copyToClipboard = (text: string) => {
    navigator.clipboard.writeText(text);
    toast.success("Copied to clipboard!");
  };

  // Apply final text
  const handleApply = () => {
    onApplyText(workingText);
    onClose();
    toast.success("Changes applied to your profile!");
  };

  // Generate fresh suggestions
  const generateFresh = () => {
    analyzeText();
    toast.info("Generating fresh suggestions...");
  };

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-5xl max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <Sparkles className="w-5 h-5 text-purple-600" />
            AI Writing Assistant
            <Badge className="bg-purple-100 text-purple-800 dark:bg-purple-900/20 dark:text-purple-400">
              {type === "bio" ? "About Me" : type === "interests" ? "Interests" : "Looking For"}
            </Badge>
          </DialogTitle>
          <DialogDescription>
            Get AI-powered suggestions to improve your {
              type === "bio" ? "profile bio" : 
              type === "interests" ? "interests" : 
              "looking for section"
            } and attract more meaningful connections.
          </DialogDescription>
        </DialogHeader>

        <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-6">
          <TabsList className="grid w-full grid-cols-3">
            <TabsTrigger value="improve" className="flex items-center gap-2">
              <Wand2 className="w-4 h-4" />
              Improve
            </TabsTrigger>
            <TabsTrigger value="grammar" className="flex items-center gap-2">
              <CheckCircle2 className="w-4 h-4" />
              Grammar & Style
            </TabsTrigger>
            <TabsTrigger value="tips" className="flex items-center gap-2">
              <Lightbulb className="w-4 h-4" />
              Writing Tips
            </TabsTrigger>
          </TabsList>

          {/* Current Text Section */}
          <Card>
            <CardHeader>
              <CardTitle className="text-lg flex items-center gap-2">
                <Edit3 className="w-5 h-5" />
                Your Current {type === "bio" ? "Bio" : type === "interests" ? "Interests" : "Looking For"}
              </CardTitle>
            </CardHeader>
            <CardContent>
              <Textarea
                value={workingText}
                onChange={(e) => setWorkingText(e.target.value)}
                placeholder={
                  type === "bio" ? "Tell others about yourself..." : 
                  type === "interests" ? "List your interests..." :
                  "Describe what you're looking for..."
                }
                className="min-h-[120px] resize-none"
              />
              <div className="flex items-center justify-between mt-2">
                <p className="text-sm text-muted-foreground">
                  {workingText.length}/{type === "bio" ? "500" : type === "interests" ? "200" : "400"} characters
                </p>
                <div className="flex gap-2">
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => copyToClipboard(workingText)}
                  >
                    <Copy className="w-4 h-4 mr-1" />
                    Copy
                  </Button>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={generateFresh}
                    disabled={isGenerating}
                  >
                    <RefreshCw className={`w-4 h-4 mr-1 ${isGenerating ? 'animate-spin' : ''}`} />
                    Analyze
                  </Button>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Improve Tab */}
          <TabsContent value="improve" className="space-y-6">
            {isGenerating ? (
              <Card>
                <CardContent className="p-8 text-center">
                  <motion.div
                    animate={{ rotate: 360 }}
                    transition={{ duration: 2, repeat: Infinity, ease: "linear" }}
                    className="w-12 h-12 mx-auto mb-4"
                  >
                    <Sparkles className="w-12 h-12 text-purple-600" />
                  </motion.div>
                  <h3 className="text-lg font-semibold mb-2">AI is analyzing your text...</h3>
                  <p className="text-muted-foreground mb-4">Generating personalized suggestions based on dating best practices</p>
                  <Progress value={65} className="w-64 mx-auto" />
                </CardContent>
              </Card>
            ) : suggestions.length > 0 ? (
              <div className="space-y-4">
                {suggestions.map((suggestion) => {
                  const isApplied = appliedSuggestions.includes(suggestion.id);
                  return (
                    <motion.div
                      key={suggestion.id}
                      initial={{ opacity: 0, y: 20 }}
                      animate={{ opacity: 1, y: 0 }}
                      transition={{ duration: 0.3 }}
                    >
                      <Card className={`border-2 ${isApplied ? 'border-green-200 bg-green-50 dark:border-green-800 dark:bg-green-900/10' : ''}`}>
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center gap-3">
                              <div className={`p-2 rounded-lg ${
                                suggestion.type === "improvement" ? "bg-blue-100 dark:bg-blue-900/20" :
                                suggestion.type === "alternative" ? "bg-orange-100 dark:bg-orange-900/20" :
                                "bg-purple-100 dark:bg-purple-900/20"
                              }`}>
                                {suggestion.type === "improvement" ? <Target className="w-5 h-5 text-blue-600" /> :
                                 suggestion.type === "alternative" ? <RefreshCw className="w-5 h-5 text-orange-600" /> :
                                 <Zap className="w-5 h-5 text-purple-600" />}
                              </div>
                              <div>
                                <CardTitle className="text-lg">{suggestion.title}</CardTitle>
                                <div className="flex items-center gap-2 mt-1">
                                  <Badge variant="outline" className="text-xs">
                                    {suggestion.type}
                                  </Badge>
                                  <div className="flex items-center gap-1">
                                    <span className="text-sm font-semibold text-green-600">{suggestion.score}%</span>
                                    <span className="text-xs text-muted-foreground">match score</span>
                                  </div>
                                </div>
                              </div>
                            </div>
                            {isApplied && (
                              <CheckCircle className="w-6 h-6 text-green-600" />
                            )}
                          </div>
                        </CardHeader>
                        <CardContent className="space-y-4">
                          <div className="bg-muted/50 rounded-lg p-4">
                            <p className="text-sm leading-relaxed">{suggestion.text}</p>
                          </div>
                          
                          <div className="flex flex-wrap gap-2">
                            {suggestion.reasons.map((reason, index) => (
                              <Badge key={index} variant="secondary" className="text-xs">
                                <ThumbsUp className="w-3 h-3 mr-1" />
                                {reason}
                              </Badge>
                            ))}
                          </div>
                          
                          <div className="flex gap-2">
                            <Button
                              onClick={() => applySuggestion(suggestion)}
                              disabled={isApplied}
                              className="bg-purple-600 hover:bg-purple-700"
                            >
                              {isApplied ? (
                                <>
                                  <CheckCircle className="w-4 h-4 mr-2" />
                                  Applied
                                </>
                              ) : (
                                <>
                                  <Wand2 className="w-4 h-4 mr-2" />
                                  Apply This
                                </>
                              )}
                            </Button>
                            <Button
                              variant="outline"
                              onClick={() => copyToClipboard(suggestion.text)}
                            >
                              <Copy className="w-4 h-4 mr-2" />
                              Copy
                            </Button>
                          </div>
                        </CardContent>
                      </Card>
                    </motion.div>
                  );
                })}
              </div>
            ) : (
              <Card>
                <CardContent className="p-8 text-center">
                  <Sparkles className="w-12 h-12 text-muted-foreground mx-auto mb-4" />
                  <h3 className="text-lg font-semibold mb-2">Ready to improve your {type}?</h3>
                  <p className="text-muted-foreground mb-4">
                    Click "Analyze" to get AI-powered suggestions for better engagement
                  </p>
                  <Button onClick={analyzeText} className="bg-purple-600 hover:bg-purple-700">
                    <Sparkles className="w-4 h-4 mr-2" />
                    Get AI Suggestions
                  </Button>
                </CardContent>
              </Card>
            )}
          </TabsContent>

          {/* Grammar & Style Tab */}
          <TabsContent value="grammar" className="space-y-6">
            {grammarIssues.length > 0 ? (
              <div className="space-y-4">
                {grammarIssues.map((issue) => (
                  <Card key={issue.id}>
                    <CardContent className="p-4">
                      <div className="flex items-start gap-3">
                        <div className={`p-2 rounded-lg mt-1 ${
                          issue.type === "spelling" ? "bg-red-100 dark:bg-red-900/20" :
                          issue.type === "grammar" ? "bg-orange-100 dark:bg-orange-900/20" :
                          "bg-blue-100 dark:bg-blue-900/20"
                        }`}>
                          {issue.type === "spelling" ? <AlertTriangle className="w-4 h-4 text-red-600" /> :
                           <CheckCircle2 className="w-4 h-4 text-blue-600" />}
                        </div>
                        <div className="flex-1">
                          <div className="flex items-center gap-2 mb-2">
                            <Badge variant="outline" className="text-xs capitalize">
                              {issue.type}
                            </Badge>
                          </div>
                          <div className="space-y-2">
                            <div>
                              <span className="text-sm text-muted-foreground">Current: </span>
                              <span className="bg-red-100 dark:bg-red-900/20 px-2 py-1 rounded text-sm">
                                {issue.text}
                              </span>
                            </div>
                            <div>
                              <span className="text-sm text-muted-foreground">Suggested: </span>
                              <span className="bg-green-100 dark:bg-green-900/20 px-2 py-1 rounded text-sm">
                                {issue.suggestion}
                              </span>
                            </div>
                          </div>
                          <div className="flex gap-2 mt-3">
                            <Button size="sm" variant="outline">
                              <CheckCircle className="w-4 h-4 mr-1" />
                              Apply Fix
                            </Button>
                            <Button size="sm" variant="ghost">
                              <X className="w-4 h-4 mr-1" />
                              Ignore
                            </Button>
                          </div>
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            ) : (
              <Card>
                <CardContent className="p-8 text-center">
                  <CheckCircle2 className="w-12 h-12 text-green-600 mx-auto mb-4" />
                  <h3 className="text-lg font-semibold mb-2">Looking great!</h3>
                  <p className="text-muted-foreground">
                    No grammar or style issues detected in your text.
                  </p>
                </CardContent>
              </Card>
            )}
          </TabsContent>

          {/* Writing Tips Tab */}
          <TabsContent value="tips" className="space-y-6">
            <div className="grid gap-4">
              {type === "bio" ? (
                <>
                  <Card>
                    <CardHeader>
                      <CardTitle className="flex items-center gap-2">
                        <Heart className="w-5 h-5 text-red-500" />
                        Bio Writing Best Practices
                      </CardTitle>
                    </CardHeader>
                    <CardContent className="space-y-3">
                      <div className="flex items-start gap-3">
                        <CheckCircle className="w-5 h-5 text-green-600 mt-0.5" />
                        <div>
                          <p className="font-medium">Show, don't tell</p>
                          <p className="text-sm text-muted-foreground">Instead of "I'm funny," share a witty observation or story</p>
                        </div>
                      </div>
                      <div className="flex items-start gap-3">
                        <CheckCircle className="w-5 h-5 text-green-600 mt-0.5" />
                        <div>
                          <p className="font-medium">Be specific</p>
                          <p className="text-sm text-muted-foreground">"Love hiking the trails at Runyon Canyon" vs "I like nature"</p>
                        </div>
                      </div>
                      <div className="flex items-start gap-3">
                        <CheckCircle className="w-5 h-5 text-green-600 mt-0.5" />
                        <div>
                          <p className="font-medium">Include conversation starters</p>
                          <p className="text-sm text-muted-foreground">Mention interests that others can easily respond to</p>
                        </div>
                      </div>
                      <div className="flex items-start gap-3">
                        <CheckCircle className="w-5 h-5 text-green-600 mt-0.5" />
                        <div>
                          <p className="font-medium">Balance is key</p>
                          <p className="text-sm text-muted-foreground">Mix serious interests with fun activities</p>
                        </div>
                      </div>
                    </CardContent>
                  </Card>

                  <Card>
                    <CardHeader>
                      <CardTitle className="flex items-center gap-2">
                        <Users className="w-5 h-5 text-blue-500" />
                        What Attracts People
                      </CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="grid grid-cols-2 gap-4">
                        <div>
                          <h4 className="font-medium text-green-600 mb-2">✓ Do This</h4>
                          <ul className="text-sm space-y-1 text-muted-foreground">
                            <li>• Be authentic and genuine</li>
                            <li>• Show your passions</li>
                            <li>• Use humor appropriately</li>
                            <li>• Mention shared activities</li>
                            <li>• Be positive and upbeat</li>
                          </ul>
                        </div>
                        <div>
                          <h4 className="font-medium text-red-600 mb-2">✗ Avoid This</h4>
                          <ul className="text-sm space-y-1 text-muted-foreground">
                            <li>• Being overly generic</li>
                            <li>• Listing requirements</li>
                            <li>• Negative language</li>
                            <li>• Too much personal info</li>
                            <li>• Clichés and overused phrases</li>
                          </ul>
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                </>
              ) : type === "interests" ? (
                <Card>
                  <CardHeader>
                    <CardTitle className="flex items-center gap-2">
                      <Lightbulb className="w-5 h-5 text-yellow-500" />
                      Interest Selection Tips
                    </CardTitle>
                  </CardHeader>
                  <CardContent className="space-y-4">
                    <div className="grid gap-3">
                      {Object.entries(interestCategories).map(([category, items]) => (
                        <div key={category}>
                          <h4 className="font-medium mb-2">{category}</h4>
                          <div className="flex flex-wrap gap-2">
                            {items.map((interest) => (
                              <Badge key={interest} variant="outline" className="text-xs">
                                {interest}
                              </Badge>
                            ))}
                          </div>
                        </div>
                      ))}
                    </div>
                    
                    <Alert>
                      <Lightbulb className="w-4 h-4" />
                      <AlertDescription>
                        <strong>Pro tip:</strong> Choose 6-8 diverse interests that show different sides of your personality. 
                        Mix popular activities with unique hobbies to stand out!
                      </AlertDescription>
                    </Alert>
                  </CardContent>
                </Card>
              ) : (
                <>
                  <Card>
                    <CardHeader>
                      <CardTitle className="flex items-center gap-2">
                        <Target className="w-5 h-5 text-blue-500" />
                        "Looking For" Best Practices
                      </CardTitle>
                    </CardHeader>
                    <CardContent className="space-y-3">
                      <div className="flex items-start gap-3">
                        <CheckCircle className="w-5 h-5 text-green-600 mt-0.5" />
                        <div>
                          <p className="font-medium">Focus on what you want, not what you don't</p>
                          <p className="text-sm text-muted-foreground">Say "I value honesty" instead of "No liars"</p>
                        </div>
                      </div>
                      <div className="flex items-start gap-3">
                        <CheckCircle className="w-5 h-5 text-green-600 mt-0.5" />
                        <div>
                          <p className="font-medium">Describe the relationship, not just the person</p>
                          <p className="text-sm text-muted-foreground">Talk about building something together, shared goals</p>
                        </div>
                      </div>
                      <div className="flex items-start gap-3">
                        <CheckCircle className="w-5 h-5 text-green-600 mt-0.5" />
                        <div>
                          <p className="font-medium">Be realistic but optimistic</p>
                          <p className="text-sm text-muted-foreground">Show what you bring to the table while expressing your needs</p>
                        </div>
                      </div>
                      <div className="flex items-start gap-3">
                        <CheckCircle className="w-5 h-5 text-green-600 mt-0.5" />
                        <div>
                          <p className="font-medium">Include emotional intelligence</p>
                          <p className="text-sm text-muted-foreground">Mention qualities like empathy, communication, growth mindset</p>
                        </div>
                      </div>
                    </CardContent>
                  </Card>

                  <Card>
                    <CardHeader>
                      <CardTitle className="flex items-center gap-2">
                        <Heart className="w-5 h-5 text-red-500" />
                        Attractive vs. Unattractive Language
                      </CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="grid grid-cols-2 gap-4">
                        <div>
                          <h4 className="font-medium text-green-600 mb-2">✓ Attractive</h4>
                          <ul className="text-sm space-y-1 text-muted-foreground">
                            <li>• "I value kindness and authenticity"</li>
                            <li>• "Building something meaningful together"</li>
                            <li>• "Someone who challenges me to grow"</li>
                            <li>• "Shared adventures and deep conversations"</li>
                            <li>• "Emotional maturity and communication"</li>
                          </ul>
                        </div>
                        <div>
                          <h4 className="font-medium text-red-600 mb-2">✗ Unattractive</h4>
                          <ul className="text-sm space-y-1 text-muted-foreground">
                            <li>• "No games, no drama"</li>
                            <li>• "Must have a job and car"</li>
                            <li>• "If you're not serious, swipe left"</li>
                            <li>• "I don't have time for immature people"</li>
                            <li>• "My last relationship was a disaster"</li>
                          </ul>
                        </div>
                      </div>
                    </CardContent>
                  </Card>

                  <Card>
                    <CardHeader>
                      <CardTitle className="flex items-center gap-2">
                        <Users className="w-5 h-5 text-purple-500" />
                        Relationship Goals Framework
                      </CardTitle>
                    </CardHeader>
                    <CardContent className="space-y-3">
                      <div className="grid gap-2">
                        <div className="bg-blue-50 dark:bg-blue-900/20 p-3 rounded-lg border border-blue-200 dark:border-blue-800">
                          <h5 className="font-medium text-blue-700 dark:text-blue-400 mb-1">Partnership Qualities</h5>
                          <p className="text-sm text-blue-600 dark:text-blue-500">What makes a good teammate in life</p>
                        </div>
                        <div className="bg-green-50 dark:bg-green-900/20 p-3 rounded-lg border border-green-200 dark:border-green-800">
                          <h5 className="font-medium text-green-700 dark:text-green-400 mb-1">Lifestyle Compatibility</h5>
                          <p className="text-sm text-green-600 dark:text-green-500">How you want to spend time together</p>
                        </div>
                        <div className="bg-purple-50 dark:bg-purple-900/20 p-3 rounded-lg border border-purple-200 dark:border-purple-800">
                          <h5 className="font-medium text-purple-700 dark:text-purple-400 mb-1">Shared Values</h5>
                          <p className="text-sm text-purple-600 dark:text-purple-500">What matters most in life and relationships</p>
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                </>
              )}
            </div>
          </TabsContent>
        </Tabs>

        {/* Footer Actions */}
        <div className="flex items-center justify-between pt-6 border-t">
          <Button variant="outline" onClick={onClose}>
            Cancel
          </Button>
          <div className="flex gap-2">
            <Button variant="outline" onClick={() => setWorkingText(currentText)}>
              Reset to Original
            </Button>
            <Button onClick={handleApply} className="bg-purple-600 hover:bg-purple-700">
              <CheckCircle className="w-4 h-4 mr-2" />
              Apply Changes
            </Button>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}