import { useState } from "react";
import { Button } from "./ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "./ui/card";
import { Switch } from "./ui/switch";
import { Alert, AlertDescription } from "./ui/alert";
import { Badge } from "./ui/badge";
import { 
  Lock, 
  Mail, 
  Phone, 
  Shield, 
  Verified, 
  CheckCircle, 
  AlertTriangle,
  ChevronRight,
  Clock,
  Eye
} from "lucide-react";

interface AccountTabWithVerificationProps {
  onNavigate: (view: string) => void;
}

export function AccountTabWithVerification({ onNavigate }: AccountTabWithVerificationProps) {
  const [accountSettings, setAccountSettings] = useState({
    twoFactor: false,
    emailVerified: true,
    phoneVerified: false,
    isVerified: false, // Identity verification status
    verificationStatus: 'not_started' // 'not_started', 'pending', 'approved', 'rejected'
  });

  const getVerificationStatusBadge = () => {
    switch (accountSettings.verificationStatus) {
      case 'approved':
        return (
          <Badge className="bg-green-500 text-white">
            <CheckCircle className="w-3 h-3 mr-1" />
            Verified
          </Badge>
        );
      case 'pending':
        return (
          <Badge className="bg-yellow-500 text-white">
            <Clock className="w-3 h-3 mr-1" />
            Pending Review
          </Badge>
        );
      case 'rejected':
        return (
          <Badge className="bg-red-500 text-white">
            <AlertTriangle className="w-3 h-3 mr-1" />
            Rejected
          </Badge>
        );
      default:
        return (
          <Badge variant="outline">
            <Shield className="w-3 h-3 mr-1" />
            Not Verified
          </Badge>
        );
    }
  };

  const getVerificationDescription = () => {
    switch (accountSettings.verificationStatus) {
      case 'approved':
        return "Your identity has been verified. You now have a verified badge on your profile.";
      case 'pending':
        return "Your verification request is being reviewed. We'll notify you once it's processed.";
      case 'rejected':
        return "Your verification request was rejected. You can submit a new request with different documents.";
      default:
        return "Verify your identity to gain trust and credibility. Verified members get increased visibility.";
    }
  };

  return (
    <div className="space-y-6">
      {/* Account Security */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Lock className="w-5 h-5" />
            Account Security
          </CardTitle>
          <CardDescription>
            Manage your account security settings and verification status
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-6">
          {/* Two-Factor Authentication */}
          <div className="flex items-center justify-between">
            <div className="space-y-1">
              <p className="font-medium">Two-Factor Authentication</p>
              <p className="text-sm text-muted-foreground">
                Add an extra layer of security to your account
              </p>
            </div>
            <Switch
              checked={accountSettings.twoFactor}
              onCheckedChange={(checked) => 
                setAccountSettings({...accountSettings, twoFactor: checked})
              }
            />
          </div>

          {/* Email Verification */}
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-3">
              <Mail className="w-5 h-5 text-muted-foreground" />
              <div className="space-y-1">
                <p className="font-medium">Email Verification</p>
                <p className="text-sm text-muted-foreground">
                  Verify your email address for security
                </p>
              </div>
            </div>
            <div className="flex items-center gap-2">
              {accountSettings.emailVerified ? (
                <Badge className="bg-green-100 text-green-800">
                  <CheckCircle className="w-3 h-3 mr-1" />
                  Verified
                </Badge>
              ) : (
                <Button size="sm" variant="outline">
                  Verify Email
                </Button>
              )}
            </div>
          </div>

          {/* Phone Verification */}
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-3">
              <Phone className="w-5 h-5 text-muted-foreground" />
              <div className="space-y-1">
                <p className="font-medium">Phone Verification</p>
                <p className="text-sm text-muted-foreground">
                  Verify your phone number for added security
                </p>
              </div>
            </div>
            <div className="flex items-center gap-2">
              {accountSettings.phoneVerified ? (
                <Badge className="bg-green-100 text-green-800">
                  <CheckCircle className="w-3 h-3 mr-1" />
                  Verified
                </Badge>
              ) : (
                <Button size="sm" variant="outline">
                  Verify Phone
                </Button>
              )}
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Identity Verification */}
      <Card className="border-2 border-blue-200 dark:border-blue-800">
        <CardHeader className="bg-blue-50 dark:bg-blue-900/20">
          <CardTitle className="flex items-center gap-2">
            <Verified className="w-5 h-5 text-blue-600" />
            Identity Verification
          </CardTitle>
          <CardDescription>
            Verify your identity with official documents to get a verified badge
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-4 pt-6">
          <div className="flex items-center justify-between">
            <div className="space-y-1">
              <div className="flex items-center gap-2">
                <p className="font-medium">Verification Status</p>
                {getVerificationStatusBadge()}
              </div>
              <p className="text-sm text-muted-foreground">
                {getVerificationDescription()}
              </p>
            </div>
            <div className="flex items-center gap-2">
              {accountSettings.verificationStatus === 'not_started' && (
                <Button 
                  onClick={() => onNavigate("verification")}
                  className="bg-blue-600 hover:bg-blue-700"
                >
                  <Shield className="w-4 h-4 mr-2" />
                  Start Verification
                  <ChevronRight className="w-4 h-4 ml-1" />
                </Button>
              )}
              {accountSettings.verificationStatus === 'pending' && (
                <Button 
                  variant="outline"
                  onClick={() => onNavigate("verification")}
                >
                  <Eye className="w-4 h-4 mr-2" />
                  View Status
                </Button>
              )}
              {accountSettings.verificationStatus === 'rejected' && (
                <Button 
                  onClick={() => onNavigate("verification")}
                  className="bg-blue-600 hover:bg-blue-700"
                >
                  <Shield className="w-4 h-4 mr-2" />
                  Try Again
                </Button>
              )}
              {accountSettings.verificationStatus === 'approved' && (
                <Button 
                  variant="outline"
                  onClick={() => onNavigate("verification")}
                >
                  <Eye className="w-4 h-4 mr-2" />
                  View Details
                </Button>
              )}
            </div>
          </div>

          {/* Benefits of Verification */}
          <Alert>
            <Verified className="h-4 w-4" />
            <AlertDescription>
              <strong>Benefits of verification:</strong> Verified badge on your profile, 
              increased trust from other members, higher visibility in search results, 
              and access to premium features.
            </AlertDescription>
          </Alert>

          {/* Verification Process */}
          <div className="space-y-3">
            <h4 className="font-medium text-sm">Verification Process:</h4>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4 text-sm">
              <div className="flex items-center gap-2">
                <div className="w-6 h-6 bg-blue-100 dark:bg-blue-900 rounded-full flex items-center justify-center text-xs font-bold text-blue-600 dark:text-blue-400">
                  1
                </div>
                <span className="text-muted-foreground">Upload valid ID</span>
              </div>
              <div className="flex items-center gap-2">
                <div className="w-6 h-6 bg-blue-100 dark:bg-blue-900 rounded-full flex items-center justify-center text-xs font-bold text-blue-600 dark:text-blue-400">
                  2
                </div>
                <span className="text-muted-foreground">Admin review (24-48h)</span>
              </div>
              <div className="flex items-center gap-2">
                <div className="w-6 h-6 bg-blue-100 dark:bg-blue-900 rounded-full flex items-center justify-center text-xs font-bold text-blue-600 dark:text-blue-400">
                  3
                </div>
                <span className="text-muted-foreground">Get verified badge</span>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Account Actions */}
      <Card>
        <CardHeader>
          <CardTitle>Account Actions</CardTitle>
          <CardDescription>
            Manage your account and data
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <Button variant="outline" className="w-full justify-start">
            <Shield className="w-4 h-4 mr-2" />
            Change Password
          </Button>
          <Button variant="outline" className="w-full justify-start">
            <Mail className="w-4 h-4 mr-2" />
            Update Email Address
          </Button>
          <Button variant="outline" className="w-full justify-start text-red-600 border-red-200 hover:bg-red-50">
            <AlertTriangle className="w-4 h-4 mr-2" />
            Deactivate Account
          </Button>
        </CardContent>
      </Card>
    </div>
  );
}