import { useState } from "react";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "./ui/dialog";
import { Button } from "./ui/button";
import { Input } from "./ui/input";
import { Label } from "./ui/label";
import { Textarea } from "./ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "./ui/select";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "./ui/card";
import { Badge } from "./ui/badge";
import { Alert, AlertDescription } from "./ui/alert";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "./ui/tabs";
import { 
  Upload, 
  Video, 
  Image as ImageIcon, 
  FileText, 
  DollarSign, 
  Calendar, 
  CreditCard, 
  CheckCircle,
  Sparkles,
  Wand2,
  Clock,
  AlertTriangle
} from "lucide-react";
import { motion } from "motion/react";
import { ImageWithFallback } from "./figma/ImageWithFallback";
import { useUser } from "./UserProvider";

interface AdCreationModalProps {
  isOpen: boolean;
  onClose: () => void;
}

interface AdPricing {
  video: { [key: string]: number };
  photo: { [key: string]: number };
  text: { [key: string]: number };
}

interface AdData {
  type: "video" | "photo" | "text";
  title: string;
  description: string;
  category: "services" | "products" | "dating-tips";
  location?: string;
  contactInfo: string;
  duration: string;
  content?: File;
  imageUrl?: string;
  videoUrl?: string;
}

export function AdCreationModal({ isOpen, onClose }: AdCreationModalProps) {
  const { user } = useUser();
  const [currentStep, setCurrentStep] = useState(1);
  const [adData, setAdData] = useState<AdData>({
    type: "photo",
    title: "",
    description: "",
    category: "services",
    location: "",
    contactInfo: "",
    duration: "7"
  });
  const [isAIGenerating, setIsAIGenerating] = useState(false);
  const [showCheckout, setShowCheckout] = useState(false);

  // Mock pricing data - in real app this would come from admin settings
  const pricing: AdPricing = {
    video: { "7": 49.99, "14": 89.99, "30": 149.99, "60": 249.99 },
    photo: { "7": 29.99, "14": 54.99, "30": 89.99, "60": 149.99 },
    text: { "7": 19.99, "14": 34.99, "30": 59.99, "60": 99.99 }
  };

  const adTypes = [
    {
      id: "video",
      label: "Video Ad",
      icon: Video,
      description: "Upload a video (max 60 seconds)",
      maxDuration: "60 seconds",
      popular: true
    },
    {
      id: "photo",
      label: "Photo Ad", 
      icon: ImageIcon,
      description: "Upload an eye-catching image",
      maxDuration: "Unlimited",
      popular: false
    },
    {
      id: "text",
      label: "Text Ad",
      icon: FileText,
      description: "Create a compelling text-only ad",
      maxDuration: "500 characters",
      popular: false
    }
  ];

  const durations = [
    { value: "7", label: "7 Days" },
    { value: "14", label: "14 Days" },
    { value: "30", label: "30 Days", popular: true },
    { value: "60", label: "60 Days" }
  ];

  const categories = [
    { value: "services", label: "Dating Services" },
    { value: "products", label: "Products" },
    { value: "dating-tips", label: "Dating Tips & Coaching" }
  ];

  const getCurrentPrice = () => {
    return pricing[adData.type][adData.duration] || 0;
  };

  const handleAIGenerate = async () => {
    setIsAIGenerating(true);
    // Simulate AI generation
    await new Promise(resolve => setTimeout(resolve, 2000));
    
    const aiSuggestions = {
      services: {
        title: "Professional Dating Photography - Get Results!",
        description: "Transform your dating profile with stunning professional photos that get 3x more matches! Award-winning photographer with 500+ success stories. Book your session today and start attracting the right person!"
      },
      products: {
        title: "Premium Date Night Cologne - Irresistible Scent",
        description: "Make an unforgettable first impression with our signature cologne blend. Scientifically formulated to boost confidence and attraction. Join thousands of satisfied customers who've found love!"
      },
      "dating-tips": {
        title: "Master the Art of Dating - Expert Coaching",
        description: "Unlock your dating potential with proven strategies from certified relationship experts. 1-on-1 coaching that transforms your approach to dating. See results in just 30 days or money back!"
      }
    };

    const suggestion = aiSuggestions[adData.category];
    setAdData({
      ...adData,
      title: suggestion.title,
      description: suggestion.description
    });
    
    setIsAIGenerating(false);
  };

  const handleFileUpload = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (file) {
      setAdData({ ...adData, content: file });
    }
  };

  const handleSubmitAd = () => {
    // In real app, this would create the ad and initiate checkout
    setShowCheckout(true);
  };

  const handlePayment = (method: string) => {
    // In real app, this would integrate with payment providers
    console.log(`Processing payment via ${method} for $${getCurrentPrice()}`);
    // Simulate payment success
    setTimeout(() => {
      setCurrentStep(4);
      setShowCheckout(false);
    }, 2000);
  };

  const resetModal = () => {
    setCurrentStep(1);
    setAdData({
      type: "photo",
      title: "",
      description: "",
      category: "services",
      location: "",
      contactInfo: "",
      duration: "7"
    });
    setShowCheckout(false);
    onClose();
  };

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <Sparkles className="w-5 h-5 text-orange-600" />
            Create Your Ad
          </DialogTitle>
          <DialogDescription>
            Reach thousands of singles on SinglesZonne with your ad
          </DialogDescription>
        </DialogHeader>

        {/* Progress Steps */}
        <div className="flex items-center justify-center space-x-4 mb-6">
          {[1, 2, 3, 4].map((step) => (
            <div key={step} className="flex items-center">
              <div className={`w-8 h-8 rounded-full flex items-center justify-center text-sm font-medium ${
                step <= currentStep ? 'bg-orange-600 text-white' : 'bg-gray-200 text-gray-600'
              }`}>
                {step < currentStep ? <CheckCircle className="w-4 h-4" /> : step}
              </div>
              {step < 4 && (
                <div className={`w-12 h-0.5 ${
                  step < currentStep ? 'bg-orange-600' : 'bg-gray-200'
                }`} />
              )}
            </div>
          ))}
        </div>

        {/* Step Content */}
        {currentStep === 1 && (
          <motion.div
            initial={{ opacity: 0, x: 20 }}
            animate={{ opacity: 1, x: 0 }}
            className="space-y-6"
          >
            <div>
              <h3 className="text-lg font-semibold mb-4">Choose Your Ad Type</h3>
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                {adTypes.map((type) => {
                  const Icon = type.icon;
                  return (
                    <Card 
                      key={type.id}
                      className={`cursor-pointer transition-all hover:shadow-md ${
                        adData.type === type.id ? 'ring-2 ring-orange-600 bg-orange-50 dark:bg-orange-900/20' : ''
                      }`}
                      onClick={() => setAdData({ ...adData, type: type.id as any })}
                    >
                      <CardHeader className="text-center">
                        <div className="relative">
                          <Icon className="w-12 h-12 mx-auto text-orange-600 mb-2" />
                          {type.popular && (
                            <Badge className="absolute -top-2 -right-2 bg-green-600 text-white text-xs">
                              Popular
                            </Badge>
                          )}
                        </div>
                        <CardTitle className="text-base">{type.label}</CardTitle>
                        <CardDescription className="text-sm">{type.description}</CardDescription>
                      </CardHeader>
                      <CardContent className="text-center">
                        <div className="text-xs text-muted-foreground">
                          Max: {type.maxDuration}
                        </div>
                      </CardContent>
                    </Card>
                  );
                })}
              </div>
            </div>

            <div>
              <h3 className="text-lg font-semibold mb-4">Ad Duration & Pricing</h3>
              <div className="grid grid-cols-2 md:grid-cols-4 gap-3">
                {durations.map((duration) => (
                  <Card 
                    key={duration.value}
                    className={`cursor-pointer transition-all hover:shadow-md ${
                      adData.duration === duration.value ? 'ring-2 ring-orange-600 bg-orange-50 dark:bg-orange-900/20' : ''
                    }`}
                    onClick={() => setAdData({ ...adData, duration: duration.value })}
                  >
                    <CardContent className="p-4 text-center">
                      <div className="relative">
                        <div className="font-medium">{duration.label}</div>
                        <div className="text-lg font-bold text-orange-600">
                          ${pricing[adData.type][duration.value]}
                        </div>
                        {duration.popular && (
                          <Badge className="absolute -top-2 -right-2 bg-blue-600 text-white text-xs">
                            Best Value
                          </Badge>
                        )}
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            </div>

            <div className="flex justify-end">
              <Button onClick={() => setCurrentStep(2)} className="bg-orange-600 hover:bg-orange-700">
                Continue
              </Button>
            </div>
          </motion.div>
        )}

        {currentStep === 2 && (
          <motion.div
            initial={{ opacity: 0, x: 20 }}
            animate={{ opacity: 1, x: 0 }}
            className="space-y-6"
          >
            <div className="flex items-center justify-between">
              <h3 className="text-lg font-semibold">Create Your Ad Content</h3>
              <Button
                onClick={handleAIGenerate}
                disabled={isAIGenerating}
                variant="outline"
                className="flex items-center gap-2"
              >
                <Wand2 className="w-4 h-4" />
                {isAIGenerating ? "Generating..." : "AI Assist"}
              </Button>
            </div>

            {isAIGenerating && (
              <Alert>
                <Sparkles className="w-4 h-4" />
                <AlertDescription>
                  AI is creating amazing content for your ad based on best practices...
                </AlertDescription>
              </Alert>
            )}

            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div className="space-y-4">
                <div>
                  <Label htmlFor="category">Category</Label>
                  <Select value={adData.category} onValueChange={(value: any) => setAdData({ ...adData, category: value })}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      {categories.map((cat) => (
                        <SelectItem key={cat.value} value={cat.value}>
                          {cat.label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>

                <div>
                  <Label htmlFor="title">Ad Title</Label>
                  <Input
                    id="title"
                    value={adData.title}
                    onChange={(e) => setAdData({ ...adData, title: e.target.value })}
                    placeholder="Create an attention-grabbing title"
                    maxLength={80}
                  />
                  <div className="text-xs text-muted-foreground mt-1">
                    {adData.title.length}/80 characters
                  </div>
                </div>

                <div>
                  <Label htmlFor="description">Description</Label>
                  <Textarea
                    id="description"
                    value={adData.description}
                    onChange={(e) => setAdData({ ...adData, description: e.target.value })}
                    placeholder="Describe your offering in detail"
                    rows={4}
                    maxLength={500}
                  />
                  <div className="text-xs text-muted-foreground mt-1">
                    {adData.description.length}/500 characters
                  </div>
                </div>

                <div>
                  <Label htmlFor="location">Location (Optional)</Label>
                  <Input
                    id="location"
                    value={adData.location}
                    onChange={(e) => setAdData({ ...adData, location: e.target.value })}
                    placeholder="City, State"
                  />
                </div>

                <div>
                  <Label htmlFor="contact">Contact Information</Label>
                  <Input
                    id="contact"
                    value={adData.contactInfo}
                    onChange={(e) => setAdData({ ...adData, contactInfo: e.target.value })}
                    placeholder="Website, email, or phone"
                    required
                  />
                </div>
              </div>

              <div className="space-y-4">
                {/* Media Upload */}
                {adData.type !== "text" && (
                  <div>
                    <Label>Upload {adData.type === "video" ? "Video" : "Image"}</Label>
                    <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center hover:border-orange-400 transition-colors">
                      <input
                        type="file"
                        accept={adData.type === "video" ? "video/*" : "image/*"}
                        onChange={handleFileUpload}
                        className="hidden"
                        id="media-upload"
                      />
                      <label htmlFor="media-upload" className="cursor-pointer">
                        <Upload className="w-8 h-8 mx-auto text-gray-400 mb-2" />
                        <div className="text-sm text-gray-600">
                          Click to upload {adData.type === "video" ? "video" : "image"}
                        </div>
                        <div className="text-xs text-gray-500 mt-1">
                          {adData.type === "video" ? "Max 60 seconds, MP4/MOV" : "JPG, PNG, max 5MB"}
                        </div>
                      </label>
                    </div>
                    {adData.content && (
                      <div className="text-sm text-green-600 mt-2 flex items-center gap-1">
                        <CheckCircle className="w-4 h-4" />
                        {adData.content.name} uploaded
                      </div>
                    )}
                  </div>
                )}

                {/* Ad Preview */}
                <div>
                  <Label>Preview</Label>
                  <Card className="p-4">
                    <div className="space-y-2">
                      <h4 className="font-semibold text-sm">{adData.title || "Your Ad Title"}</h4>
                      <p className="text-xs text-muted-foreground">
                        {adData.description || "Your ad description will appear here..."}
                      </p>
                      {adData.location && (
                        <div className="text-xs text-muted-foreground flex items-center gap-1">
                          <span>📍 {adData.location}</span>
                        </div>
                      )}
                      <Badge variant="secondary" className="text-xs">
                        {categories.find(c => c.value === adData.category)?.label}
                      </Badge>
                    </div>
                  </Card>
                </div>
              </div>
            </div>

            <div className="flex justify-between">
              <Button variant="outline" onClick={() => setCurrentStep(1)}>
                Back
              </Button>
              <Button 
                onClick={() => setCurrentStep(3)} 
                className="bg-orange-600 hover:bg-orange-700"
                disabled={!adData.title || !adData.description || !adData.contactInfo}
              >
                Review & Pay
              </Button>
            </div>
          </motion.div>
        )}

        {currentStep === 3 && !showCheckout && (
          <motion.div
            initial={{ opacity: 0, x: 20 }}
            animate={{ opacity: 1, x: 0 }}
            className="space-y-6"
          >
            <h3 className="text-lg font-semibold">Review Your Ad</h3>
            
            <Card>
              <CardHeader>
                <div className="flex items-center justify-between">
                  <CardTitle className="text-base">{adData.title}</CardTitle>
                  <Badge>{adData.type.charAt(0).toUpperCase() + adData.type.slice(1)} Ad</Badge>
                </div>
              </CardHeader>
              <CardContent className="space-y-3">
                <p className="text-sm">{adData.description}</p>
                {adData.location && (
                  <div className="text-sm text-muted-foreground flex items-center gap-1">
                    <span>📍 {adData.location}</span>
                  </div>
                )}
                <div className="text-sm">
                  <strong>Contact:</strong> {adData.contactInfo}
                </div>
                <div className="flex items-center gap-4 text-sm">
                  <span className="flex items-center gap-1">
                    <Calendar className="w-4 h-4" />
                    {durations.find(d => d.value === adData.duration)?.label}
                  </span>
                  <span className="flex items-center gap-1">
                    <DollarSign className="w-4 h-4" />
                    ${getCurrentPrice()}
                  </span>
                </div>
              </CardContent>
            </Card>

            <Alert>
              <AlertTriangle className="w-4 h-4" />
              <AlertDescription>
                Your ad will be reviewed by our team before going live. This typically takes 1-2 business days. 
                You'll only be charged once your ad is approved.
              </AlertDescription>
            </Alert>

            <div className="flex justify-between">
              <Button variant="outline" onClick={() => setCurrentStep(2)}>
                Edit Ad
              </Button>
              <Button onClick={handleSubmitAd} className="bg-orange-600 hover:bg-orange-700">
                Proceed to Payment
              </Button>
            </div>
          </motion.div>
        )}

        {showCheckout && (
          <motion.div
            initial={{ opacity: 0, scale: 0.95 }}
            animate={{ opacity: 1, scale: 1 }}
            className="space-y-6"
          >
            <h3 className="text-lg font-semibold text-center">Secure Checkout</h3>
            
            <Card className="bg-gradient-to-r from-orange-50 to-orange-100 dark:from-orange-900/20 dark:to-orange-800/20">
              <CardContent className="p-6 text-center">
                <div className="text-2xl font-bold text-orange-600">
                  ${getCurrentPrice()}
                </div>
                <div className="text-sm text-muted-foreground">
                  {adData.type.charAt(0).toUpperCase() + adData.type.slice(1)} Ad • {durations.find(d => d.value === adData.duration)?.label}
                </div>
              </CardContent>
            </Card>

            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              <Button
                onClick={() => handlePayment("paypal")}
                className="bg-blue-600 hover:bg-blue-700 h-12"
              >
                <span className="font-bold">Pay</span>Pal
              </Button>
              <Button
                onClick={() => handlePayment("stripe")}
                className="bg-purple-600 hover:bg-purple-700 h-12"
              >
                Stripe
              </Button>
              <Button
                onClick={() => handlePayment("credit-card")}
                variant="outline"
                className="h-12"
              >
                <CreditCard className="w-4 h-4 mr-2" />
                Credit Card
              </Button>
            </div>

            <div className="text-center">
              <Button variant="ghost" onClick={() => setShowCheckout(false)}>
                Cancel
              </Button>
            </div>
          </motion.div>
        )}

        {currentStep === 4 && (
          <motion.div
            initial={{ opacity: 0, scale: 0.95 }}
            animate={{ opacity: 1, scale: 1 }}
            className="text-center space-y-6"
          >
            <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto">
              <CheckCircle className="w-8 h-8 text-green-600" />
            </div>
            <div>
              <h3 className="text-xl font-semibold text-green-600">Ad Submitted Successfully!</h3>
              <p className="text-muted-foreground mt-2">
                Your ad is now under review. You'll receive an email notification once it's approved and goes live.
              </p>
            </div>
            <div className="bg-gray-50 dark:bg-gray-800 rounded-lg p-4">
              <h4 className="font-medium mb-2">What happens next?</h4>
              <ul className="text-sm text-left space-y-1 text-muted-foreground">
                <li>• Our team reviews your ad (1-2 business days)</li>
                <li>• You'll receive email confirmation once approved</li>
                <li>• Your ad goes live and starts reaching singles</li>
                <li>• Track performance in your account dashboard</li>
              </ul>
            </div>
            <Button onClick={resetModal} className="bg-orange-600 hover:bg-orange-700">
              Create Another Ad
            </Button>
          </motion.div>
        )}
      </DialogContent>
    </Dialog>
  );
}