import { useState, useEffect, useRef } from "react";
import { Dialog, DialogContent } from "./ui/dialog";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Progress } from "./ui/progress";
import { Card, CardContent } from "./ui/card";
import { 
  Play, 
  Pause, 
  Volume2, 
  VolumeX, 
  X, 
  Coins, 
  Eye, 
  MousePointer,
  Timer,
  CheckCircle,
  Gift
} from "lucide-react";
import { motion, AnimatePresence } from "motion/react";
import { ImageWithFallback } from "./figma/ImageWithFallback";
import { usePoints } from "./PointsProvider";
import { toast } from "sonner";

interface AdData {
  id: number;
  title: string;
  description: string;
  type: "video" | "photo" | "text";
  category: "services" | "products" | "dating-tips";
  location?: string;
  contactInfo: string;
  duration: string;
  price: number;
  pointsReward: number;
  imageUrl?: string;
  videoUrl?: string;
  impressions: number;
  clicks: number;
}

interface AdViewModalProps {
  isOpen: boolean;
  onClose: () => void;
  ad: AdData | null;
}

export function AdViewModal({ isOpen, onClose, ad }: AdViewModalProps) {
  const { earnPoints } = usePoints();
  const [isPlaying, setIsPlaying] = useState(false);
  const [isMuted, setIsMuted] = useState(false);
  const [currentTime, setCurrentTime] = useState(0);
  const [duration, setDuration] = useState(0);
  const [hasEarnedPoints, setHasEarnedPoints] = useState(false);
  const [viewStartTime, setViewStartTime] = useState<number | null>(null);
  const [minimumViewTime] = useState(3); // Minimum seconds to view for points
  const [showPointsEarned, setShowPointsEarned] = useState(false);
  
  const videoRef = useRef<HTMLVideoElement>(null);
  const viewTimerRef = useRef<NodeJS.Timeout>();

  useEffect(() => {
    if (isOpen && ad) {
      setViewStartTime(Date.now());
      setHasEarnedPoints(false);
      setCurrentTime(0);
      setShowPointsEarned(false);
      
      // For non-video ads, start timer immediately
      if (ad.type !== "video") {
        viewTimerRef.current = setTimeout(() => {
          awardPoints();
        }, minimumViewTime * 1000);
      }
      
      // Track impression
      trackImpression();
    }

    return () => {
      if (viewTimerRef.current) {
        clearTimeout(viewTimerRef.current);
      }
    };
  }, [isOpen, ad]);

  const trackImpression = () => {
    if (ad) {
      // In real app, would send to analytics
      console.log(`Impression tracked for ad ${ad.id}`);
    }
  };

  const trackClick = () => {
    if (ad) {
      // In real app, would send to analytics
      console.log(`Click tracked for ad ${ad.id}`);
    }
  };

  const awardPoints = () => {
    if (!hasEarnedPoints && ad) {
      setHasEarnedPoints(true);
      earnPoints(ad.pointsReward, `Viewed ${ad.type} ad: ${ad.title}`);
      setShowPointsEarned(true);
      
      toast.success(`+${ad.pointsReward} points earned!`, {
        description: `You watched the ad and earned points!`,
        duration: 3000,
      });

      // Hide the points notification after 3 seconds
      setTimeout(() => {
        setShowPointsEarned(false);
      }, 3000);
    }
  };

  const handleVideoTimeUpdate = () => {
    if (videoRef.current) {
      const current = videoRef.current.currentTime;
      const total = videoRef.current.duration;
      
      setCurrentTime(current);
      setDuration(total);

      // Award points when video is watched to the end (95% completion)
      if (current / total >= 0.95 && !hasEarnedPoints) {
        awardPoints();
      }
    }
  };

  const handleVideoPlay = () => {
    setIsPlaying(true);
  };

  const handleVideoPause = () => {
    setIsPlaying(false);
  };

  const handlePlayPause = () => {
    if (videoRef.current) {
      if (isPlaying) {
        videoRef.current.pause();
      } else {
        videoRef.current.play();
      }
    }
  };

  const handleMuteToggle = () => {
    if (videoRef.current) {
      videoRef.current.muted = !isMuted;
      setIsMuted(!isMuted);
    }
  };

  const handleLearnMore = () => {
    trackClick();
    // In real app, would open ad link
    toast.info("Opening advertiser website...");
  };

  const formatTime = (time: number) => {
    const minutes = Math.floor(time / 60);
    const seconds = Math.floor(time % 60);
    return `${minutes}:${seconds.toString().padStart(2, '0')}`;
  };

  if (!ad) return null;

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto p-0">
        {/* Header with close button and points */}
        <div className="flex items-center justify-between p-4 border-b">
          <div className="flex items-center gap-3">
            <Badge className="bg-orange-600 text-white">
              {ad.type.charAt(0).toUpperCase() + ad.type.slice(1)} Ad
            </Badge>
            <div className="flex items-center gap-1 text-orange-600">
              <Coins className="w-4 h-4" />
              <span className="font-semibold">+{ad.pointsReward} points</span>
            </div>
          </div>
          <Button variant="ghost" size="sm" onClick={onClose}>
            <X className="w-4 h-4" />
          </Button>
        </div>

        {/* Ad Content */}
        <div className="p-6 space-y-6">
          {/* Video Ad */}
          {ad.type === "video" && (
            <div className="relative">
              <video
                ref={videoRef}
                className="w-full h-64 md:h-96 bg-black rounded-lg"
                onTimeUpdate={handleVideoTimeUpdate}
                onPlay={handleVideoPlay}
                onPause={handleVideoPause}
                onLoadedMetadata={() => {
                  if (videoRef.current) {
                    setDuration(videoRef.current.duration);
                  }
                }}
              >
                <source src={ad.videoUrl || "https://example.com/demo-video.mp4"} type="video/mp4" />
                Your browser does not support the video tag.
              </video>

              {/* Video Controls */}
              <div className="absolute bottom-4 left-4 right-4">
                <div className="bg-black/70 rounded-lg p-3 space-y-2">
                  <Progress value={(currentTime / duration) * 100} className="h-1" />
                  <div className="flex items-center justify-between text-white">
                    <div className="flex items-center gap-2">
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={handlePlayPause}
                        className="text-white hover:bg-white/20"
                      >
                        {isPlaying ? <Pause className="w-4 h-4" /> : <Play className="w-4 h-4" />}
                      </Button>
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={handleMuteToggle}
                        className="text-white hover:bg-white/20"
                      >
                        {isMuted ? <VolumeX className="w-4 h-4" /> : <Volume2 className="w-4 h-4" />}
                      </Button>
                      <span className="text-sm">
                        {formatTime(currentTime)} / {formatTime(duration)}
                      </span>
                    </div>
                    <div className="flex items-center gap-1 text-orange-400">
                      <Timer className="w-4 h-4" />
                      <span className="text-sm">Watch to end for points</span>
                    </div>
                  </div>
                </div>
              </div>

              {/* Points Earned Animation */}
              <AnimatePresence>
                {showPointsEarned && (
                  <motion.div
                    initial={{ opacity: 0, scale: 0.5, y: 20 }}
                    animate={{ opacity: 1, scale: 1, y: 0 }}
                    exit={{ opacity: 0, scale: 0.5, y: -20 }}
                    className="absolute top-4 right-4"
                  >
                    <Card className="bg-green-600 text-white border-green-500">
                      <CardContent className="p-3 flex items-center gap-2">
                        <CheckCircle className="w-5 h-5" />
                        <span className="font-semibold">+{ad.pointsReward} Points!</span>
                      </CardContent>
                    </Card>
                  </motion.div>
                )}
              </AnimatePresence>
            </div>
          )}

          {/* Photo Ad */}
          {ad.type === "photo" && ad.imageUrl && (
            <div className="relative">
              <ImageWithFallback
                src={ad.imageUrl}
                alt={ad.title}
                className="w-full h-64 md:h-96 object-cover rounded-lg"
              />
              
              {/* Points Earned Animation */}
              <AnimatePresence>
                {showPointsEarned && (
                  <motion.div
                    initial={{ opacity: 0, scale: 0.5, y: 20 }}
                    animate={{ opacity: 1, scale: 1, y: 0 }}
                    exit={{ opacity: 0, scale: 0.5, y: -20 }}
                    className="absolute top-4 right-4"
                  >
                    <Card className="bg-green-600 text-white border-green-500">
                      <CardContent className="p-3 flex items-center gap-2">
                        <Gift className="w-5 h-5" />
                        <span className="font-semibold">+{ad.pointsReward} Points!</span>
                      </CardContent>
                    </Card>
                  </motion.div>
                )}
              </AnimatePresence>
            </div>
          )}

          {/* Text Ad */}
          {ad.type === "text" && (
            <div className="relative bg-gradient-to-r from-orange-50 to-orange-100 dark:from-orange-900/20 dark:to-orange-800/20 rounded-lg p-8 text-center">
              <div className="space-y-4">
                <h3 className="text-2xl font-bold">{ad.title}</h3>
                <p className="text-lg text-muted-foreground max-w-2xl mx-auto">{ad.description}</p>
              </div>

              {/* Points Earned Animation */}
              <AnimatePresence>
                {showPointsEarned && (
                  <motion.div
                    initial={{ opacity: 0, scale: 0.5, y: 20 }}
                    animate={{ opacity: 1, scale: 1, y: 0 }}
                    exit={{ opacity: 0, scale: 0.5, y: -20 }}
                    className="absolute top-4 right-4"
                  >
                    <Card className="bg-green-600 text-white border-green-500">
                      <CardContent className="p-3 flex items-center gap-2">
                        <Coins className="w-5 h-5" />
                        <span className="font-semibold">+{ad.pointsReward} Points!</span>
                      </CardContent>
                    </Card>
                  </motion.div>
                )}
              </AnimatePresence>
            </div>
          )}

          {/* Ad Details */}
          <div className="space-y-4">
            <div>
              <h3 className="font-semibold text-lg">{ad.title}</h3>
              <p className="text-muted-foreground">{ad.description}</p>
            </div>

            {ad.location && (
              <div className="text-sm text-muted-foreground">
                📍 {ad.location}
              </div>
            )}

            {/* Ad Stats */}
            <div className="flex items-center gap-6 text-sm text-muted-foreground">
              <div className="flex items-center gap-1">
                <Eye className="w-4 h-4" />
                <span>{ad.impressions.toLocaleString()} views</span>
              </div>
              <div className="flex items-center gap-1">
                <MousePointer className="w-4 h-4" />
                <span>{ad.clicks.toLocaleString()} clicks</span>
              </div>
              <Badge variant="secondary" className="text-xs">
                {ad.category.replace("-", " ").replace(/\b\w/g, l => l.toUpperCase())}
              </Badge>
            </div>

            {/* Action Buttons */}
            <div className="flex gap-3 pt-4">
              <Button 
                onClick={handleLearnMore}
                className="bg-orange-600 hover:bg-orange-700 flex-1"
              >
                Learn More
              </Button>
              <Button variant="outline" onClick={onClose}>
                Close
              </Button>
            </div>

            {/* Points Status */}
            <div className="bg-muted/50 rounded-lg p-3">
              <div className="flex items-center justify-between text-sm">
                <span>Points Status:</span>
                <span className={hasEarnedPoints ? "text-green-600 font-semibold" : "text-muted-foreground"}>
                  {hasEarnedPoints ? (
                    <span className="flex items-center gap-1">
                      <CheckCircle className="w-4 h-4" />
                      Points Earned!
                    </span>
                  ) : ad.type === "video" ? (
                    "Watch to end to earn points"
                  ) : (
                    "Viewing ad..."
                  )}
                </span>
              </div>
            </div>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}