import { useState } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "./ui/card";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Input } from "./ui/input";
import { Label } from "./ui/label";
import { Textarea } from "./ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "./ui/select";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "./ui/dialog";
import { Alert, AlertDescription } from "./ui/alert";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "./ui/tabs";
import { 
  Search, 
  Filter, 
  CheckCircle, 
  X, 
  Eye, 
  Settings, 
  DollarSign,
  Calendar,
  Clock,
  User,
  Video,
  Image as ImageIcon,
  FileText,
  MoreVertical,
  Trash2,
  Ban,
  Play
} from "lucide-react";
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger } from "./ui/dropdown-menu";
import { ImageWithFallback } from "./figma/ImageWithFallback";

interface PendingAd {
  id: number;
  title: string;
  description: string;
  type: "video" | "photo" | "text";
  category: "services" | "products" | "dating-tips";
  location?: string;
  contactInfo: string;
  duration: string;
  price: number;
  submittedBy: string;
  submittedAt: string;
  status: "pending" | "approved" | "rejected";
  content?: string;
  imageUrl?: string;
  videoUrl?: string;
}

interface AdPricing {
  video: { [key: string]: number };
  photo: { [key: string]: number };
  text: { [key: string]: number };
}

export function AdminAdManagement() {
  const [activeTab, setActiveTab] = useState("pending");
  const [searchQuery, setSearchQuery] = useState("");
  const [selectedAd, setSelectedAd] = useState<PendingAd | null>(null);
  const [isPreviewOpen, setIsPreviewOpen] = useState(false);
  const [isPricingOpen, setIsPricingOpen] = useState(false);

  // Mock pricing data
  const [adPricing, setAdPricing] = useState<AdPricing>({
    video: { "7": 49.99, "14": 89.99, "30": 149.99, "60": 249.99 },
    photo: { "7": 29.99, "14": 54.99, "30": 89.99, "60": 149.99 },
    text: { "7": 19.99, "14": 34.99, "30": 59.99, "60": 99.99 }
  });

  // Mock data for pending ads
  const [pendingAds, setPendingAds] = useState<PendingAd[]>([
    {
      id: 1,
      title: "Professional Dating Photography - Get Results!",
      description: "Transform your dating profile with stunning professional photos that get 3x more matches! Award-winning photographer with 500+ success stories.",
      type: "photo",
      category: "services",
      location: "Los Angeles, CA",
      contactInfo: "contact@photopro.com",
      duration: "30",
      price: 89.99,
      submittedBy: "sarah_photographer",
      submittedAt: "2024-12-25T10:30:00Z",
      status: "pending",
      imageUrl: "https://images.unsplash.com/photo-1554048612-b6a482b224b8?w=400&h=300&fit=crop"
    },
    {
      id: 2,
      title: "Master the Art of Dating - Expert Coaching",
      description: "Unlock your dating potential with proven strategies from certified relationship experts. 1-on-1 coaching that transforms your approach to dating.",
      type: "video",
      category: "dating-tips",
      location: "Online",
      contactInfo: "info@datingmaster.com",
      duration: "14",
      price: 89.99,
      submittedBy: "dating_coach_mike",
      submittedAt: "2024-12-24T14:20:00Z",
      status: "pending",
      videoUrl: "https://example.com/video.mp4"
    },
    {
      id: 3,
      title: "Premium Cologne Collection - Irresistible Scent",
      description: "Make an unforgettable first impression with our signature cologne blend. Scientifically formulated to boost confidence and attraction.",
      type: "text",
      category: "products",
      contactInfo: "orders@luxscent.com",
      duration: "7",
      price: 19.99,
      submittedBy: "luxury_scents",
      submittedAt: "2024-12-23T16:45:00Z",
      status: "pending"
    }
  ]);

  const handleApproveAd = (adId: number) => {
    setPendingAds(pendingAds.map(ad => 
      ad.id === adId ? { ...ad, status: "approved" } : ad
    ));
  };

  const handleRejectAd = (adId: number, reason?: string) => {
    setPendingAds(pendingAds.map(ad => 
      ad.id === adId ? { ...ad, status: "rejected" } : ad
    ));
    // In real app, would also trigger refund and send notification to user
  };

  const handleDeleteAd = (adId: number) => {
    setPendingAds(pendingAds.filter(ad => ad.id !== adId));
  };

  const updatePricing = (type: keyof AdPricing, duration: string, newPrice: number) => {
    setAdPricing({
      ...adPricing,
      [type]: {
        ...adPricing[type],
        [duration]: newPrice
      }
    });
  };

  const getStatusBadge = (status: string) => {
    switch (status) {
      case "pending":
        return <Badge className="bg-yellow-100 text-yellow-800">Pending Review</Badge>;
      case "approved":
        return <Badge className="bg-green-100 text-green-800">Approved</Badge>;
      case "rejected":
        return <Badge className="bg-red-100 text-red-800">Rejected</Badge>;
      default:
        return <Badge>{status}</Badge>;
    }
  };

  const getTypeIcon = (type: string) => {
    switch (type) {
      case "video":
        return <Video className="w-4 h-4" />;
      case "photo":
        return <ImageIcon className="w-4 h-4" />;
      case "text":
        return <FileText className="w-4 h-4" />;
      default:
        return <FileText className="w-4 h-4" />;
    }
  };

  const filteredAds = pendingAds.filter(ad => {
    const matchesSearch = ad.title.toLowerCase().includes(searchQuery.toLowerCase()) ||
                         ad.description.toLowerCase().includes(searchQuery.toLowerCase()) ||
                         ad.submittedBy.toLowerCase().includes(searchQuery.toLowerCase());
    const matchesTab = activeTab === "all" || ad.status === activeTab;
    return matchesSearch && matchesTab;
  });

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-3xl font-bold">Ad Management</h2>
          <p className="text-muted-foreground">Review and manage member-submitted ads</p>
        </div>
        <Dialog open={isPricingOpen} onOpenChange={setIsPricingOpen}>
          <DialogTrigger asChild>
            <Button className="bg-orange-600 hover:bg-orange-700">
              <Settings className="w-4 h-4 mr-2" />
              Manage Pricing
            </Button>
          </DialogTrigger>
          <DialogContent className="max-w-2xl">
            <DialogHeader>
              <DialogTitle>Ad Pricing Management</DialogTitle>
              <DialogDescription>
                Set pricing for different ad types and durations
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-6">
              {Object.entries(adPricing).map(([type, prices]) => (
                <div key={type}>
                  <h4 className="font-semibold mb-3 flex items-center gap-2">
                    {getTypeIcon(type)}
                    {type.charAt(0).toUpperCase() + type.slice(1)} Ads
                  </h4>
                  <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                    {Object.entries(prices).map(([duration, price]) => (
                      <div key={duration}>
                        <Label htmlFor={`${type}-${duration}`}>
                          {duration} Days
                        </Label>
                        <div className="flex items-center gap-1">
                          <span className="text-sm">$</span>
                          <Input
                            id={`${type}-${duration}`}
                            type="number"
                            value={price}
                            onChange={(e) => updatePricing(type as keyof AdPricing, duration, parseFloat(e.target.value))}
                            step="0.01"
                            min="0"
                          />
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              ))}
            </div>
            <div className="flex justify-end">
              <Button onClick={() => setIsPricingOpen(false)} className="bg-orange-600 hover:bg-orange-700">
                Save Pricing
              </Button>
            </div>
          </DialogContent>
        </Dialog>
      </div>

      <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-6">
        <div className="flex items-center justify-between">
          <TabsList className="grid w-full max-w-md grid-cols-3">
            <TabsTrigger value="pending">Pending ({pendingAds.filter(ad => ad.status === "pending").length})</TabsTrigger>
            <TabsTrigger value="approved">Approved ({pendingAds.filter(ad => ad.status === "approved").length})</TabsTrigger>
            <TabsTrigger value="rejected">Rejected ({pendingAds.filter(ad => ad.status === "rejected").length})</TabsTrigger>
          </TabsList>

          <div className="flex gap-4">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
              <Input
                placeholder="Search ads..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="pl-10 w-64"
              />
            </div>
          </div>
        </div>

        <TabsContent value={activeTab} className="space-y-4">
          {filteredAds.length === 0 ? (
            <div className="text-center py-12">
              <div className="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
                <Search className="w-8 h-8 text-gray-400" />
              </div>
              <h3 className="text-lg font-semibold mb-2">No ads found</h3>
              <p className="text-muted-foreground">
                {searchQuery ? "Try adjusting your search criteria" : "No ads to review at this time"}
              </p>
            </div>
          ) : (
            <div className="grid gap-4">
              {filteredAds.map((ad) => (
                <Card key={ad.id}>
                  <CardContent className="p-6">
                    <div className="flex items-start gap-4">
                      {/* Ad Preview */}
                      <div className="w-24 h-24 bg-gray-100 rounded-lg flex items-center justify-center flex-shrink-0">
                        {ad.type === "photo" && ad.imageUrl ? (
                          <ImageWithFallback
                            src={ad.imageUrl}
                            alt={ad.title}
                            className="w-full h-full object-cover rounded-lg"
                          />
                        ) : ad.type === "video" ? (
                          <div className="relative">
                            <Video className="w-8 h-8 text-gray-400" />
                            <Play className="w-4 h-4 absolute -bottom-1 -right-1 text-orange-600" />
                          </div>
                        ) : (
                          <FileText className="w-8 h-8 text-gray-400" />
                        )}
                      </div>

                      {/* Ad Details */}
                      <div className="flex-1">
                        <div className="flex items-start justify-between mb-2">
                          <div>
                            <div className="flex items-center gap-2 mb-1">
                              <h3 className="font-semibold">{ad.title}</h3>
                              {getStatusBadge(ad.status)}
                              <Badge variant="outline" className="flex items-center gap-1">
                                {getTypeIcon(ad.type)}
                                {ad.type}
                              </Badge>
                            </div>
                            <p className="text-muted-foreground text-sm mb-2">{ad.description}</p>
                          </div>
                          <DropdownMenu>
                            <DropdownMenuTrigger asChild>
                              <Button variant="ghost" size="sm">
                                <MoreVertical className="w-4 h-4" />
                              </Button>
                            </DropdownMenuTrigger>
                            <DropdownMenuContent align="end">
                              <DropdownMenuItem onClick={() => {
                                setSelectedAd(ad);
                                setIsPreviewOpen(true);
                              }}>
                                <Eye className="w-4 h-4 mr-2" />
                                Preview Ad
                              </DropdownMenuItem>
                              {ad.status === "pending" && (
                                <>
                                  <DropdownMenuItem onClick={() => handleApproveAd(ad.id)}>
                                    <CheckCircle className="w-4 h-4 mr-2" />
                                    Approve
                                  </DropdownMenuItem>
                                  <DropdownMenuItem onClick={() => handleRejectAd(ad.id)}>
                                    <X className="w-4 h-4 mr-2" />
                                    Reject
                                  </DropdownMenuItem>
                                </>
                              )}
                              <DropdownMenuItem 
                                onClick={() => handleDeleteAd(ad.id)}
                                className="text-red-600"
                              >
                                <Trash2 className="w-4 h-4 mr-2" />
                                Delete
                              </DropdownMenuItem>
                            </DropdownMenuContent>
                          </DropdownMenu>
                        </div>

                        <div className="flex items-center gap-6 text-sm text-muted-foreground">
                          <span className="flex items-center gap-1">
                            <User className="w-4 h-4" />
                            {ad.submittedBy}
                          </span>
                          <span className="flex items-center gap-1">
                            <Calendar className="w-4 h-4" />
                            {ad.duration} days
                          </span>
                          <span className="flex items-center gap-1">
                            <DollarSign className="w-4 h-4" />
                            ${ad.price}
                          </span>
                          <span className="flex items-center gap-1">
                            <Clock className="w-4 h-4" />
                            {new Date(ad.submittedAt).toLocaleDateString()}
                          </span>
                          {ad.location && (
                            <span>📍 {ad.location}</span>
                          )}
                        </div>

                        {/* Quick Actions for Pending Ads */}
                        {ad.status === "pending" && (
                          <div className="flex gap-2 mt-4">
                            <Button
                              onClick={() => handleApproveAd(ad.id)}
                              size="sm"
                              className="bg-green-600 hover:bg-green-700"
                            >
                              <CheckCircle className="w-4 h-4 mr-1" />
                              Approve
                            </Button>
                            <Button
                              onClick={() => handleRejectAd(ad.id)}
                              variant="outline"
                              size="sm"
                              className="text-red-600 border-red-200 hover:bg-red-50"
                            >
                              <X className="w-4 h-4 mr-1" />
                              Reject
                            </Button>
                            <Button
                              onClick={() => {
                                setSelectedAd(ad);
                                setIsPreviewOpen(true);
                              }}
                              variant="outline"
                              size="sm"
                            >
                              <Eye className="w-4 h-4 mr-1" />
                              Preview
                            </Button>
                          </div>
                        )}
                      </div>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          )}
        </TabsContent>
      </Tabs>

      {/* Ad Preview Modal */}
      <Dialog open={isPreviewOpen} onOpenChange={setIsPreviewOpen}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>Ad Preview</DialogTitle>
            <DialogDescription>
              Preview how this ad will appear to users
            </DialogDescription>
          </DialogHeader>
          {selectedAd && (
            <div className="space-y-4">
              {/* Mock ad display */}
              <Card className="p-0 hover:shadow-lg transition-shadow">
                <div className="relative">
                  {selectedAd.imageUrl && (
                    <ImageWithFallback
                      src={selectedAd.imageUrl}
                      alt={selectedAd.title}
                      className="w-full h-48 object-cover rounded-t-lg"
                    />
                  )}
                  {selectedAd.type === "video" && (
                    <div className="w-full h-48 bg-gray-100 rounded-t-lg flex items-center justify-center">
                      <div className="text-center">
                        <Video className="w-12 h-12 text-gray-400 mx-auto mb-2" />
                        <p className="text-sm text-gray-600">Video Ad Preview</p>
                      </div>
                    </div>
                  )}
                  <Badge className="absolute top-2 left-2 bg-orange-600 text-white">
                    Sponsored
                  </Badge>
                </div>
                
                <div className="p-4 space-y-3">
                  <div>
                    <h3 className="font-semibold">{selectedAd.title}</h3>
                    <p className="text-muted-foreground text-sm mt-1">{selectedAd.description}</p>
                  </div>
                  
                  <div className="flex items-center justify-between text-sm">
                    {selectedAd.location && (
                      <div className="flex items-center gap-1 text-muted-foreground">
                        <span>📍 {selectedAd.location}</span>
                      </div>
                    )}
                    <div className="font-semibold text-orange-600">
                      Contact: {selectedAd.contactInfo}
                    </div>
                  </div>
                  
                  <div className="flex items-center justify-between pt-2">
                    <Badge variant="secondary" className="text-xs capitalize">
                      {selectedAd.category.replace("-", " ")}
                    </Badge>
                    <Button size="sm" variant="outline">
                      Learn More
                    </Button>
                  </div>
                </div>
              </Card>

              {/* Admin actions */}
              {selectedAd.status === "pending" && (
                <div className="flex gap-2">
                  <Button
                    onClick={() => {
                      handleApproveAd(selectedAd.id);
                      setIsPreviewOpen(false);
                    }}
                    className="bg-green-600 hover:bg-green-700 flex-1"
                  >
                    <CheckCircle className="w-4 h-4 mr-2" />
                    Approve Ad
                  </Button>
                  <Button
                    onClick={() => {
                      handleRejectAd(selectedAd.id);
                      setIsPreviewOpen(false);
                    }}
                    variant="outline"
                    className="text-red-600 border-red-200 hover:bg-red-50 flex-1"
                  >
                    <X className="w-4 h-4 mr-2" />
                    Reject Ad
                  </Button>
                </div>
              )}
            </div>
          )}
        </DialogContent>
      </Dialog>
    </div>
  );
}