import React, { useState } from 'react';
import {
  Users,
  CheckCircle,
  XCircle,
  Eye,
  EyeOff,
  Trash2,
  Calendar,
  Video,
  Mic,
  DollarSign,
  AlertTriangle,
  Search,
  Filter,
  Star,
  Award,
  Clock,
  MapPin
} from 'lucide-react';
import { Button } from './ui/button';
import { Card } from './ui/card';
import { Badge } from './ui/badge';
import { Input } from './ui/input';
import { Tabs, TabsContent, TabsList, TabsTrigger } from './ui/tabs';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription, DialogFooter } from './ui/dialog';
import { Avatar, AvatarImage, AvatarFallback } from './ui/avatar';
import { Textarea } from './ui/textarea';
import { Label } from './ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from './ui/select';
import { toast } from 'sonner@2.0.3';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from './ui/table';

interface CoachApplication {
  id: string;
  fullName: string;
  email: string;
  phone: string;
  location: string;
  country: string;
  yearsExperience: number;
  specialties: string[];
  bio: string;
  certifications: string;
  why: string;
  appliedDate: string;
  status: 'pending' | 'approved' | 'rejected';
}

interface Coach {
  id: string;
  name: string;
  email: string;
  photo: string;
  location: string;
  country: string;
  specialties: string[];
  rating: number;
  totalSessions: number;
  reviewCount: number;
  yearsExperience: number;
  status: 'active' | 'suspended';
  joinedDate: string;
  totalEarnings: number;
}

interface CoachSession {
  id: string;
  coachId: string;
  coachName: string;
  title: string;
  sessionType: 'video' | 'audio';
  date: string;
  time: string;
  duration: number;
  maxParticipants: number;
  currentParticipants: number;
  price: number;
  status: 'scheduled' | 'completed' | 'cancelled';
  category: string;
}

// Mock data
const mockApplications: CoachApplication[] = [
  {
    id: 'app1',
    fullName: 'Jessica Williams',
    email: 'jessica.w@email.com',
    phone: '+1-555-0123',
    location: 'Austin',
    country: 'USA',
    yearsExperience: 7,
    specialties: ['Confidence Building', 'First Dates', 'Communication'],
    bio: 'Licensed therapist specializing in relationship coaching with a focus on building authentic connections and overcoming dating anxiety.',
    certifications: 'MA in Clinical Psychology, Certified Relationship Coach (CRC), Life Coach Certificate',
    why: 'I\'m passionate about helping singles navigate the complexities of modern dating and build the confidence they need to find meaningful relationships.',
    appliedDate: '2025-10-18',
    status: 'pending'
  },
  {
    id: 'app2',
    fullName: 'Michael Torres',
    email: 'michael.t@email.com',
    phone: '+44-7700-900123',
    location: 'Manchester',
    country: 'UK',
    yearsExperience: 5,
    specialties: ['Profile Optimization', 'Online Dating', 'First Messages'],
    bio: 'Former dating app product manager with extensive experience in profile optimization and online dating strategy.',
    certifications: 'Digital Marketing Certificate, Dating Coach Certification from IDC',
    why: 'Having worked in the online dating industry for years, I understand what makes profiles stand out and want to share this knowledge.',
    appliedDate: '2025-10-17',
    status: 'pending'
  }
];

const mockCoaches: Coach[] = [
  {
    id: 'c1',
    name: 'Dr. Emily Roberts',
    email: 'emily.roberts@email.com',
    photo: 'https://images.unsplash.com/photo-1494790108377-be9c29b29330?w=400',
    location: 'New York',
    country: 'USA',
    specialties: ['Confidence Building', 'First Dates', 'Communication Skills'],
    rating: 4.8,
    totalSessions: 234,
    reviewCount: 156,
    yearsExperience: 10,
    status: 'active',
    joinedDate: '2024-03-15',
    totalEarnings: 12450
  },
  {
    id: 'c2',
    name: 'Marcus Johnson',
    email: 'marcus.j@email.com',
    photo: 'https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?w=400',
    location: 'London',
    country: 'UK',
    specialties: ['Profile Tips', 'Online Dating', 'First Messages'],
    rating: 4.9,
    totalSessions: 312,
    reviewCount: 203,
    yearsExperience: 8,
    status: 'active',
    joinedDate: '2024-01-20',
    totalEarnings: 18900
  }
];

const mockSessions: CoachSession[] = [
  {
    id: 's1',
    coachId: 'c1',
    coachName: 'Dr. Emily Roberts',
    title: 'First Date Confidence Masterclass',
    sessionType: 'video',
    date: '2025-10-25',
    time: '18:00',
    duration: 60,
    maxParticipants: 20,
    currentParticipants: 12,
    price: 25,
    status: 'scheduled',
    category: 'Confidence Building'
  },
  {
    id: 's2',
    coachId: 'c2',
    coachName: 'Marcus Johnson',
    title: 'Online Dating Profile Optimization',
    sessionType: 'video',
    date: '2025-10-23',
    time: '19:30',
    duration: 90,
    maxParticipants: 15,
    currentParticipants: 8,
    price: 0,
    status: 'scheduled',
    category: 'Profile Tips'
  }
];

export default function AdminCoachManagement() {
  const [activeTab, setActiveTab] = useState('applications');
  const [searchQuery, setSearchQuery] = useState('');
  const [selectedApplication, setSelectedApplication] = useState<CoachApplication | null>(null);
  const [selectedCoach, setSelectedCoach] = useState<Coach | null>(null);
  const [selectedSession, setSelectedSession] = useState<CoachSession | null>(null);
  const [showApplicationModal, setShowApplicationModal] = useState(false);
  const [showCoachModal, setShowCoachModal] = useState(false);
  const [showSessionModal, setShowSessionModal] = useState(false);
  const [showRejectModal, setShowRejectModal] = useState(false);
  const [showSuspendModal, setShowSuspendModal] = useState(false);
  const [rejectReason, setRejectReason] = useState('');
  const [suspendReason, setSuspendReason] = useState('');
  const [applications, setApplications] = useState(mockApplications);
  const [coaches, setCoaches] = useState(mockCoaches);
  const [sessions, setSessions] = useState(mockSessions);

  const handleApproveApplication = (appId: string) => {
    setApplications(prev => prev.map(app => 
      app.id === appId ? { ...app, status: 'approved' as const } : app
    ));
    toast.success('Application approved! Coach account created.');
    setShowApplicationModal(false);
    setSelectedApplication(null);
  };

  const handleRejectApplication = () => {
    if (selectedApplication && rejectReason.trim()) {
      setApplications(prev => prev.map(app => 
        app.id === selectedApplication.id ? { ...app, status: 'rejected' as const } : app
      ));
      toast.success('Application rejected. Notification sent to applicant.');
      setShowRejectModal(false);
      setShowApplicationModal(false);
      setSelectedApplication(null);
      setRejectReason('');
    }
  };

  const handleSuspendCoach = () => {
    if (selectedCoach && suspendReason.trim()) {
      setCoaches(prev => prev.map(coach => 
        coach.id === selectedCoach.id ? { ...coach, status: 'suspended' as const } : coach
      ));
      toast.success('Coach suspended. All their sessions have been cancelled.');
      setShowSuspendModal(false);
      setShowCoachModal(false);
      setSelectedCoach(null);
      setSuspendReason('');
    }
  };

  const handleReactivateCoach = (coachId: string) => {
    setCoaches(prev => prev.map(coach => 
      coach.id === coachId ? { ...coach, status: 'active' as const } : coach
    ));
    toast.success('Coach reactivated successfully.');
  };

  const handleDeleteCoach = (coachId: string) => {
    if (confirm('Are you sure you want to permanently delete this coach? This action cannot be undone.')) {
      setCoaches(prev => prev.filter(coach => coach.id !== coachId));
      toast.success('Coach deleted permanently.');
      setShowCoachModal(false);
      setSelectedCoach(null);
    }
  };

  const handleCancelSession = (sessionId: string) => {
    if (confirm('Cancel this session? All registered members will be refunded and notified.')) {
      setSessions(prev => prev.map(session => 
        session.id === sessionId ? { ...session, status: 'cancelled' as const } : session
      ));
      toast.success('Session cancelled. Members notified and refunded.');
      setShowSessionModal(false);
      setSelectedSession(null);
    }
  };

  const handleMarkFree = (sessionId: string) => {
    setSessions(prev => prev.map(session => 
      session.id === sessionId ? { ...session, price: 0 } : session
    ));
    toast.success('Session marked as free. Registered members will be refunded.');
  };

  const filteredApplications = applications.filter(app =>
    app.fullName.toLowerCase().includes(searchQuery.toLowerCase()) ||
    app.email.toLowerCase().includes(searchQuery.toLowerCase()) ||
    app.location.toLowerCase().includes(searchQuery.toLowerCase())
  );

  const filteredCoaches = coaches.filter(coach =>
    coach.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
    coach.email.toLowerCase().includes(searchQuery.toLowerCase()) ||
    coach.location.toLowerCase().includes(searchQuery.toLowerCase())
  );

  const filteredSessions = sessions.filter(session =>
    session.title.toLowerCase().includes(searchQuery.toLowerCase()) ||
    session.coachName.toLowerCase().includes(searchQuery.toLowerCase()) ||
    session.category.toLowerCase().includes(searchQuery.toLowerCase())
  );

  const pendingCount = applications.filter(a => a.status === 'pending').length;
  const activeCoachesCount = coaches.filter(c => c.status === 'active').length;
  const upcomingSessionsCount = sessions.filter(s => s.status === 'scheduled').length;

  return (
    <div className="space-y-6">
      {/* Header */}
      <div>
        <h2 className="text-2xl font-semibold text-gray-900 dark:text-white mb-2">Dating Coach Management</h2>
        <p className="text-gray-600 dark:text-gray-400">
          Manage coach applications, active coaches, and coaching sessions
        </p>
      </div>

      {/* Stats */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card className="p-4">
          <div className="flex items-center gap-3">
            <div className="w-10 h-10 rounded-full bg-orange-100 dark:bg-orange-900/30 flex items-center justify-center">
              <AlertTriangle className="w-5 h-5 text-orange-600 dark:text-orange-400" />
            </div>
            <div>
              <div className="text-2xl font-semibold">{pendingCount}</div>
              <div className="text-sm text-gray-600 dark:text-gray-400">Pending Applications</div>
            </div>
          </div>
        </Card>
        <Card className="p-4">
          <div className="flex items-center gap-3">
            <div className="w-10 h-10 rounded-full bg-green-100 dark:bg-green-900/30 flex items-center justify-center">
              <Users className="w-5 h-5 text-green-600 dark:text-green-400" />
            </div>
            <div>
              <div className="text-2xl font-semibold">{activeCoachesCount}</div>
              <div className="text-sm text-gray-600 dark:text-gray-400">Active Coaches</div>
            </div>
          </div>
        </Card>
        <Card className="p-4">
          <div className="flex items-center gap-3">
            <div className="w-10 h-10 rounded-full bg-blue-100 dark:bg-blue-900/30 flex items-center justify-center">
              <Calendar className="w-5 h-5 text-blue-600 dark:text-blue-400" />
            </div>
            <div>
              <div className="text-2xl font-semibold">{upcomingSessionsCount}</div>
              <div className="text-sm text-gray-600 dark:text-gray-400">Upcoming Sessions</div>
            </div>
          </div>
        </Card>
        <Card className="p-4">
          <div className="flex items-center gap-3">
            <div className="w-10 h-10 rounded-full bg-purple-100 dark:bg-purple-900/30 flex items-center justify-center">
              <DollarSign className="w-5 h-5 text-purple-600 dark:text-purple-400" />
            </div>
            <div>
              <div className="text-2xl font-semibold">$31.3k</div>
              <div className="text-sm text-gray-600 dark:text-gray-400">Total Earnings</div>
            </div>
          </div>
        </Card>
      </div>

      {/* Search */}
      <div className="relative">
        <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
        <Input
          placeholder="Search applications, coaches, or sessions..."
          value={searchQuery}
          onChange={(e) => setSearchQuery(e.target.value)}
          className="pl-10"
        />
      </div>

      {/* Tabs */}
      <Tabs value={activeTab} onValueChange={setActiveTab}>
        <TabsList className="grid w-full max-w-2xl grid-cols-3">
          <TabsTrigger value="applications">
            Applications {pendingCount > 0 && <Badge variant="destructive" className="ml-2">{pendingCount}</Badge>}
          </TabsTrigger>
          <TabsTrigger value="coaches">Active Coaches</TabsTrigger>
          <TabsTrigger value="sessions">All Sessions</TabsTrigger>
        </TabsList>

        {/* Applications Tab */}
        <TabsContent value="applications" className="space-y-4">
          {filteredApplications.length === 0 ? (
            <Card className="p-12 text-center">
              <Users className="w-16 h-16 mx-auto mb-4 text-gray-400" />
              <h3 className="text-lg font-medium mb-2">No applications found</h3>
              <p className="text-gray-600 dark:text-gray-400">
                {searchQuery ? 'Try adjusting your search' : 'New applications will appear here'}
              </p>
            </Card>
          ) : (
            <div className="space-y-3">
              {filteredApplications.map(app => (
                <Card key={app.id} className="p-4">
                  <div className="flex items-start justify-between">
                    <div className="flex-1">
                      <div className="flex items-center gap-3 mb-2">
                        <h3 className="font-medium">{app.fullName}</h3>
                        <Badge variant={
                          app.status === 'pending' ? 'default' :
                          app.status === 'approved' ? 'default' : 'destructive'
                        } className={
                          app.status === 'pending' ? 'bg-orange-500' :
                          app.status === 'approved' ? 'bg-green-500' : ''
                        }>
                          {app.status.charAt(0).toUpperCase() + app.status.slice(1)}
                        </Badge>
                      </div>
                      <div className="grid grid-cols-2 md:grid-cols-4 gap-2 text-sm text-gray-600 dark:text-gray-400 mb-3">
                        <div>{app.email}</div>
                        <div className="flex items-center gap-1">
                          <MapPin className="w-4 h-4" />
                          {app.location}, {app.country}
                        </div>
                        <div>{app.yearsExperience} years experience</div>
                        <div>Applied: {new Date(app.appliedDate).toLocaleDateString()}</div>
                      </div>
                      <div className="flex flex-wrap gap-2">
                        {app.specialties.map(specialty => (
                          <Badge key={specialty} variant="secondary">{specialty}</Badge>
                        ))}
                      </div>
                    </div>
                    <Button
                      onClick={() => {
                        setSelectedApplication(app);
                        setShowApplicationModal(true);
                      }}
                      variant="outline"
                      size="sm"
                    >
                      Review
                    </Button>
                  </div>
                </Card>
              ))}
            </div>
          )}
        </TabsContent>

        {/* Coaches Tab */}
        <TabsContent value="coaches" className="space-y-4">
          {filteredCoaches.length === 0 ? (
            <Card className="p-12 text-center">
              <Users className="w-16 h-16 mx-auto mb-4 text-gray-400" />
              <h3 className="text-lg font-medium mb-2">No coaches found</h3>
              <p className="text-gray-600 dark:text-gray-400">
                {searchQuery ? 'Try adjusting your search' : 'Approved coaches will appear here'}
              </p>
            </Card>
          ) : (
            <div className="space-y-3">
              {filteredCoaches.map(coach => (
                <Card key={coach.id} className="p-4">
                  <div className="flex items-start gap-4">
                    <Avatar className="w-16 h-16 ring-2 ring-pink-200 dark:ring-pink-800">
                      <AvatarImage src={coach.photo} alt={coach.name} />
                      <AvatarFallback>{coach.name[0]}</AvatarFallback>
                    </Avatar>
                    <div className="flex-1">
                      <div className="flex items-center gap-3 mb-2">
                        <h3 className="font-medium">{coach.name}</h3>
                        <Badge variant={coach.status === 'active' ? 'default' : 'destructive'}
                               className={coach.status === 'active' ? 'bg-green-500' : ''}>
                          {coach.status.charAt(0).toUpperCase() + coach.status.slice(1)}
                        </Badge>
                      </div>
                      <div className="grid grid-cols-2 md:grid-cols-5 gap-2 text-sm text-gray-600 dark:text-gray-400 mb-3">
                        <div>{coach.email}</div>
                        <div className="flex items-center gap-1">
                          <MapPin className="w-4 h-4" />
                          {coach.location}, {coach.country}
                        </div>
                        <div className="flex items-center gap-1">
                          <Star className="w-4 h-4 text-yellow-500 fill-yellow-500" />
                          {coach.rating} ({coach.reviewCount})
                        </div>
                        <div>{coach.totalSessions} sessions</div>
                        <div>${coach.totalEarnings.toLocaleString()} earned</div>
                      </div>
                      <div className="flex flex-wrap gap-2">
                        {coach.specialties.map(specialty => (
                          <Badge key={specialty} variant="secondary">{specialty}</Badge>
                        ))}
                      </div>
                    </div>
                    <Button
                      onClick={() => {
                        setSelectedCoach(coach);
                        setShowCoachModal(true);
                      }}
                      variant="outline"
                      size="sm"
                    >
                      Manage
                    </Button>
                  </div>
                </Card>
              ))}
            </div>
          )}
        </TabsContent>

        {/* Sessions Tab */}
        <TabsContent value="sessions" className="space-y-4">
          {filteredSessions.length === 0 ? (
            <Card className="p-12 text-center">
              <Calendar className="w-16 h-16 mx-auto mb-4 text-gray-400" />
              <h3 className="text-lg font-medium mb-2">No sessions found</h3>
              <p className="text-gray-600 dark:text-gray-400">
                {searchQuery ? 'Try adjusting your search' : 'Scheduled sessions will appear here'}
              </p>
            </Card>
          ) : (
            <div className="space-y-3">
              {filteredSessions.map(session => (
                <Card key={session.id} className="p-4">
                  <div className="flex items-start justify-between">
                    <div className="flex-1">
                      <div className="flex items-center gap-3 mb-2">
                        <h3 className="font-medium">{session.title}</h3>
                        <Badge variant={
                          session.status === 'scheduled' ? 'default' :
                          session.status === 'completed' ? 'default' : 'destructive'
                        } className={
                          session.status === 'scheduled' ? 'bg-blue-500' :
                          session.status === 'completed' ? 'bg-green-500' : ''
                        }>
                          {session.status.charAt(0).toUpperCase() + session.status.slice(1)}
                        </Badge>
                        {session.sessionType === 'video' ? (
                          <Video className="w-4 h-4 text-purple-500" />
                        ) : (
                          <Mic className="w-4 h-4 text-blue-500" />
                        )}
                      </div>
                      <div className="text-sm text-gray-600 dark:text-gray-400 mb-2">
                        Coach: {session.coachName}
                      </div>
                      <div className="grid grid-cols-2 md:grid-cols-5 gap-2 text-sm text-gray-600 dark:text-gray-400">
                        <div className="flex items-center gap-1">
                          <Calendar className="w-4 h-4" />
                          {new Date(session.date).toLocaleDateString()}
                        </div>
                        <div className="flex items-center gap-1">
                          <Clock className="w-4 h-4" />
                          {session.time}
                        </div>
                        <div className="flex items-center gap-1">
                          <Users className="w-4 h-4" />
                          {session.currentParticipants}/{session.maxParticipants}
                        </div>
                        <div className="flex items-center gap-1">
                          <DollarSign className="w-4 h-4" />
                          {session.price === 0 ? 'Free' : `$${session.price}`}
                        </div>
                        <div>{session.duration} min</div>
                      </div>
                    </div>
                    <Button
                      onClick={() => {
                        setSelectedSession(session);
                        setShowSessionModal(true);
                      }}
                      variant="outline"
                      size="sm"
                    >
                      Manage
                    </Button>
                  </div>
                </Card>
              ))}
            </div>
          )}
        </TabsContent>
      </Tabs>

      {/* Application Review Modal */}
      <Dialog open={showApplicationModal} onOpenChange={setShowApplicationModal}>
        <DialogContent className="max-w-3xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>Review Coach Application</DialogTitle>
          </DialogHeader>
          {selectedApplication && (
            <div className="space-y-6">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label>Full Name</Label>
                  <div className="font-medium">{selectedApplication.fullName}</div>
                </div>
                <div>
                  <Label>Email</Label>
                  <div className="font-medium">{selectedApplication.email}</div>
                </div>
                <div>
                  <Label>Phone</Label>
                  <div className="font-medium">{selectedApplication.phone}</div>
                </div>
                <div>
                  <Label>Years of Experience</Label>
                  <div className="font-medium">{selectedApplication.yearsExperience} years</div>
                </div>
                <div>
                  <Label>Location</Label>
                  <div className="font-medium">{selectedApplication.location}, {selectedApplication.country}</div>
                </div>
                <div>
                  <Label>Application Date</Label>
                  <div className="font-medium">{new Date(selectedApplication.appliedDate).toLocaleDateString()}</div>
                </div>
              </div>

              <div>
                <Label>Areas of Expertise</Label>
                <div className="flex flex-wrap gap-2 mt-2">
                  {selectedApplication.specialties.map(specialty => (
                    <Badge key={specialty} className="bg-gradient-to-r from-pink-500 to-purple-500 text-white">
                      {specialty}
                    </Badge>
                  ))}
                </div>
              </div>

              <div>
                <Label>Professional Bio</Label>
                <div className="mt-2 p-3 bg-gray-50 dark:bg-gray-900/50 rounded-lg">
                  {selectedApplication.bio}
                </div>
              </div>

              <div>
                <Label>Certifications & Credentials</Label>
                <div className="mt-2 p-3 bg-gray-50 dark:bg-gray-900/50 rounded-lg">
                  {selectedApplication.certifications || 'No certifications provided'}
                </div>
              </div>

              <div>
                <Label>Why They Want to Be a Coach</Label>
                <div className="mt-2 p-3 bg-gray-50 dark:bg-gray-900/50 rounded-lg">
                  {selectedApplication.why}
                </div>
              </div>

              {selectedApplication.status === 'pending' && (
                <div className="flex gap-3 pt-4 border-t dark:border-gray-700">
                  <Button
                    onClick={() => {
                      setShowRejectModal(true);
                    }}
                    variant="destructive"
                    className="flex-1"
                  >
                    <XCircle className="w-4 h-4 mr-2" />
                    Reject Application
                  </Button>
                  <Button
                    onClick={() => handleApproveApplication(selectedApplication.id)}
                    className="flex-1 bg-gradient-to-r from-green-500 to-emerald-500 hover:from-green-600 hover:to-emerald-600"
                  >
                    <CheckCircle className="w-4 h-4 mr-2" />
                    Approve & Create Account
                  </Button>
                </div>
              )}
            </div>
          )}
        </DialogContent>
      </Dialog>

      {/* Coach Management Modal */}
      <Dialog open={showCoachModal} onOpenChange={setShowCoachModal}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>Manage Coach</DialogTitle>
          </DialogHeader>
          {selectedCoach && (
            <div className="space-y-6">
              <div className="flex items-start gap-4">
                <Avatar className="w-20 h-20 ring-2 ring-pink-200 dark:ring-pink-800">
                  <AvatarImage src={selectedCoach.photo} alt={selectedCoach.name} />
                  <AvatarFallback>{selectedCoach.name[0]}</AvatarFallback>
                </Avatar>
                <div className="flex-1">
                  <h3 className="text-xl font-medium mb-2">{selectedCoach.name}</h3>
                  <div className="grid grid-cols-2 gap-2 text-sm text-gray-600 dark:text-gray-400">
                    <div>{selectedCoach.email}</div>
                    <div>{selectedCoach.location}, {selectedCoach.country}</div>
                    <div>Rating: {selectedCoach.rating} ⭐</div>
                    <div>{selectedCoach.totalSessions} sessions</div>
                    <div>${selectedCoach.totalEarnings.toLocaleString()} earned</div>
                    <div>Joined: {new Date(selectedCoach.joinedDate).toLocaleDateString()}</div>
                  </div>
                </div>
              </div>

              <div className="grid grid-cols-2 gap-4">
                <Button
                  variant="outline"
                  onClick={() => toast.info('View full profile functionality')}
                  className="w-full"
                >
                  <Eye className="w-4 h-4 mr-2" />
                  View Full Profile
                </Button>
                <Button
                  variant="outline"
                  onClick={() => toast.info('Drop into session (invisible mode)')}
                  className="w-full"
                >
                  <EyeOff className="w-4 h-4 mr-2" />
                  Join Session Invisibly
                </Button>
              </div>

              <div className="flex gap-3 pt-4 border-t dark:border-gray-700">
                {selectedCoach.status === 'active' ? (
                  <>
                    <Button
                      onClick={() => setShowSuspendModal(true)}
                      variant="outline"
                      className="flex-1 border-orange-300 text-orange-600 hover:bg-orange-50 dark:border-orange-800 dark:text-orange-400"
                    >
                      <AlertTriangle className="w-4 h-4 mr-2" />
                      Suspend Coach
                    </Button>
                    <Button
                      onClick={() => handleDeleteCoach(selectedCoach.id)}
                      variant="destructive"
                      className="flex-1"
                    >
                      <Trash2 className="w-4 h-4 mr-2" />
                      Delete Permanently
                    </Button>
                  </>
                ) : (
                  <>
                    <Button
                      onClick={() => handleReactivateCoach(selectedCoach.id)}
                      className="flex-1 bg-gradient-to-r from-green-500 to-emerald-500"
                    >
                      <CheckCircle className="w-4 h-4 mr-2" />
                      Reactivate Coach
                    </Button>
                    <Button
                      onClick={() => handleDeleteCoach(selectedCoach.id)}
                      variant="destructive"
                      className="flex-1"
                    >
                      <Trash2 className="w-4 h-4 mr-2" />
                      Delete Permanently
                    </Button>
                  </>
                )}
              </div>
            </div>
          )}
        </DialogContent>
      </Dialog>

      {/* Session Management Modal */}
      <Dialog open={showSessionModal} onOpenChange={setShowSessionModal}>
        <DialogContent className="max-w-xl">
          <DialogHeader>
            <DialogTitle>Manage Session</DialogTitle>
          </DialogHeader>
          {selectedSession && (
            <div className="space-y-4">
              <div>
                <h3 className="font-medium text-lg mb-2">{selectedSession.title}</h3>
                <div className="text-sm text-gray-600 dark:text-gray-400">
                  Coach: {selectedSession.coachName}
                </div>
              </div>

              <div className="grid grid-cols-2 gap-4 p-4 bg-gray-50 dark:bg-gray-900/50 rounded-lg text-sm">
                <div>
                  <Label>Date & Time</Label>
                  <div>{new Date(selectedSession.date).toLocaleDateString()} at {selectedSession.time}</div>
                </div>
                <div>
                  <Label>Duration</Label>
                  <div>{selectedSession.duration} minutes</div>
                </div>
                <div>
                  <Label>Participants</Label>
                  <div>{selectedSession.currentParticipants} / {selectedSession.maxParticipants}</div>
                </div>
                <div>
                  <Label>Price</Label>
                  <div>{selectedSession.price === 0 ? 'Free' : `$${selectedSession.price}`}</div>
                </div>
                <div>
                  <Label>Type</Label>
                  <div className="capitalize">{selectedSession.sessionType}</div>
                </div>
                <div>
                  <Label>Category</Label>
                  <div>{selectedSession.category}</div>
                </div>
              </div>

              {selectedSession.status === 'scheduled' && (
                <div className="grid grid-cols-2 gap-3 pt-4 border-t dark:border-gray-700">
                  <Button
                    variant="outline"
                    onClick={() => toast.info('Drop into session (invisible mode)')}
                  >
                    <EyeOff className="w-4 h-4 mr-2" />
                    Join Invisibly
                  </Button>
                  {selectedSession.price > 0 && (
                    <Button
                      variant="outline"
                      onClick={() => handleMarkFree(selectedSession.id)}
                    >
                      <DollarSign className="w-4 h-4 mr-2" />
                      Mark as Free
                    </Button>
                  )}
                  <Button
                    variant="destructive"
                    onClick={() => handleCancelSession(selectedSession.id)}
                    className="col-span-2"
                  >
                    <XCircle className="w-4 h-4 mr-2" />
                    Cancel Session
                  </Button>
                </div>
              )}
            </div>
          )}
        </DialogContent>
      </Dialog>

      {/* Reject Application Modal */}
      <Dialog open={showRejectModal} onOpenChange={setShowRejectModal}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Reject Application</DialogTitle>
            <DialogDescription>
              Provide a reason for rejecting this application. The applicant will receive this feedback.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <Textarea
              placeholder="Enter rejection reason..."
              rows={4}
              value={rejectReason}
              onChange={(e) => setRejectReason(e.target.value)}
            />
            <div className="flex gap-3">
              <Button
                variant="outline"
                onClick={() => {
                  setShowRejectModal(false);
                  setRejectReason('');
                }}
                className="flex-1"
              >
                Cancel
              </Button>
              <Button
                onClick={handleRejectApplication}
                disabled={!rejectReason.trim()}
                variant="destructive"
                className="flex-1"
              >
                Reject Application
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>

      {/* Suspend Coach Modal */}
      <Dialog open={showSuspendModal} onOpenChange={setShowSuspendModal}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Suspend Coach</DialogTitle>
            <DialogDescription>
              Suspending this coach will cancel all their upcoming sessions and prevent them from creating new ones.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <Textarea
              placeholder="Enter reason for suspension..."
              rows={4}
              value={suspendReason}
              onChange={(e) => setSuspendReason(e.target.value)}
            />
            <div className="flex gap-3">
              <Button
                variant="outline"
                onClick={() => {
                  setShowSuspendModal(false);
                  setSuspendReason('');
                }}
                className="flex-1"
              >
                Cancel
              </Button>
              <Button
                onClick={handleSuspendCoach}
                disabled={!suspendReason.trim()}
                variant="destructive"
                className="flex-1"
              >
                Suspend Coach
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>
    </div>
  );
}
