import { useState } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "./ui/card";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Input } from "./ui/input";
import { Label } from "./ui/label";
import { Textarea } from "./ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "./ui/select";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "./ui/dialog";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "./ui/tabs";
import { Alert, AlertDescription } from "./ui/alert";
import { Avatar } from "./ui/avatar";
import { 
  Trophy,
  Plus,
  Edit,
  Trash2,
  Eye,
  Users,
  Vote,
  Crown,
  Calendar,
  Gift,
  Settings,
  CheckCircle,
  XCircle,
  Star,
  Award,
  UserPlus,
  UserMinus,
  Sparkles,
  Heart,
  AlertTriangle,
  Save,
  Clock,
  TrendingUp
} from "lucide-react";
import { ImageWithFallback } from "./figma/ImageWithFallback";
import { useUser } from "./UserProvider";

interface Contest {
  id: string;
  title: string;
  description: string;
  category: 'beauty' | 'personality' | 'photo' | 'story';
  status: 'draft' | 'active' | 'upcoming' | 'ended';
  startDate: string;
  endDate: string;
  prize: string;
  prizeValue: number;
  maxParticipants: number;
  currentParticipants: number;
  totalVotes: number;
  userVotes: number;
  adminVotes: number;
  image: string;
  rules: string[];
  votingType: 'public' | 'admin' | 'mixed';
  createdBy: string;
  createdAt: string;
  participants: ContestParticipant[];
}

interface ContestParticipant {
  id: string;
  userId: string;
  username: string;
  email: string;
  avatar: string;
  photos: string[];
  story?: string;
  votes: number;
  adminVotes: number;
  totalScore: number;
  rank: number;
  isVerified: boolean;
  joinedAt: string;
  status: 'pending' | 'approved' | 'rejected';
}

export function AdminContestsManagement() {
  const { user, isRole } = useUser();
  const [activeTab, setActiveTab] = useState("overview");
  const [selectedContest, setSelectedContest] = useState<string | null>(null);
  const [isCreatingContest, setIsCreatingContest] = useState(false);
  const [isEditingContest, setIsEditingContest] = useState<string | null>(null);

  // Mock contests data
  const [contests, setContests] = useState<Contest[]>([
    {
      id: "1",
      title: "💖 Most Charming Smile",
      description: "Show off your most captivating smile and win amazing prizes!",
      category: "beauty",
      status: "active",
      startDate: "2024-01-15",
      endDate: "2024-01-30",
      prize: "Premium Subscription + $500 Gift Card",
      prizeValue: 500,
      maxParticipants: 100,
      currentParticipants: 67,
      totalVotes: 2134,
      userVotes: 1820,
      adminVotes: 314,
      image: "/api/placeholder/400/300",
      votingType: "mixed",
      createdBy: "admin",
      createdAt: "2024-01-10",
      rules: [
        "Must submit 1-3 photos showing your smile",
        "Photos must be recent (within 6 months)",
        "No filters or heavy editing allowed",
        "Must be a verified member to participate"
      ],
      participants: [
        {
          id: "p1",
          userId: "u1",
          username: "SunnySmile",
          email: "sunny@example.com",
          avatar: "/api/placeholder/50/50",
          photos: ["/api/placeholder/200/300"],
          votes: 345,
          adminVotes: 50,
          totalScore: 395,
          rank: 1,
          isVerified: true,
          joinedAt: "2024-01-16",
          status: "approved"
        },
        {
          id: "p2",
          userId: "u2", 
          username: "CharmingGlow",
          email: "charming@example.com",
          avatar: "/api/placeholder/50/50",
          photos: ["/api/placeholder/200/300"],
          votes: 298,
          adminVotes: 45,
          totalScore: 343,
          rank: 2,
          isVerified: true,
          joinedAt: "2024-01-17",
          status: "approved"
        },
        {
          id: "p3",
          userId: "u3",
          username: "NewParticipant",
          email: "new@example.com",
          avatar: "/api/placeholder/50/50",
          photos: ["/api/placeholder/200/300"],
          votes: 0,
          adminVotes: 0,
          totalScore: 0,
          rank: 0,
          isVerified: false,
          joinedAt: "2024-01-20",
          status: "pending"
        }
      ]
    },
    {
      id: "2",
      title: "✨ Best Love Story",
      description: "Share your most romantic dating experience!",
      category: "story",
      status: "active",
      startDate: "2024-01-10",
      endDate: "2024-01-25",
      prize: "Romantic Getaway for Two",
      prizeValue: 1000,
      maxParticipants: 50,
      currentParticipants: 32,
      totalVotes: 1456,
      userVotes: 1201,
      adminVotes: 255,
      image: "/api/placeholder/400/300",
      votingType: "mixed",
      createdBy: "admin",
      createdAt: "2024-01-05",
      rules: [
        "Submit a written story (500-1000 words)",
        "Story must be original and personal",
        "Appropriate content only"
      ],
      participants: []
    },
    {
      id: "3",
      title: "🌟 Rising Star",
      description: "New members showcase contest",
      category: "personality",
      status: "draft",
      startDate: "2024-02-01",
      endDate: "2024-02-15",
      prize: "1 Year Premium + Shopping Spree",
      prizeValue: 300,
      maxParticipants: 75,
      currentParticipants: 0,
      totalVotes: 0,
      userVotes: 0,
      adminVotes: 0,
      image: "/api/placeholder/400/300",
      votingType: "public",
      createdBy: "admin",
      createdAt: "2024-01-20",
      rules: [],
      participants: []
    }
  ]);

  // New contest form state
  const [newContest, setNewContest] = useState({
    title: "",
    description: "",
    category: "beauty" as Contest['category'],
    startDate: "",
    endDate: "",
    prize: "",
    prizeValue: 0,
    maxParticipants: 100,
    votingType: "mixed" as Contest['votingType'],
    rules: [""],
    image: ""
  });

  const getStatusBadge = (status: Contest['status']) => {
    switch (status) {
      case 'active':
        return <Badge className="bg-green-100 text-green-800 dark:bg-green-900/20">🟢 Active</Badge>;
      case 'upcoming':
        return <Badge className="bg-blue-100 text-blue-800 dark:bg-blue-900/20">🔵 Upcoming</Badge>;
      case 'ended':
        return <Badge className="bg-gray-100 text-gray-800 dark:bg-gray-900/20">⚫ Ended</Badge>;
      case 'draft':
        return <Badge className="bg-orange-100 text-orange-800 dark:bg-orange-900/20">🟠 Draft</Badge>;
      default:
        return <Badge>{status}</Badge>;
    }
  };

  const getCategoryIcon = (category: Contest['category']) => {
    switch (category) {
      case 'beauty':
        return <Sparkles className="w-4 h-4" />;
      case 'personality':
        return <Heart className="w-4 h-4" />;
      case 'photo':
        return <Eye className="w-4 h-4" />;
      case 'story':
        return <Award className="w-4 h-4" />;
      default:
        return <Trophy className="w-4 h-4" />;
    }
  };

  const handleCreateContest = () => {
    const contestId = Date.now().toString();
    const contest: Contest = {
      id: contestId,
      ...newContest,
      status: 'draft',
      currentParticipants: 0,
      totalVotes: 0,
      userVotes: 0,
      adminVotes: 0,
      createdBy: user?.username || 'admin',
      createdAt: new Date().toISOString().split('T')[0],
      participants: []
    };
    
    setContests(prev => [...prev, contest]);
    setIsCreatingContest(false);
    setNewContest({
      title: "",
      description: "",
      category: "beauty",
      startDate: "",
      endDate: "",
      prize: "",
      prizeValue: 0,
      maxParticipants: 100,
      votingType: "mixed",
      rules: [""],
      image: ""
    });
  };

  const handleDeleteContest = (contestId: string) => {
    setContests(prev => prev.filter(c => c.id !== contestId));
  };

  const handleUpdateContestStatus = (contestId: string, status: Contest['status']) => {
    setContests(prev => prev.map(c => 
      c.id === contestId ? { ...c, status } : c
    ));
  };

  const handleApproveParticipant = (contestId: string, participantId: string) => {
    setContests(prev => prev.map(contest => 
      contest.id === contestId 
        ? {
            ...contest,
            participants: contest.participants.map(p => 
              p.id === participantId ? { ...p, status: 'approved' as const } : p
            )
          }
        : contest
    ));
  };

  const handleRejectParticipant = (contestId: string, participantId: string) => {
    setContests(prev => prev.map(contest => 
      contest.id === contestId 
        ? {
            ...contest,
            participants: contest.participants.map(p => 
              p.id === participantId ? { ...p, status: 'rejected' as const } : p
            )
          }
        : contest
    ));
  };

  const handleAddAdminVote = (contestId: string, participantId: string, votes: number) => {
    setContests(prev => prev.map(contest => 
      contest.id === contestId 
        ? {
            ...contest,
            participants: contest.participants.map(p => 
              p.id === participantId 
                ? { 
                    ...p, 
                    adminVotes: p.adminVotes + votes,
                    totalScore: p.votes + p.adminVotes + votes
                  } 
                : p
            ),
            adminVotes: contest.adminVotes + votes,
            totalVotes: contest.totalVotes + votes
          }
        : contest
    ));
  };

  const activeContests = contests.filter(c => c.status === 'active');
  const draftContests = contests.filter(c => c.status === 'draft');
  const endedContests = contests.filter(c => c.status === 'ended');
  const totalParticipants = contests.reduce((sum, c) => sum + c.currentParticipants, 0);
  const totalVotes = contests.reduce((sum, c) => sum + c.totalVotes, 0);

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-2xl font-bold flex items-center gap-2">
            <Trophy className="w-6 h-6 text-yellow-500" />
            Contest Management
          </h2>
          <p className="text-muted-foreground">Create and manage community contests</p>
        </div>
        <Dialog open={isCreatingContest} onOpenChange={setIsCreatingContest}>
          <DialogTrigger asChild>
            <Button className="bg-purple-600 hover:bg-purple-700">
              <Plus className="w-4 h-4 mr-2" />
              Create Contest
            </Button>
          </DialogTrigger>
          <DialogContent className="max-w-2xl">
            <DialogHeader>
              <DialogTitle>Create New Contest</DialogTitle>
              <DialogDescription>
                Set up a new contest for community members to participate in
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4 max-h-[60vh] overflow-y-auto">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="title">Contest Title</Label>
                  <Input
                    id="title"
                    value={newContest.title}
                    onChange={(e) => setNewContest(prev => ({ ...prev, title: e.target.value }))}
                    placeholder="e.g., Most Charming Smile"
                  />
                </div>
                <div>
                  <Label htmlFor="category">Category</Label>
                  <Select 
                    value={newContest.category} 
                    onValueChange={(value: Contest['category']) => 
                      setNewContest(prev => ({ ...prev, category: value }))
                    }
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="beauty">Beauty</SelectItem>
                      <SelectItem value="personality">Personality</SelectItem>
                      <SelectItem value="photo">Photo</SelectItem>
                      <SelectItem value="story">Story</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>

              <div>
                <Label htmlFor="description">Description</Label>
                <Textarea
                  id="description"
                  value={newContest.description}
                  onChange={(e) => setNewContest(prev => ({ ...prev, description: e.target.value }))}
                  placeholder="Describe what this contest is about..."
                  rows={3}
                />
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="startDate">Start Date</Label>
                  <Input
                    id="startDate"
                    type="date"
                    value={newContest.startDate}
                    onChange={(e) => setNewContest(prev => ({ ...prev, startDate: e.target.value }))}
                  />
                </div>
                <div>
                  <Label htmlFor="endDate">End Date</Label>
                  <Input
                    id="endDate"
                    type="date"
                    value={newContest.endDate}
                    onChange={(e) => setNewContest(prev => ({ ...prev, endDate: e.target.value }))}
                  />
                </div>
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="prize">Prize Description</Label>
                  <Input
                    id="prize"
                    value={newContest.prize}
                    onChange={(e) => setNewContest(prev => ({ ...prev, prize: e.target.value }))}
                    placeholder="e.g., Premium Subscription + $500"
                  />
                </div>
                <div>
                  <Label htmlFor="prizeValue">Prize Value ($)</Label>
                  <Input
                    id="prizeValue"
                    type="number"
                    value={newContest.prizeValue}
                    onChange={(e) => setNewContest(prev => ({ ...prev, prizeValue: Number(e.target.value) }))}
                    placeholder="0"
                  />
                </div>
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="maxParticipants">Max Participants</Label>
                  <Input
                    id="maxParticipants"
                    type="number"
                    value={newContest.maxParticipants}
                    onChange={(e) => setNewContest(prev => ({ ...prev, maxParticipants: Number(e.target.value) }))}
                  />
                </div>
                <div>
                  <Label htmlFor="votingType">Voting Type</Label>
                  <Select 
                    value={newContest.votingType} 
                    onValueChange={(value: Contest['votingType']) => 
                      setNewContest(prev => ({ ...prev, votingType: value }))
                    }
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="public">Public Voting Only</SelectItem>
                      <SelectItem value="admin">Admin Voting Only</SelectItem>
                      <SelectItem value="mixed">Mixed (Public + Admin)</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>

              <div>
                <Label>Contest Rules</Label>
                <div className="space-y-2">
                  {newContest.rules.map((rule, index) => (
                    <div key={index} className="flex gap-2">
                      <Input
                        value={rule}
                        onChange={(e) => {
                          const newRules = [...newContest.rules];
                          newRules[index] = e.target.value;
                          setNewContest(prev => ({ ...prev, rules: newRules }));
                        }}
                        placeholder={`Rule ${index + 1}`}
                      />
                      {newContest.rules.length > 1 && (
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => {
                            const newRules = newContest.rules.filter((_, i) => i !== index);
                            setNewContest(prev => ({ ...prev, rules: newRules }));
                          }}
                        >
                          <Trash2 className="w-4 h-4" />
                        </Button>
                      )}
                    </div>
                  ))}
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => setNewContest(prev => ({ ...prev, rules: [...prev.rules, ""] }))}
                  >
                    <Plus className="w-4 h-4 mr-2" />
                    Add Rule
                  </Button>
                </div>
              </div>

              <div className="flex gap-2 pt-4">
                <Button onClick={handleCreateContest} className="flex-1">
                  <Save className="w-4 h-4 mr-2" />
                  Create Contest
                </Button>
                <Button variant="outline" onClick={() => setIsCreatingContest(false)}>
                  Cancel
                </Button>
              </div>
            </div>
          </DialogContent>
        </Dialog>
      </div>

      {/* Quick Stats */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-muted-foreground">Active Contests</p>
                <p className="text-2xl font-bold">{activeContests.length}</p>
              </div>
              <Trophy className="w-8 h-8 text-green-600" />
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-muted-foreground">Total Participants</p>
                <p className="text-2xl font-bold">{totalParticipants}</p>
              </div>
              <Users className="w-8 h-8 text-blue-600" />
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-muted-foreground">Total Votes</p>
                <p className="text-2xl font-bold">{totalVotes.toLocaleString()}</p>
              </div>
              <Vote className="w-8 h-8 text-purple-600" />
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-muted-foreground">Draft Contests</p>
                <p className="text-2xl font-bold">{draftContests.length}</p>
              </div>
              <Settings className="w-8 h-8 text-orange-600" />
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Main Content Tabs */}
      <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-6">
        <TabsList className="grid w-full grid-cols-4">
          <TabsTrigger value="overview">Overview</TabsTrigger>
          <TabsTrigger value="active">Active Contests</TabsTrigger>
          <TabsTrigger value="participants">Participants</TabsTrigger>
          <TabsTrigger value="voting">Voting Management</TabsTrigger>
        </TabsList>

        {/* Overview Tab */}
        <TabsContent value="overview" className="space-y-6">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            {/* All Contests List */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <TrendingUp className="w-5 h-5" />
                  All Contests
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-3">
                {contests.map((contest) => (
                  <div key={contest.id} className="flex items-center justify-between p-3 border rounded-lg">
                    <div className="flex items-center gap-3">
                      <div className="p-2 bg-muted rounded-lg">
                        {getCategoryIcon(contest.category)}
                      </div>
                      <div>
                        <p className="font-medium">{contest.title}</p>
                        <p className="text-sm text-muted-foreground">
                          {contest.currentParticipants} participants • {contest.totalVotes} votes
                        </p>
                      </div>
                    </div>
                    <div className="flex items-center gap-2">
                      {getStatusBadge(contest.status)}
                      <Button variant="ghost" size="sm">
                        <Eye className="w-4 h-4" />
                      </Button>
                    </div>
                  </div>
                ))}
              </CardContent>
            </Card>

            {/* Recent Activity */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Activity className="w-5 h-5" />
                  Recent Activity
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-3">
                <div className="space-y-2">
                  <div className="flex items-center gap-2 text-sm">
                    <UserPlus className="w-4 h-4 text-green-600" />
                    <span><strong>NewParticipant</strong> joined "Most Charming Smile"</span>
                    <span className="text-muted-foreground text-xs ml-auto">2h ago</span>
                  </div>
                  <div className="flex items-center gap-2 text-sm">
                    <Vote className="w-4 h-4 text-purple-600" />
                    <span><strong>123 votes</strong> cast in "Best Love Story"</span>
                    <span className="text-muted-foreground text-xs ml-auto">4h ago</span>
                  </div>
                  <div className="flex items-center gap-2 text-sm">
                    <Trophy className="w-4 h-4 text-yellow-600" />
                    <span>Contest "Rising Star" created</span>
                    <span className="text-muted-foreground text-xs ml-auto">1d ago</span>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        {/* Active Contests Tab */}
        <TabsContent value="active" className="space-y-6">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            {contests.map((contest) => (
              <Card key={contest.id} className="overflow-hidden">
                <div className="relative">
                  <ImageWithFallback
                    src={contest.image}
                    alt={contest.title}
                    className="w-full h-32 object-cover"
                  />
                  <div className="absolute top-2 left-2">
                    {getStatusBadge(contest.status)}
                  </div>
                  <div className="absolute top-2 right-2">
                    <Badge className="bg-white/90 text-gray-800">
                      <Gift className="w-3 h-3 mr-1" />
                      ${contest.prizeValue}
                    </Badge>
                  </div>
                </div>

                <CardHeader className="pb-2">
                  <CardTitle className="text-lg">{contest.title}</CardTitle>
                  <CardDescription>{contest.description}</CardDescription>
                </CardHeader>

                <CardContent className="space-y-4">
                  {/* Stats */}
                  <div className="grid grid-cols-3 gap-2 text-center">
                    <div className="p-2 bg-muted/50 rounded">
                      <p className="text-lg font-bold">{contest.currentParticipants}</p>
                      <p className="text-xs text-muted-foreground">Participants</p>
                    </div>
                    <div className="p-2 bg-muted/50 rounded">
                      <p className="text-lg font-bold">{contest.totalVotes}</p>
                      <p className="text-xs text-muted-foreground">Total Votes</p>
                    </div>
                    <div className="p-2 bg-muted/50 rounded">
                      <p className="text-lg font-bold">{contest.adminVotes}</p>
                      <p className="text-xs text-muted-foreground">Admin Votes</p>
                    </div>
                  </div>

                  {/* Actions */}
                  <div className="flex gap-2">
                    <Button variant="outline" size="sm" className="flex-1">
                      <Eye className="w-4 h-4 mr-2" />
                      View
                    </Button>
                    <Button variant="outline" size="sm" className="flex-1">
                      <Edit className="w-4 h-4 mr-2" />
                      Edit
                    </Button>
                    {contest.status === 'draft' && (
                      <Button 
                        size="sm" 
                        className="flex-1"
                        onClick={() => handleUpdateContestStatus(contest.id, 'active')}
                      >
                        <CheckCircle className="w-4 h-4 mr-2" />
                        Publish
                      </Button>
                    )}
                    {contest.status === 'active' && (
                      <Button 
                        variant="destructive" 
                        size="sm"
                        onClick={() => handleUpdateContestStatus(contest.id, 'ended')}
                      >
                        <XCircle className="w-4 h-4 mr-2" />
                        End
                      </Button>
                    )}
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </TabsContent>

        {/* Participants Tab */}
        <TabsContent value="participants" className="space-y-6">
          {contests.filter(c => c.participants.length > 0).map((contest) => (
            <Card key={contest.id}>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Trophy className="w-5 h-5" />
                  {contest.title} - Participants
                </CardTitle>
                <CardDescription>
                  Manage participants and their submissions
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  {contest.participants.map((participant) => (
                    <div key={participant.id} className="flex items-center justify-between p-4 border rounded-lg">
                      <div className="flex items-center gap-4">
                        <Avatar className="w-10 h-10">
                          <ImageWithFallback 
                            src={participant.avatar} 
                            alt={participant.username}
                            className="w-full h-full object-cover"
                          />
                        </Avatar>
                        <div>
                          <div className="flex items-center gap-2">
                            <p className="font-medium">{participant.username}</p>
                            {participant.isVerified && (
                              <CheckCircle className="w-4 h-4 text-blue-600" />
                            )}
                          </div>
                          <p className="text-sm text-muted-foreground">
                            {participant.email} • Joined {participant.joinedAt}
                          </p>
                          <div className="flex items-center gap-4 text-sm mt-1">
                            <span>User Votes: {participant.votes}</span>
                            <span>Admin Votes: {participant.adminVotes}</span>
                            <span className="font-medium">Total: {participant.totalScore}</span>
                          </div>
                        </div>
                      </div>
                      <div className="flex items-center gap-2">
                        <Badge className={
                          participant.status === 'approved' ? 'bg-green-100 text-green-800' :
                          participant.status === 'rejected' ? 'bg-red-100 text-red-800' :
                          'bg-yellow-100 text-yellow-800'
                        }>
                          {participant.status}
                        </Badge>
                        {participant.status === 'pending' && (
                          <>
                            <Button 
                              size="sm" 
                              className="bg-green-600 hover:bg-green-700"
                              onClick={() => handleApproveParticipant(contest.id, participant.id)}
                            >
                              <CheckCircle className="w-4 h-4" />
                            </Button>
                            <Button 
                              size="sm" 
                              variant="destructive"
                              onClick={() => handleRejectParticipant(contest.id, participant.id)}
                            >
                              <XCircle className="w-4 h-4" />
                            </Button>
                          </>
                        )}
                        <Button variant="outline" size="sm">
                          <Eye className="w-4 h-4" />
                        </Button>
                        <Button variant="outline" size="sm">
                          <UserMinus className="w-4 h-4" />
                        </Button>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          ))}
        </TabsContent>

        {/* Voting Management Tab */}
        <TabsContent value="voting" className="space-y-6">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            {contests.filter(c => c.status === 'active').map((contest) => (
              <Card key={contest.id}>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Vote className="w-5 h-5" />
                    {contest.title} - Admin Voting
                  </CardTitle>
                  <CardDescription>
                    Cast admin votes to influence contest results
                  </CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  {contest.participants.filter(p => p.status === 'approved').map((participant) => (
                    <div key={participant.id} className="flex items-center justify-between p-3 border rounded-lg">
                      <div className="flex items-center gap-3">
                        <Avatar className="w-8 h-8">
                          <ImageWithFallback 
                            src={participant.avatar} 
                            alt={participant.username}
                            className="w-full h-full object-cover"
                          />
                        </Avatar>
                        <div>
                          <p className="font-medium">{participant.username}</p>
                          <p className="text-sm text-muted-foreground">
                            Rank #{participant.rank || 'Unranked'} • Score: {participant.totalScore}
                          </p>
                        </div>
                      </div>
                      <div className="flex items-center gap-2">
                        <Button 
                          size="sm" 
                          variant="outline"
                          onClick={() => handleAddAdminVote(contest.id, participant.id, 1)}
                        >
                          <Star className="w-4 h-4 mr-1" />
                          +1
                        </Button>
                        <Button 
                          size="sm" 
                          variant="outline"
                          onClick={() => handleAddAdminVote(contest.id, participant.id, 5)}
                        >
                          <Crown className="w-4 h-4 mr-1" />
                          +5
                        </Button>
                        <Button 
                          size="sm" 
                          className="bg-yellow-600 hover:bg-yellow-700"
                          onClick={() => handleAddAdminVote(contest.id, participant.id, 10)}
                        >
                          <Award className="w-4 h-4 mr-1" />
                          +10
                        </Button>
                      </div>
                    </div>
                  ))}
                </CardContent>
              </Card>
            ))}
          </div>
        </TabsContent>
      </Tabs>
    </div>
  );
}