import { useState } from "react";
import { Header } from "./Header";
import { Footer } from "./Footer";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "./ui/card";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "./ui/tabs";
import { Alert, AlertDescription } from "./ui/alert";
import { Separator } from "./ui/separator";
import {
  Users,
  Settings,
  Shield,
  BarChart3,
  Flag,
  Mail,
  Crown,
  Ban,
  CheckCircle,
  AlertTriangle,
  Eye,
  Edit,
  Trash2,
  UserCheck,
  UserX,
  MessageSquare,
  Calendar,
  TrendingUp,
  DollarSign,
  Activity,
  Database,
  FileText,
  Globe,
  Cookie,
  Scale,
  Lock,
  Save,
  ChevronRight,
  HelpCircle,
  Heart,
  Phone,
  Megaphone,
  Coins,
  Facebook,
  Twitter,
  Instagram,
  Youtube,
  ExternalLink,
  Link,
  Trophy,
  Award,
  Gamepad2
} from "lucide-react";
import { useTranslation } from "./TranslationProvider";
import { useUser } from "./UserProvider";
import { Input } from "./ui/input";
import { Label } from "./ui/label";
import { Textarea } from "./ui/textarea";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "./ui/dialog";
import { AdminUsernameRequests } from "./AdminUsernameRequests";
import { AdminSupportTab } from "./AdminSupportTab";
import { AdminGroupsManagement } from "./AdminGroupsManagement";
import { AdminAdManagement } from "./AdminAdManagement";
import { AdminPointsManagement } from "./AdminPointsManagement";
import { AdminVerificationManagement } from "./AdminVerificationManagement";
import { AdminContestsManagement } from "./AdminContestsManagement";
import AdminCoachManagement from "./AdminCoachManagement";
import { AdminGamesManagement } from "./AdminGamesManagement";

interface AdminControlPanelProps {
  onNavigate: (view: string) => void;
  onOpenCookieConsent?: () => void;
}

export function AdminControlPanel({ onNavigate, onOpenCookieConsent }: AdminControlPanelProps) {
  const { t } = useTranslation();
  const { user, isRole, hasPermission } = useUser();
  const [activeTab, setActiveTab] = useState("overview");
  const [editingLegalPage, setEditingLegalPage] = useState<string | null>(null);
  
  // Legal pages content state
  const [legalPages, setLegalPages] = useState({
    privacy: {
      title: "Privacy Policy",
      lastUpdated: "December 20, 2024",
      content: "This is the privacy policy content...",
      status: "published"
    },
    terms: {
      title: "Terms of Service", 
      lastUpdated: "December 20, 2024",
      content: "This is the terms of service content...",
      status: "published"
    },
    cookies: {
      title: "Cookie Policy",
      lastUpdated: "December 20, 2024", 
      content: "This is the cookie policy content...",
      status: "published"
    }
  });

  // Support pages content state
  const [supportPages, setSupportPages] = useState({
    helpCenter: {
      title: "Help Center",
      lastUpdated: "December 20, 2024",
      content: "This is the help center content...",
      status: "published"
    },
    safetyTips: {
      title: "Safety Tips", 
      lastUpdated: "December 20, 2024",
      content: "This is the safety tips content...",
      status: "published"
    },
    communityGuidelines: {
      title: "Community Guidelines",
      lastUpdated: "December 20, 2024", 
      content: "This is the community guidelines content...",
      status: "published"
    }
  });

  // Social media links state
  const [socialMediaLinks, setSocialMediaLinks] = useState({
    hotzonne: "https://www.hotzonne.com",
    facebook: "https://www.facebook.com/hotzonne",
    twitter: "https://www.x.com/hotzonne",
    instagram: "https://www.instagram.com/hotzonne",
    youtube: "https://www.youtube.com/hotzonne",
    tiktok: "https://www.tiktok.com/@hotzonne"
  });

  // Mock admin data
  const adminStats = {
    totalUsers: 45321,
    activeUsers: 12847,
    newUsers: 234,
    premiumUsers: 8439,
    reportedUsers: 23,
    pendingReports: 15,
    totalMatches: 125473,
    totalMessages: 892364,
    revenue: 89234.50,
    conversionRate: 18.7
  };

  const recentUsers = [
    { id: 1, name: "Emily Johnson", email: "emily.j@example.com", status: "active", premium: true, joined: "2024-01-15" },
    { id: 2, name: "Michael Chen", email: "m.chen@example.com", status: "suspended", premium: false, joined: "2024-01-14" },
    { id: 3, name: "Sarah Williams", email: "sarah.w@example.com", status: "active", premium: true, joined: "2024-01-14" },
    { id: 4, name: "James Rodriguez", email: "j.rodriguez@example.com", status: "pending", premium: false, joined: "2024-01-13" },
    { id: 5, name: "Lisa Anderson", email: "lisa.a@example.com", status: "active", premium: false, joined: "2024-01-13" }
  ];

  const recentReports = [
    { id: 1, reported: "John Doe", reporter: "Jane Smith", reason: "Inappropriate content", status: "pending", date: "2024-01-15" },
    { id: 2, reported: "Mike Johnson", reporter: "Sarah Lee", reason: "Harassment", status: "investigating", date: "2024-01-15" },
    { id: 3, reported: "Alex Brown", reporter: "Chris Wilson", reason: "Fake profile", status: "resolved", date: "2024-01-14" },
    { id: 4, reported: "Emma Davis", reporter: "Tom Miller", reason: "Spam messages", status: "pending", date: "2024-01-14" }
  ];

  const getStatusBadge = (status: string) => {
    switch (status) {
      case "active":
        return <Badge className="bg-green-100 text-green-800">Active</Badge>;
      case "suspended":
        return <Badge className="bg-red-100 text-red-800">Suspended</Badge>;
      case "pending":
        return <Badge className="bg-yellow-100 text-yellow-800">Pending</Badge>;
      case "investigating":
        return <Badge className="bg-blue-100 text-blue-800">Investigating</Badge>;
      case "resolved":
        return <Badge className="bg-gray-100 text-gray-800">Resolved</Badge>;
      default:
        return <Badge>{status}</Badge>;
    }
  };

  return (
    <div className="min-h-screen bg-background">
      <Header currentView="admin-panel" onNavigate={onNavigate} />
      
      <div className="max-w-7xl mx-auto px-6 py-8">
        {/* Header */}
        <div className="mb-8">
          <div className="flex items-center gap-3 mb-2">
            {isRole('admin') ? (
              <Shield className="w-8 h-8 text-purple-600" />
            ) : (
              <UserCheck className="w-8 h-8 text-blue-600" />
            )}
            <h1 className="text-3xl font-bold">
              {isRole('admin') ? 'Admin Control Panel' : 'Moderation Panel'}
            </h1>
            <Badge className={isRole('admin') ? "bg-purple-100 text-purple-800" : "bg-blue-100 text-blue-800"}>
              {user?.role === 'admin' ? 'Admin Access' : 'Moderator Access'}
            </Badge>
          </div>
          <p className="text-muted-foreground">
            {isRole('admin') 
              ? 'Manage users, monitor reports, and oversee platform operations'
              : 'Monitor content, review reports, and moderate user interactions'
            }
          </p>
        </div>

        {/* Quick Stats */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-muted-foreground">Total Users</p>
                  <p className="text-2xl font-bold">{adminStats.totalUsers.toLocaleString()}</p>
                </div>
                <Users className="w-8 h-8 text-blue-600" />
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-muted-foreground">Active Users</p>
                  <p className="text-2xl font-bold">{adminStats.activeUsers.toLocaleString()}</p>
                </div>
                <Activity className="w-8 h-8 text-green-600" />
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-muted-foreground">Premium Users</p>
                  <p className="text-2xl font-bold">{adminStats.premiumUsers.toLocaleString()}</p>
                </div>
                <Crown className="w-8 h-8 text-yellow-600" />
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-muted-foreground">Pending Reports</p>
                  <p className="text-2xl font-bold text-red-600">{adminStats.pendingReports}</p>
                </div>
                <Flag className="w-8 h-8 text-red-600" />
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Admin/Moderator Tabs */}
        <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-6">
          <TabsList className={`grid w-full ${isRole('admin') ? 'grid-cols-17' : 'grid-cols-14'}`}>
            <TabsTrigger value="overview">Overview</TabsTrigger>
            {hasPermission('user-management') && <TabsTrigger value="users">Users</TabsTrigger>}
            {hasPermission('user-management') && <TabsTrigger value="username-requests">Usernames</TabsTrigger>}
            {hasPermission('user-management') && <TabsTrigger value="verification">Verification</TabsTrigger>}
            <TabsTrigger value="reports">Reports</TabsTrigger>
            <TabsTrigger value="content">Content</TabsTrigger>
            <TabsTrigger value="contests">Contests</TabsTrigger>
            <TabsTrigger value="coaches">Dating Coaches</TabsTrigger>
            <TabsTrigger value="games">Dating Games</TabsTrigger>
            <TabsTrigger value="ads">Ads</TabsTrigger>
            <TabsTrigger value="points">Points</TabsTrigger>
            <TabsTrigger value="legal">Legal</TabsTrigger>
            <TabsTrigger value="support">Support</TabsTrigger>
            <TabsTrigger value="social">Social</TabsTrigger>
            {hasPermission('user-management') && <TabsTrigger value="groups">Groups</TabsTrigger>}
            {hasPermission('analytics') && <TabsTrigger value="analytics">Analytics</TabsTrigger>}
            {hasPermission('settings') && <TabsTrigger value="settings">Settings</TabsTrigger>}
          </TabsList>

          {/* Overview Tab */}
          <TabsContent value="overview">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <TrendingUp className="w-5 h-5" />
                    Platform Metrics
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="flex justify-between">
                    <span>Total Matches</span>
                    <span className="font-semibold">{adminStats.totalMatches.toLocaleString()}</span>
                  </div>
                  <div className="flex justify-between">
                    <span>Total Messages</span>
                    <span className="font-semibold">{adminStats.totalMessages.toLocaleString()}</span>
                  </div>
                  <div className="flex justify-between">
                    <span>Conversion Rate</span>
                    <span className="font-semibold text-green-600">{adminStats.conversionRate}%</span>
                  </div>
                  <div className="flex justify-between">
                    <span>Monthly Revenue</span>
                    <span className="font-semibold text-green-600">${adminStats.revenue.toLocaleString()}</span>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <AlertTriangle className="w-5 h-5" />
                    Recent Alerts
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-3">
                  <Alert>
                    <AlertTriangle className="w-4 h-4" />
                    <AlertDescription>
                      High number of reports in the last 24 hours (15 pending)
                    </AlertDescription>
                  </Alert>
                  <Alert>
                    <AlertTriangle className="w-4 h-4" />
                    <AlertDescription>
                      Server load is above normal threshold
                    </AlertDescription>
                  </Alert>
                  <Alert>
                    <AlertTriangle className="w-4 h-4" />
                    <AlertDescription>
                      Unusual spike in new user registrations
                    </AlertDescription>
                  </Alert>
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          {/* Users Tab */}
          <TabsContent value="users">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Users className="w-5 h-5" />
                  Recent Users
                </CardTitle>
                <CardDescription>
                  Manage and monitor user accounts
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {recentUsers.map((user) => (
                    <div key={user.id} className="flex items-center justify-between p-4 border rounded-lg">
                      <div className="flex items-center gap-4">
                        <div>
                          <div className="flex items-center gap-2">
                            <p className="font-medium">{user.name}</p>
                            {user.premium && <Crown className="w-4 h-4 text-yellow-500" />}
                          </div>
                          <p className="text-sm text-muted-foreground">{user.email}</p>
                          <p className="text-xs text-muted-foreground">Joined: {user.joined}</p>
                        </div>
                      </div>
                      <div className="flex items-center gap-3">
                        {getStatusBadge(user.status)}
                        <div className="flex gap-2">
                          <Button size="sm" variant="outline">
                            <Eye className="w-4 h-4" />
                          </Button>
                          <Button size="sm" variant="outline">
                            <Edit className="w-4 h-4" />
                          </Button>
                          <Button size="sm" variant="outline" className="text-red-600">
                            <Ban className="w-4 h-4" />
                          </Button>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
                <div className="mt-6 flex justify-between">
                  <Button variant="outline">Export Users</Button>
                  <Button>View All Users</Button>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Username Requests Tab */}
          <TabsContent value="username-requests">
            <AdminUsernameRequests />
          </TabsContent>

          {/* Verification Tab */}
          <TabsContent value="verification">
            <AdminVerificationManagement />
          </TabsContent>

          {/* Reports Tab */}
          <TabsContent value="reports">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Flag className="w-5 h-5" />
                  Recent Reports
                </CardTitle>
                <CardDescription>
                  Review and manage user reports
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {recentReports.map((report) => (
                    <div key={report.id} className="flex items-center justify-between p-4 border rounded-lg">
                      <div>
                        <p className="font-medium">Report against {report.reported}</p>
                        <p className="text-sm text-muted-foreground">
                          Reported by {report.reporter} • {report.reason}
                        </p>
                        <p className="text-xs text-muted-foreground">{report.date}</p>
                      </div>
                      <div className="flex items-center gap-3">
                        {getStatusBadge(report.status)}
                        <div className="flex gap-2">
                          <Button size="sm" variant="outline">
                            <Eye className="w-4 h-4" />
                          </Button>
                          <Button size="sm" className="bg-green-600 hover:bg-green-700">
                            <CheckCircle className="w-4 h-4" />
                          </Button>
                          <Button size="sm" variant="destructive">
                            <Ban className="w-4 h-4" />
                          </Button>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
                <div className="mt-6 flex justify-between">
                  <Button variant="outline">Export Reports</Button>
                  <Button>View All Reports</Button>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Content Tab */}
          <TabsContent value="content">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <FileText className="w-5 h-5" />
                    Content Moderation
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <Button className="w-full justify-start">
                    <MessageSquare className="w-4 h-4 mr-2" />
                    Review Flagged Messages (23)
                  </Button>
                  <Button className="w-full justify-start">
                    <Eye className="w-4 h-4 mr-2" />
                    Review Flagged Photos (12)
                  </Button>
                  <Button 
                    className="w-full justify-start"
                    onClick={() => setActiveTab("verification")}
                  >
                    <UserCheck className="w-4 h-4 mr-2" />
                    Identity Verification (8)
                    <ChevronRight className="w-4 h-4 ml-auto" />
                  </Button>
                  <Button className="w-full justify-start">
                    <Shield className="w-4 h-4 mr-2" />
                    Automated Filters Settings
                  </Button>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Globe className="w-5 h-5" />
                    Platform Content
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <Button className="w-full justify-start">
                    <Mail className="w-4 h-4 mr-2" />
                    Manage Email Templates
                  </Button>
                  <Button 
                    className="w-full justify-start"
                    onClick={() => setActiveTab("ads")}
                  >
                    <Megaphone className="w-4 h-4 mr-2" />
                    Manage Member Ads
                    <ChevronRight className="w-4 h-4 ml-auto" />
                  </Button>
                  <Button 
                    className="w-full justify-start"
                    onClick={() => setActiveTab("points")}
                  >
                    <Coins className="w-4 h-4 mr-2" />
                    Points & Rewards
                    <ChevronRight className="w-4 h-4 ml-auto" />
                  </Button>
                  <Button 
                    className="w-full justify-start"
                    onClick={() => setActiveTab("legal")}
                  >
                    <FileText className="w-4 h-4 mr-2" />
                    Manage Legal Pages
                    <ChevronRight className="w-4 h-4 ml-auto" />
                  </Button>
                  <Button className="w-full justify-start">
                    <Settings className="w-4 h-4 mr-2" />
                    App Announcements
                  </Button>
                  <Button className="w-full justify-start">
                    <Database className="w-4 h-4 mr-2" />
                    Backup & Restore
                  </Button>
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          {/* Contests Tab */}
          <TabsContent value="contests">
            <AdminContestsManagement />
          </TabsContent>

          {/* Dating Coaches Tab */}
          <TabsContent value="coaches">
            <AdminCoachManagement />
          </TabsContent>

          {/* Dating Games Tab */}
          <TabsContent value="games">
            <AdminGamesManagement />
          </TabsContent>

          {/* Ads Tab */}
          <TabsContent value="ads">
            <AdminAdManagement />
          </TabsContent>

          {/* Points Tab */}
          <TabsContent value="points">
            <AdminPointsManagement />
          </TabsContent>

          {/* Legal Pages Tab */}
          <TabsContent value="legal">
            <div className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Scale className="w-5 h-5" />
                    Legal Pages Management
                  </CardTitle>
                  <CardDescription>
                    Manage Privacy Policy, Terms of Service, and Cookie Policy content
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                    {/* Privacy Policy */}
                    <Card className="border-2">
                      <CardHeader className="pb-3">
                        <div className="flex items-center justify-between">
                          <CardTitle className="text-lg flex items-center gap-2">
                            <Lock className="w-5 h-5 text-blue-600" />
                            Privacy Policy
                          </CardTitle>
                          <Badge className="bg-green-100 text-green-800">
                            {legalPages.privacy.status}
                          </Badge>
                        </div>
                      </CardHeader>
                      <CardContent className="space-y-4">
                        <div>
                          <p className="text-sm text-muted-foreground">
                            Last updated: {legalPages.privacy.lastUpdated}
                          </p>
                        </div>
                        <div className="space-y-2">
                          <Dialog>
                            <DialogTrigger asChild>
                              <Button className="w-full" size="sm">
                                <Edit className="w-4 h-4 mr-2" />
                                Edit Content
                              </Button>
                            </DialogTrigger>
                            <DialogContent className="max-w-4xl">
                              <DialogHeader>
                                <DialogTitle>Edit Privacy Policy</DialogTitle>
                                <DialogDescription>
                                  Update the privacy policy content that users see
                                </DialogDescription>
                              </DialogHeader>
                              <div className="space-y-4">
                                <div>
                                  <Label htmlFor="privacy-title">Title</Label>
                                  <Input
                                    id="privacy-title"
                                    value={legalPages.privacy.title}
                                    onChange={(e) => setLegalPages(prev => ({
                                      ...prev,
                                      privacy: { ...prev.privacy, title: e.target.value }
                                    }))}
                                  />
                                </div>
                                <div>
                                  <Label htmlFor="privacy-content">Content</Label>
                                  <Textarea
                                    id="privacy-content"
                                    value={legalPages.privacy.content}
                                    onChange={(e) => setLegalPages(prev => ({
                                      ...prev,
                                      privacy: { ...prev.privacy, content: e.target.value }
                                    }))}
                                    rows={12}
                                    className="font-mono text-sm"
                                  />
                                </div>
                                <div className="flex gap-2 pt-4">
                                  <Button 
                                    onClick={() => {
                                      setLegalPages(prev => ({
                                        ...prev,
                                        privacy: { ...prev.privacy, lastUpdated: new Date().toLocaleDateString() }
                                      }));
                                    }}
                                    className="flex items-center gap-2"
                                  >
                                    <Save className="w-4 h-4" />
                                    Save Changes
                                  </Button>
                                  <Button variant="outline" onClick={() => onNavigate("privacy")}>
                                    <Eye className="w-4 h-4 mr-2" />
                                    Preview
                                  </Button>
                                </div>
                              </div>
                            </DialogContent>
                          </Dialog>
                          <Button variant="outline" size="sm" className="w-full" onClick={() => onNavigate("privacy")}>
                            <Eye className="w-4 h-4 mr-2" />
                            View Live Page
                          </Button>
                        </div>
                      </CardContent>
                    </Card>

                    {/* Terms of Service */}
                    <Card className="border-2">
                      <CardHeader className="pb-3">
                        <div className="flex items-center justify-between">
                          <CardTitle className="text-lg flex items-center gap-2">
                            <FileText className="w-5 h-5 text-purple-600" />
                            Terms of Service
                          </CardTitle>
                          <Badge className="bg-green-100 text-green-800">
                            {legalPages.terms.status}
                          </Badge>
                        </div>
                      </CardHeader>
                      <CardContent className="space-y-4">
                        <div>
                          <p className="text-sm text-muted-foreground">
                            Last updated: {legalPages.terms.lastUpdated}
                          </p>
                        </div>
                        <div className="space-y-2">
                          <Dialog>
                            <DialogTrigger asChild>
                              <Button className="w-full" size="sm">
                                <Edit className="w-4 h-4 mr-2" />
                                Edit Content
                              </Button>
                            </DialogTrigger>
                            <DialogContent className="max-w-4xl">
                              <DialogHeader>
                                <DialogTitle>Edit Terms of Service</DialogTitle>
                                <DialogDescription>
                                  Update the terms of service content that users see
                                </DialogDescription>
                              </DialogHeader>
                              <div className="space-y-4">
                                <div>
                                  <Label htmlFor="terms-title">Title</Label>
                                  <Input
                                    id="terms-title"
                                    value={legalPages.terms.title}
                                    onChange={(e) => setLegalPages(prev => ({
                                      ...prev,
                                      terms: { ...prev.terms, title: e.target.value }
                                    }))}
                                  />
                                </div>
                                <div>
                                  <Label htmlFor="terms-content">Content</Label>
                                  <Textarea
                                    id="terms-content"
                                    value={legalPages.terms.content}
                                    onChange={(e) => setLegalPages(prev => ({
                                      ...prev,
                                      terms: { ...prev.terms, content: e.target.value }
                                    }))}
                                    rows={12}
                                    className="font-mono text-sm"
                                  />
                                </div>
                                <div className="flex gap-2 pt-4">
                                  <Button 
                                    onClick={() => {
                                      setLegalPages(prev => ({
                                        ...prev,
                                        terms: { ...prev.terms, lastUpdated: new Date().toLocaleDateString() }
                                      }));
                                    }}
                                    className="flex items-center gap-2"
                                  >
                                    <Save className="w-4 h-4" />
                                    Save Changes
                                  </Button>
                                  <Button variant="outline" onClick={() => onNavigate("terms")}>
                                    <Eye className="w-4 h-4 mr-2" />
                                    Preview
                                  </Button>
                                </div>
                              </div>
                            </DialogContent>
                          </Dialog>
                          <Button variant="outline" size="sm" className="w-full" onClick={() => onNavigate("terms")}>
                            <Eye className="w-4 h-4 mr-2" />
                            View Live Page
                          </Button>
                        </div>
                      </CardContent>
                    </Card>

                    {/* Cookie Policy */}
                    <Card className="border-2">
                      <CardHeader className="pb-3">
                        <div className="flex items-center justify-between">
                          <CardTitle className="text-lg flex items-center gap-2">
                            <Cookie className="w-5 h-5 text-orange-600" />
                            Cookie Policy
                          </CardTitle>
                          <Badge className="bg-green-100 text-green-800">
                            {legalPages.cookies.status}
                          </Badge>
                        </div>
                      </CardHeader>
                      <CardContent className="space-y-4">
                        <div>
                          <p className="text-sm text-muted-foreground">
                            Last updated: {legalPages.cookies.lastUpdated}
                          </p>
                        </div>
                        <div className="space-y-2">
                          <Dialog>
                            <DialogTrigger asChild>
                              <Button className="w-full" size="sm">
                                <Edit className="w-4 h-4 mr-2" />
                                Edit Content
                              </Button>
                            </DialogTrigger>
                            <DialogContent className="max-w-4xl">
                              <DialogHeader>
                                <DialogTitle>Edit Cookie Policy</DialogTitle>
                                <DialogDescription>
                                  Update the cookie policy content that users see
                                </DialogDescription>
                              </DialogHeader>
                              <div className="space-y-4">
                                <div>
                                  <Label htmlFor="cookies-title">Title</Label>
                                  <Input
                                    id="cookies-title"
                                    value={legalPages.cookies.title}
                                    onChange={(e) => setLegalPages(prev => ({
                                      ...prev,
                                      cookies: { ...prev.cookies, title: e.target.value }
                                    }))}
                                  />
                                </div>
                                <div>
                                  <Label htmlFor="cookies-content">Content</Label>
                                  <Textarea
                                    id="cookies-content"
                                    value={legalPages.cookies.content}
                                    onChange={(e) => setLegalPages(prev => ({
                                      ...prev,
                                      cookies: { ...prev.cookies, content: e.target.value }
                                    }))}
                                    rows={12}
                                    className="font-mono text-sm"
                                  />
                                </div>
                                <div className="flex gap-2 pt-4">
                                  <Button 
                                    onClick={() => {
                                      setLegalPages(prev => ({
                                        ...prev,
                                        cookies: { ...prev.cookies, lastUpdated: new Date().toLocaleDateString() }
                                      }));
                                    }}
                                    className="flex items-center gap-2"
                                  >
                                    <Save className="w-4 h-4" />
                                    Save Changes
                                  </Button>
                                  <Button variant="outline" onClick={() => onNavigate("cookies")}>
                                    <Eye className="w-4 h-4 mr-2" />
                                    Preview
                                  </Button>
                                </div>
                              </div>
                            </DialogContent>
                          </Dialog>
                          <Button variant="outline" size="sm" className="w-full" onClick={() => onNavigate("cookies")}>
                            <Eye className="w-4 h-4 mr-2" />
                            View Live Page
                          </Button>
                        </div>
                      </CardContent>
                    </Card>
                  </div>
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          {/* Support Tab */}
          <TabsContent value="support">
            <AdminSupportTab />
          </TabsContent>

          {/* Social Media Tab */}
          <TabsContent value="social">
            <div className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Link className="w-5 h-5" />
                    Social Media Links Management
                  </CardTitle>
                  <CardDescription>
                    Manage social media account links displayed in the footer and across the platform
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="grid grid-cols-1 lg:grid-cols-2 xl:grid-cols-3 gap-6">
                    {/* HotZonne Main Site */}
                    <Card className="border-2">
                      <CardHeader className="pb-3">
                        <div className="flex items-center justify-between">
                          <CardTitle className="text-lg flex items-center gap-2">
                            <Heart className="w-5 h-5 text-purple-600" />
                            HotZonne
                          </CardTitle>
                          <Badge className="bg-green-100 text-green-800">
                            Active
                          </Badge>
                        </div>
                      </CardHeader>
                      <CardContent className="space-y-4">
                        <div>
                          <Label htmlFor="hotzonne-url">Website URL</Label>
                          <Input
                            id="hotzonne-url"
                            value={socialMediaLinks.hotzonne}
                            onChange={(e) => setSocialMediaLinks(prev => ({
                              ...prev,
                              hotzonne: e.target.value
                            }))}
                            placeholder="https://www.hotzonne.com"
                          />
                        </div>
                        <div className="flex gap-2">
                          <Button size="sm" className="flex-1">
                            <Save className="w-4 h-4 mr-2" />
                            Save
                          </Button>
                          <Button size="sm" variant="outline" asChild>
                            <a href={socialMediaLinks.hotzonne} target="_blank" rel="noopener noreferrer">
                              <ExternalLink className="w-4 h-4" />
                            </a>
                          </Button>
                        </div>
                      </CardContent>
                    </Card>

                    {/* Facebook */}
                    <Card className="border-2">
                      <CardHeader className="pb-3">
                        <div className="flex items-center justify-between">
                          <CardTitle className="text-lg flex items-center gap-2">
                            <Facebook className="w-5 h-5 text-blue-600" />
                            Facebook
                          </CardTitle>
                          <Badge className="bg-green-100 text-green-800">
                            Active
                          </Badge>
                        </div>
                      </CardHeader>
                      <CardContent className="space-y-4">
                        <div>
                          <Label htmlFor="facebook-url">Facebook Page URL</Label>
                          <Input
                            id="facebook-url"
                            value={socialMediaLinks.facebook}
                            onChange={(e) => setSocialMediaLinks(prev => ({
                              ...prev,
                              facebook: e.target.value
                            }))}
                            placeholder="https://www.facebook.com/hotzonne"
                          />
                        </div>
                        <div className="flex gap-2">
                          <Button size="sm" className="flex-1">
                            <Save className="w-4 h-4 mr-2" />
                            Save
                          </Button>
                          <Button size="sm" variant="outline" asChild>
                            <a href={socialMediaLinks.facebook} target="_blank" rel="noopener noreferrer">
                              <ExternalLink className="w-4 h-4" />
                            </a>
                          </Button>
                        </div>
                      </CardContent>
                    </Card>

                    {/* X (Twitter) */}
                    <Card className="border-2">
                      <CardHeader className="pb-3">
                        <div className="flex items-center justify-between">
                          <CardTitle className="text-lg flex items-center gap-2">
                            <Twitter className="w-5 h-5 text-black" />
                            X (Twitter)
                          </CardTitle>
                          <Badge className="bg-green-100 text-green-800">
                            Active
                          </Badge>
                        </div>
                      </CardHeader>
                      <CardContent className="space-y-4">
                        <div>
                          <Label htmlFor="twitter-url">X Profile URL</Label>
                          <Input
                            id="twitter-url"
                            value={socialMediaLinks.twitter}
                            onChange={(e) => setSocialMediaLinks(prev => ({
                              ...prev,
                              twitter: e.target.value
                            }))}
                            placeholder="https://www.x.com/hotzonne"
                          />
                        </div>
                        <div className="flex gap-2">
                          <Button size="sm" className="flex-1">
                            <Save className="w-4 h-4 mr-2" />
                            Save
                          </Button>
                          <Button size="sm" variant="outline" asChild>
                            <a href={socialMediaLinks.twitter} target="_blank" rel="noopener noreferrer">
                              <ExternalLink className="w-4 h-4" />
                            </a>
                          </Button>
                        </div>
                      </CardContent>
                    </Card>

                    {/* Instagram */}
                    <Card className="border-2">
                      <CardHeader className="pb-3">
                        <div className="flex items-center justify-between">
                          <CardTitle className="text-lg flex items-center gap-2">
                            <Instagram className="w-5 h-5 text-pink-600" />
                            Instagram
                          </CardTitle>
                          <Badge className="bg-green-100 text-green-800">
                            Active
                          </Badge>
                        </div>
                      </CardHeader>
                      <CardContent className="space-y-4">
                        <div>
                          <Label htmlFor="instagram-url">Instagram Profile URL</Label>
                          <Input
                            id="instagram-url"
                            value={socialMediaLinks.instagram}
                            onChange={(e) => setSocialMediaLinks(prev => ({
                              ...prev,
                              instagram: e.target.value
                            }))}
                            placeholder="https://www.instagram.com/hotzonne"
                          />
                        </div>
                        <div className="flex gap-2">
                          <Button size="sm" className="flex-1">
                            <Save className="w-4 h-4 mr-2" />
                            Save
                          </Button>
                          <Button size="sm" variant="outline" asChild>
                            <a href={socialMediaLinks.instagram} target="_blank" rel="noopener noreferrer">
                              <ExternalLink className="w-4 h-4" />
                            </a>
                          </Button>
                        </div>
                      </CardContent>
                    </Card>

                    {/* YouTube */}
                    <Card className="border-2">
                      <CardHeader className="pb-3">
                        <div className="flex items-center justify-between">
                          <CardTitle className="text-lg flex items-center gap-2">
                            <Youtube className="w-5 h-5 text-red-600" />
                            YouTube
                          </CardTitle>
                          <Badge className="bg-green-100 text-green-800">
                            Active
                          </Badge>
                        </div>
                      </CardHeader>
                      <CardContent className="space-y-4">
                        <div>
                          <Label htmlFor="youtube-url">YouTube Channel URL</Label>
                          <Input
                            id="youtube-url"
                            value={socialMediaLinks.youtube}
                            onChange={(e) => setSocialMediaLinks(prev => ({
                              ...prev,
                              youtube: e.target.value
                            }))}
                            placeholder="https://www.youtube.com/hotzonne"
                          />
                        </div>
                        <div className="flex gap-2">
                          <Button size="sm" className="flex-1">
                            <Save className="w-4 h-4 mr-2" />
                            Save
                          </Button>
                          <Button size="sm" variant="outline" asChild>
                            <a href={socialMediaLinks.youtube} target="_blank" rel="noopener noreferrer">
                              <ExternalLink className="w-4 h-4" />
                            </a>
                          </Button>
                        </div>
                      </CardContent>
                    </Card>

                    {/* TikTok */}
                    <Card className="border-2">
                      <CardHeader className="pb-3">
                        <div className="flex items-center justify-between">
                          <CardTitle className="text-lg flex items-center gap-2">
                            <svg className="w-5 h-5 fill-current" viewBox="0 0 24 24">
                              <path d="M19.59 6.69a4.83 4.83 0 01-3.77-4.25V2h-3.45v13.67a2.89 2.89 0 01-5.2 1.74 2.89 2.89 0 012.31-4.64 2.93 2.93 0 01.88.13V9.4a6.84 6.84 0 00-.88-.05A6.33 6.33 0 005 20.1a6.34 6.34 0 0010.86-4.43v-7a8.16 8.16 0 004.77 1.52v-3.4a4.85 4.85 0 01-1-.1z"/>
                            </svg>
                            TikTok
                          </CardTitle>
                          <Badge className="bg-green-100 text-green-800">
                            Active
                          </Badge>
                        </div>
                      </CardHeader>
                      <CardContent className="space-y-4">
                        <div>
                          <Label htmlFor="tiktok-url">TikTok Profile URL</Label>
                          <Input
                            id="tiktok-url"
                            value={socialMediaLinks.tiktok}
                            onChange={(e) => setSocialMediaLinks(prev => ({
                              ...prev,
                              tiktok: e.target.value
                            }))}
                            placeholder="https://www.tiktok.com/@hotzonne"
                          />
                        </div>
                        <div className="flex gap-2">
                          <Button size="sm" className="flex-1">
                            <Save className="w-4 h-4 mr-2" />
                            Save
                          </Button>
                          <Button size="sm" variant="outline" asChild>
                            <a href={socialMediaLinks.tiktok} target="_blank" rel="noopener noreferrer">
                              <ExternalLink className="w-4 h-4" />
                            </a>
                          </Button>
                        </div>
                      </CardContent>
                    </Card>
                  </div>

                  <Separator className="my-6" />

                  <div className="space-y-4">
                    <h3 className="text-lg font-semibold">API Integration</h3>
                    <p className="text-sm text-muted-foreground">
                      Configure API endpoints to automatically sync social media data and analytics.
                    </p>
                    
                    <div className="grid grid-cols-1 lg:grid-cols-2 gap-4">
                      <Card>
                        <CardHeader className="pb-3">
                          <CardTitle className="text-base">Social Media API</CardTitle>
                        </CardHeader>
                        <CardContent className="space-y-4">
                          <div>
                            <Label htmlFor="api-endpoint">API Endpoint</Label>
                            <Input
                              id="api-endpoint"
                              placeholder="https://api.hotzonne.com/social-media"
                              value=""
                            />
                          </div>
                          <div>
                            <Label htmlFor="api-key">API Key</Label>
                            <Input
                              id="api-key"
                              type="password"
                              placeholder="Enter API key..."
                              value=""
                            />
                          </div>
                          <Button className="w-full">
                            <Settings className="w-4 h-4 mr-2" />
                            Configure API
                          </Button>
                        </CardContent>
                      </Card>

                      <Card>
                        <CardHeader className="pb-3">
                          <CardTitle className="text-base">Analytics Integration</CardTitle>
                        </CardHeader>
                        <CardContent className="space-y-4">
                          <div>
                            <Label htmlFor="analytics-endpoint">Analytics Endpoint</Label>
                            <Input
                              id="analytics-endpoint"
                              placeholder="https://api.hotzonne.com/analytics"
                              value=""
                            />
                          </div>
                          <div>
                            <Label htmlFor="analytics-key">Analytics Key</Label>
                            <Input
                              id="analytics-key"
                              type="password"
                              placeholder="Enter analytics key..."
                              value=""
                            />
                          </div>
                          <Button className="w-full" variant="outline">
                            <BarChart3 className="w-4 h-4 mr-2" />
                            Setup Analytics
                          </Button>
                        </CardContent>
                      </Card>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          {/* Groups Tab */}
          <TabsContent value="groups">
            <AdminGroupsManagement />
          </TabsContent>

          {/* Analytics Tab */}
          <TabsContent value="analytics">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <BarChart3 className="w-5 h-5" />
                  Platform Analytics
                </CardTitle>
              </CardHeader>
              <CardContent>
                <p className="text-muted-foreground">
                  Analytics dashboard coming soon...
                </p>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Settings Tab */}
          <TabsContent value="settings">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Settings className="w-5 h-5" />
                  Platform Settings
                </CardTitle>
              </CardHeader>
              <CardContent>
                <p className="text-muted-foreground">
                  Platform settings coming soon...
                </p>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>

      <Footer onNavigate={onNavigate} onOpenCookieConsent={onOpenCookieConsent} />
    </div>
  );
}