import { useState } from "react";
import { Card } from "./ui/card";
import { Button } from "./ui/button";
import { Input } from "./ui/input";
import { Textarea } from "./ui/textarea";
import { Badge } from "./ui/badge";
import { Avatar, AvatarImage, AvatarFallback } from "./ui/avatar";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "./ui/tabs";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
  DialogFooter,
} from "./ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "./ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "./ui/table";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "./ui/alert-dialog";
import {
  Gamepad2,
  Plus,
  Edit,
  Trash2,
  Eye,
  Flag,
  TrendingUp,
  Users,
  Clock,
  Trophy,
  Search,
  Shield,
  CheckCircle,
  XCircle,
  AlertTriangle,
} from "lucide-react";
import { toast } from "sonner@2.0.3";

interface Question {
  id: string;
  category: string;
  points: number;
  question: string;
  answer: string;
  difficulty: "easy" | "medium" | "hard";
  status: "active" | "inactive";
  createdAt: Date;
  usageCount: number;
}

interface GameReport {
  id: string;
  reportedBy: string;
  reportedUser: string;
  gameId: string;
  reason: string;
  status: "pending" | "reviewed" | "resolved" | "dismissed";
  createdAt: Date;
  reviewedBy?: string;
  reviewedAt?: Date;
}

interface GameActivity {
  id: string;
  players: string[];
  startTime: Date;
  endTime?: Date;
  winner?: string;
  totalQuestions: number;
  scores: { [playerId: string]: number };
  status: "active" | "completed" | "abandoned";
}

export function AdminGamesManagement() {
  const [activeTab, setActiveTab] = useState("questions");
  const [isAddQuestionOpen, setIsAddQuestionOpen] = useState(false);
  const [isEditQuestionOpen, setIsEditQuestionOpen] = useState(false);
  const [isDeleteDialogOpen, setIsDeleteDialogOpen] = useState(false);
  const [selectedQuestion, setSelectedQuestion] = useState<Question | null>(null);
  const [questionToDelete, setQuestionToDelete] = useState<string | null>(null);
  const [searchQuery, setSearchQuery] = useState("");
  const [selectedCategory, setSelectedCategory] = useState("all");
  const [selectedReportStatus, setSelectedReportStatus] = useState("all");

  // Form state
  const [formData, setFormData] = useState({
    category: "",
    points: 100,
    question: "",
    answer: "",
    difficulty: "easy" as "easy" | "medium" | "hard",
  });

  const [questions, setQuestions] = useState<Question[]>([
    {
      id: "1",
      category: "First Date Tips",
      points: 100,
      question: "What percentage of first dates lead to a second date?",
      answer: "53%",
      difficulty: "easy",
      status: "active",
      createdAt: new Date("2024-01-15"),
      usageCount: 145,
    },
    {
      id: "2",
      category: "Relationship Facts",
      points: 200,
      question: "How many dates before most couples become exclusive?",
      answer: "5-6 dates",
      difficulty: "easy",
      status: "active",
      createdAt: new Date("2024-01-16"),
      usageCount: 132,
    },
    {
      id: "3",
      category: "Love Languages",
      points: 300,
      question: "What is the most common love language?",
      answer: "Words of Affirmation",
      difficulty: "medium",
      status: "active",
      createdAt: new Date("2024-01-17"),
      usageCount: 98,
    },
    {
      id: "4",
      category: "Dating Etiquette",
      points: 400,
      question: "How many days should you wait before a second date?",
      answer: "3-5 days",
      difficulty: "medium",
      status: "active",
      createdAt: new Date("2024-01-18"),
      usageCount: 76,
    },
    {
      id: "5",
      category: "Romance History",
      points: 500,
      question: "Who invented speed dating?",
      answer: "Rabbi Yaacov Deyo",
      difficulty: "hard",
      status: "active",
      createdAt: new Date("2024-01-19"),
      usageCount: 54,
    },
  ]);

  const [reports, setReports] = useState<GameReport[]>([
    {
      id: "1",
      reportedBy: "Sarah K.",
      reportedUser: "John D.",
      gameId: "game-123",
      reason: "Used offensive language during game chat",
      status: "pending",
      createdAt: new Date("2024-10-26"),
    },
    {
      id: "2",
      reportedBy: "Mike R.",
      reportedUser: "Alex P.",
      gameId: "game-124",
      reason: "Harassment and inappropriate behavior",
      status: "reviewed",
      createdAt: new Date("2024-10-25"),
      reviewedBy: "Admin",
      reviewedAt: new Date("2024-10-26"),
    },
    {
      id: "3",
      reportedBy: "Emma W.",
      reportedUser: "Tom S.",
      gameId: "game-125",
      reason: "Spam messages in chat",
      status: "resolved",
      createdAt: new Date("2024-10-24"),
      reviewedBy: "Admin",
      reviewedAt: new Date("2024-10-25"),
    },
  ]);

  const [gameActivities, setGameActivities] = useState<GameActivity[]>([
    {
      id: "game-001",
      players: ["Sarah K.", "Mike R."],
      startTime: new Date("2024-10-27T10:30:00"),
      endTime: new Date("2024-10-27T10:45:00"),
      winner: "Sarah K.",
      totalQuestions: 10,
      scores: { "sarah": 800, "mike": 600 },
      status: "completed",
    },
    {
      id: "game-002",
      players: ["Emma W.", "Alex P."],
      startTime: new Date("2024-10-27T11:00:00"),
      endTime: new Date("2024-10-27T11:20:00"),
      winner: "Emma W.",
      totalQuestions: 12,
      scores: { "emma": 1000, "alex": 700 },
      status: "completed",
    },
    {
      id: "game-003",
      players: ["Jessica T.", "Tom S."],
      startTime: new Date("2024-10-27T11:30:00"),
      totalQuestions: 5,
      scores: { "jessica": 400, "tom": 300 },
      status: "active",
    },
  ]);

  const categories = [
    "First Date Tips",
    "Relationship Facts",
    "Love Languages",
    "Dating Etiquette",
    "Romance History",
  ];

  const handleAddQuestion = () => {
    if (!formData.category || !formData.question || !formData.answer) {
      toast.error("Please fill in all required fields");
      return;
    }

    const newQuestion: Question = {
      id: Date.now().toString(),
      category: formData.category,
      points: formData.points,
      question: formData.question,
      answer: formData.answer,
      difficulty: formData.difficulty,
      status: "active",
      createdAt: new Date(),
      usageCount: 0,
    };

    setQuestions([...questions, newQuestion]);
    toast.success("Question added successfully!");
    setIsAddQuestionOpen(false);
    resetForm();
  };

  const handleEditQuestion = () => {
    if (!selectedQuestion) return;

    const updatedQuestions = questions.map((q) =>
      q.id === selectedQuestion.id
        ? {
            ...q,
            category: formData.category,
            points: formData.points,
            question: formData.question,
            answer: formData.answer,
            difficulty: formData.difficulty,
          }
        : q
    );

    setQuestions(updatedQuestions);
    toast.success("Question updated successfully!");
    setIsEditQuestionOpen(false);
    setSelectedQuestion(null);
    resetForm();
  };

  const handleDeleteQuestion = () => {
    if (!questionToDelete) return;

    setQuestions(questions.filter((q) => q.id !== questionToDelete));
    toast.success("Question deleted successfully!");
    setIsDeleteDialogOpen(false);
    setQuestionToDelete(null);
  };

  const handleToggleQuestionStatus = (id: string) => {
    const updatedQuestions = questions.map((q) =>
      q.id === id
        ? { ...q, status: q.status === "active" ? "inactive" : "active" }
        : q
    );
    setQuestions(updatedQuestions as Question[]);
    toast.success("Question status updated!");
  };

  const openEditDialog = (question: Question) => {
    setSelectedQuestion(question);
    setFormData({
      category: question.category,
      points: question.points,
      question: question.question,
      answer: question.answer,
      difficulty: question.difficulty,
    });
    setIsEditQuestionOpen(true);
  };

  const openDeleteDialog = (id: string) => {
    setQuestionToDelete(id);
    setIsDeleteDialogOpen(true);
  };

  const resetForm = () => {
    setFormData({
      category: "",
      points: 100,
      question: "",
      answer: "",
      difficulty: "easy",
    });
  };

  const handleResolveReport = (id: string, action: "resolved" | "dismissed") => {
    const updatedReports = reports.map((r) =>
      r.id === id
        ? {
            ...r,
            status: action,
            reviewedBy: "Current Admin",
            reviewedAt: new Date(),
          }
        : r
    );
    setReports(updatedReports as GameReport[]);
    toast.success(`Report ${action} successfully!`);
  };

  const filteredQuestions = questions.filter((q) => {
    const matchesSearch =
      q.question.toLowerCase().includes(searchQuery.toLowerCase()) ||
      q.answer.toLowerCase().includes(searchQuery.toLowerCase());
    const matchesCategory =
      selectedCategory === "all" || q.category === selectedCategory;
    return matchesSearch && matchesCategory;
  });

  const filteredReports = reports.filter((r) => {
    return selectedReportStatus === "all" || r.status === selectedReportStatus;
  });

  // Statistics
  const stats = {
    totalQuestions: questions.length,
    activeQuestions: questions.filter((q) => q.status === "active").length,
    totalGames: gameActivities.length,
    activeGames: gameActivities.filter((g) => g.status === "active").length,
    pendingReports: reports.filter((r) => r.status === "pending").length,
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div>
        <div className="flex items-center gap-3 mb-2">
          <div className="bg-gradient-to-br from-purple-500 to-pink-500 p-3 rounded-xl">
            <Gamepad2 className="w-6 h-6 text-white" />
          </div>
          <div>
            <h2 className="text-2xl font-bold">Dating Games Management</h2>
            <p className="text-sm text-muted-foreground">
              Manage questions, monitor games, and review reports
            </p>
          </div>
        </div>
      </div>

      {/* Statistics Cards */}
      <div className="grid grid-cols-1 md:grid-cols-5 gap-4">
        <Card className="p-4">
          <div className="flex items-center gap-3">
            <div className="bg-blue-100 dark:bg-blue-900/30 p-2 rounded-lg">
              <Trophy className="w-5 h-5 text-blue-600" />
            </div>
            <div>
              <p className="text-2xl font-bold">{stats.totalQuestions}</p>
              <p className="text-xs text-muted-foreground">Total Questions</p>
            </div>
          </div>
        </Card>

        <Card className="p-4">
          <div className="flex items-center gap-3">
            <div className="bg-green-100 dark:bg-green-900/30 p-2 rounded-lg">
              <CheckCircle className="w-5 h-5 text-green-600" />
            </div>
            <div>
              <p className="text-2xl font-bold">{stats.activeQuestions}</p>
              <p className="text-xs text-muted-foreground">Active Questions</p>
            </div>
          </div>
        </Card>

        <Card className="p-4">
          <div className="flex items-center gap-3">
            <div className="bg-purple-100 dark:bg-purple-900/30 p-2 rounded-lg">
              <Gamepad2 className="w-5 h-5 text-purple-600" />
            </div>
            <div>
              <p className="text-2xl font-bold">{stats.totalGames}</p>
              <p className="text-xs text-muted-foreground">Total Games</p>
            </div>
          </div>
        </Card>

        <Card className="p-4">
          <div className="flex items-center gap-3">
            <div className="bg-orange-100 dark:bg-orange-900/30 p-2 rounded-lg">
              <Clock className="w-5 h-5 text-orange-600" />
            </div>
            <div>
              <p className="text-2xl font-bold">{stats.activeGames}</p>
              <p className="text-xs text-muted-foreground">Active Games</p>
            </div>
          </div>
        </Card>

        <Card className="p-4">
          <div className="flex items-center gap-3">
            <div className="bg-red-100 dark:bg-red-900/30 p-2 rounded-lg">
              <Flag className="w-5 h-5 text-red-600" />
            </div>
            <div>
              <p className="text-2xl font-bold">{stats.pendingReports}</p>
              <p className="text-xs text-muted-foreground">Pending Reports</p>
            </div>
          </div>
        </Card>
      </div>

      {/* Main Content */}
      <Tabs value={activeTab} onValueChange={setActiveTab}>
        <TabsList className="grid w-full grid-cols-3">
          <TabsTrigger value="questions">
            <Trophy className="w-4 h-4 mr-2" />
            Questions
          </TabsTrigger>
          <TabsTrigger value="activity">
            <TrendingUp className="w-4 h-4 mr-2" />
            Game Activity
          </TabsTrigger>
          <TabsTrigger value="reports">
            <Flag className="w-4 h-4 mr-2" />
            Reports
            {stats.pendingReports > 0 && (
              <Badge className="ml-2 bg-red-500">{stats.pendingReports}</Badge>
            )}
          </TabsTrigger>
        </TabsList>

        {/* Questions Tab */}
        <TabsContent value="questions" className="space-y-4">
          <Card className="p-4">
            <div className="flex flex-col md:flex-row gap-4 mb-4">
              <div className="flex-1">
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-muted-foreground" />
                  <Input
                    placeholder="Search questions..."
                    value={searchQuery}
                    onChange={(e) => setSearchQuery(e.target.value)}
                    className="pl-10"
                  />
                </div>
              </div>
              <Select value={selectedCategory} onValueChange={setSelectedCategory}>
                <SelectTrigger className="w-full md:w-[200px]">
                  <SelectValue placeholder="Category" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Categories</SelectItem>
                  {categories.map((cat) => (
                    <SelectItem key={cat} value={cat}>
                      {cat}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <Button
                onClick={() => setIsAddQuestionOpen(true)}
                className="bg-gradient-to-r from-purple-600 to-pink-600"
              >
                <Plus className="w-4 h-4 mr-2" />
                Add Question
              </Button>
            </div>

            <div className="border rounded-lg overflow-hidden">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Category</TableHead>
                    <TableHead>Question</TableHead>
                    <TableHead>Answer</TableHead>
                    <TableHead>Points</TableHead>
                    <TableHead>Difficulty</TableHead>
                    <TableHead>Usage</TableHead>
                    <TableHead>Status</TableHead>
                    <TableHead>Actions</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {filteredQuestions.map((question) => (
                    <TableRow key={question.id}>
                      <TableCell className="font-medium">{question.category}</TableCell>
                      <TableCell className="max-w-xs truncate">
                        {question.question}
                      </TableCell>
                      <TableCell>{question.answer}</TableCell>
                      <TableCell>
                        <Badge variant="outline">${question.points}</Badge>
                      </TableCell>
                      <TableCell>
                        <Badge
                          className={
                            question.difficulty === "easy"
                              ? "bg-green-100 text-green-800"
                              : question.difficulty === "medium"
                              ? "bg-yellow-100 text-yellow-800"
                              : "bg-red-100 text-red-800"
                          }
                        >
                          {question.difficulty}
                        </Badge>
                      </TableCell>
                      <TableCell>{question.usageCount}</TableCell>
                      <TableCell>
                        <Button
                          size="sm"
                          variant="ghost"
                          onClick={() => handleToggleQuestionStatus(question.id)}
                        >
                          {question.status === "active" ? (
                            <CheckCircle className="w-4 h-4 text-green-600" />
                          ) : (
                            <XCircle className="w-4 h-4 text-gray-400" />
                          )}
                        </Button>
                      </TableCell>
                      <TableCell>
                        <div className="flex gap-2">
                          <Button
                            size="sm"
                            variant="ghost"
                            onClick={() => openEditDialog(question)}
                          >
                            <Edit className="w-4 h-4" />
                          </Button>
                          <Button
                            size="sm"
                            variant="ghost"
                            onClick={() => openDeleteDialog(question.id)}
                          >
                            <Trash2 className="w-4 h-4 text-red-600" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          </Card>
        </TabsContent>

        {/* Game Activity Tab */}
        <TabsContent value="activity" className="space-y-4">
          <Card className="p-4">
            <h3 className="font-semibold mb-4 flex items-center gap-2">
              <Clock className="w-5 h-5 text-purple-600" />
              Recent Game Activity
            </h3>

            <div className="space-y-3">
              {gameActivities.map((game) => (
                <div
                  key={game.id}
                  className="p-4 border rounded-lg hover:bg-accent transition-colors"
                >
                  <div className="flex items-start justify-between mb-2">
                    <div>
                      <div className="flex items-center gap-2 mb-1">
                        <Users className="w-4 h-4 text-muted-foreground" />
                        <span className="font-medium">{game.players.join(" vs ")}</span>
                      </div>
                      <div className="text-sm text-muted-foreground">
                        Game ID: {game.id}
                      </div>
                    </div>
                    <Badge
                      className={
                        game.status === "active"
                          ? "bg-green-100 text-green-800"
                          : game.status === "completed"
                          ? "bg-blue-100 text-blue-800"
                          : "bg-gray-100 text-gray-800"
                      }
                    >
                      {game.status}
                    </Badge>
                  </div>

                  <div className="grid grid-cols-3 gap-4 text-sm">
                    <div>
                      <span className="text-muted-foreground">Started:</span>
                      <div className="font-medium">
                        {game.startTime.toLocaleTimeString()}
                      </div>
                    </div>
                    {game.endTime && (
                      <div>
                        <span className="text-muted-foreground">Ended:</span>
                        <div className="font-medium">
                          {game.endTime.toLocaleTimeString()}
                        </div>
                      </div>
                    )}
                    <div>
                      <span className="text-muted-foreground">Questions:</span>
                      <div className="font-medium">{game.totalQuestions}</div>
                    </div>
                  </div>

                  {game.winner && (
                    <div className="mt-2 pt-2 border-t">
                      <div className="flex items-center gap-2 text-sm">
                        <Trophy className="w-4 h-4 text-yellow-600" />
                        <span className="text-muted-foreground">Winner:</span>
                        <span className="font-semibold">{game.winner}</span>
                      </div>
                    </div>
                  )}

                  {game.status === "active" && (
                    <Button size="sm" variant="outline" className="mt-2 w-full">
                      <Eye className="w-4 h-4 mr-2" />
                      Monitor Game
                    </Button>
                  )}
                </div>
              ))}
            </div>
          </Card>
        </TabsContent>

        {/* Reports Tab */}
        <TabsContent value="reports" className="space-y-4">
          <Card className="p-4">
            <div className="flex items-center justify-between mb-4">
              <h3 className="font-semibold flex items-center gap-2">
                <Shield className="w-5 h-5 text-red-600" />
                User Reports
              </h3>
              <Select
                value={selectedReportStatus}
                onValueChange={setSelectedReportStatus}
              >
                <SelectTrigger className="w-[180px]">
                  <SelectValue placeholder="Status" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Reports</SelectItem>
                  <SelectItem value="pending">Pending</SelectItem>
                  <SelectItem value="reviewed">Reviewed</SelectItem>
                  <SelectItem value="resolved">Resolved</SelectItem>
                  <SelectItem value="dismissed">Dismissed</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-3">
              {filteredReports.map((report) => (
                <div
                  key={report.id}
                  className="p-4 border rounded-lg hover:bg-accent transition-colors"
                >
                  <div className="flex items-start justify-between mb-3">
                    <div className="flex-1">
                      <div className="flex items-center gap-2 mb-2">
                        <AlertTriangle className="w-4 h-4 text-red-600" />
                        <span className="font-semibold">{report.reportedUser}</span>
                        <span className="text-sm text-muted-foreground">
                          reported by {report.reportedBy}
                        </span>
                      </div>
                      <p className="text-sm mb-2">{report.reason}</p>
                      <div className="text-xs text-muted-foreground">
                        Game ID: {report.gameId} • {report.createdAt.toLocaleDateString()}
                      </div>
                    </div>
                    <Badge
                      className={
                        report.status === "pending"
                          ? "bg-yellow-100 text-yellow-800"
                          : report.status === "reviewed"
                          ? "bg-blue-100 text-blue-800"
                          : report.status === "resolved"
                          ? "bg-green-100 text-green-800"
                          : "bg-gray-100 text-gray-800"
                      }
                    >
                      {report.status}
                    </Badge>
                  </div>

                  {report.status === "pending" && (
                    <div className="flex gap-2">
                      <Button
                        size="sm"
                        onClick={() => handleResolveReport(report.id, "resolved")}
                        className="flex-1 bg-green-600 hover:bg-green-700"
                      >
                        <CheckCircle className="w-4 h-4 mr-1" />
                        Resolve
                      </Button>
                      <Button
                        size="sm"
                        variant="outline"
                        onClick={() => handleResolveReport(report.id, "dismissed")}
                        className="flex-1"
                      >
                        <XCircle className="w-4 h-4 mr-1" />
                        Dismiss
                      </Button>
                    </div>
                  )}

                  {report.reviewedBy && (
                    <div className="mt-2 pt-2 border-t text-xs text-muted-foreground">
                      Reviewed by {report.reviewedBy} on{" "}
                      {report.reviewedAt?.toLocaleDateString()}
                    </div>
                  )}
                </div>
              ))}
            </div>
          </Card>
        </TabsContent>
      </Tabs>

      {/* Add Question Dialog */}
      <Dialog open={isAddQuestionOpen} onOpenChange={setIsAddQuestionOpen}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>Add New Question</DialogTitle>
            <DialogDescription>
              Create a new question for the Dating Jeopardy game
            </DialogDescription>
          </DialogHeader>

          <div className="space-y-4">
            <div className="grid grid-cols-2 gap-4">
              <div>
                <label className="text-sm font-medium mb-2 block">Category</label>
                <Select
                  value={formData.category}
                  onValueChange={(value) =>
                    setFormData({ ...formData, category: value })
                  }
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Select category" />
                  </SelectTrigger>
                  <SelectContent>
                    {categories.map((cat) => (
                      <SelectItem key={cat} value={cat}>
                        {cat}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div>
                <label className="text-sm font-medium mb-2 block">Points</label>
                <Select
                  value={formData.points.toString()}
                  onValueChange={(value) =>
                    setFormData({ ...formData, points: parseInt(value) })
                  }
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="100">100</SelectItem>
                    <SelectItem value="200">200</SelectItem>
                    <SelectItem value="300">300</SelectItem>
                    <SelectItem value="400">400</SelectItem>
                    <SelectItem value="500">500</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>

            <div>
              <label className="text-sm font-medium mb-2 block">Difficulty</label>
              <Select
                value={formData.difficulty}
                onValueChange={(value: any) =>
                  setFormData({ ...formData, difficulty: value })
                }
              >
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="easy">Easy</SelectItem>
                  <SelectItem value="medium">Medium</SelectItem>
                  <SelectItem value="hard">Hard</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div>
              <label className="text-sm font-medium mb-2 block">Question</label>
              <Textarea
                value={formData.question}
                onChange={(e) =>
                  setFormData({ ...formData, question: e.target.value })
                }
                placeholder="Enter the question..."
                className="min-h-[100px]"
              />
            </div>

            <div>
              <label className="text-sm font-medium mb-2 block">Answer</label>
              <Input
                value={formData.answer}
                onChange={(e) =>
                  setFormData({ ...formData, answer: e.target.value })
                }
                placeholder="Enter the correct answer..."
              />
            </div>
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={() => setIsAddQuestionOpen(false)}>
              Cancel
            </Button>
            <Button
              onClick={handleAddQuestion}
              className="bg-gradient-to-r from-purple-600 to-pink-600"
            >
              Add Question
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Edit Question Dialog */}
      <Dialog open={isEditQuestionOpen} onOpenChange={setIsEditQuestionOpen}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>Edit Question</DialogTitle>
            <DialogDescription>
              Update the question details
            </DialogDescription>
          </DialogHeader>

          <div className="space-y-4">
            <div className="grid grid-cols-2 gap-4">
              <div>
                <label className="text-sm font-medium mb-2 block">Category</label>
                <Select
                  value={formData.category}
                  onValueChange={(value) =>
                    setFormData({ ...formData, category: value })
                  }
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Select category" />
                  </SelectTrigger>
                  <SelectContent>
                    {categories.map((cat) => (
                      <SelectItem key={cat} value={cat}>
                        {cat}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div>
                <label className="text-sm font-medium mb-2 block">Points</label>
                <Select
                  value={formData.points.toString()}
                  onValueChange={(value) =>
                    setFormData({ ...formData, points: parseInt(value) })
                  }
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="100">100</SelectItem>
                    <SelectItem value="200">200</SelectItem>
                    <SelectItem value="300">300</SelectItem>
                    <SelectItem value="400">400</SelectItem>
                    <SelectItem value="500">500</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>

            <div>
              <label className="text-sm font-medium mb-2 block">Difficulty</label>
              <Select
                value={formData.difficulty}
                onValueChange={(value: any) =>
                  setFormData({ ...formData, difficulty: value })
                }
              >
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="easy">Easy</SelectItem>
                  <SelectItem value="medium">Medium</SelectItem>
                  <SelectItem value="hard">Hard</SelectItem>
                  <SelectItem value="hard">Hard</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div>
              <label className="text-sm font-medium mb-2 block">Question</label>
              <Textarea
                value={formData.question}
                onChange={(e) =>
                  setFormData({ ...formData, question: e.target.value })
                }
                placeholder="Enter the question..."
                className="min-h-[100px]"
              />
            </div>

            <div>
              <label className="text-sm font-medium mb-2 block">Answer</label>
              <Input
                value={formData.answer}
                onChange={(e) =>
                  setFormData({ ...formData, answer: e.target.value })
                }
                placeholder="Enter the correct answer..."
              />
            </div>
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={() => setIsEditQuestionOpen(false)}>
              Cancel
            </Button>
            <Button
              onClick={handleEditQuestion}
              className="bg-gradient-to-r from-purple-600 to-pink-600"
            >
              Save Changes
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Delete Confirmation Dialog */}
      <AlertDialog open={isDeleteDialogOpen} onOpenChange={setIsDeleteDialogOpen}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Are you sure?</AlertDialogTitle>
            <AlertDialogDescription>
              This action cannot be undone. This will permanently delete the question
              from the game.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancel</AlertDialogCancel>
            <AlertDialogAction
              onClick={handleDeleteQuestion}
              className="bg-red-600 hover:bg-red-700"
            >
              Delete
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );
}
