import { useState } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "./ui/card";
import { Button } from "./ui/button";
import { Input } from "./ui/input";
import { Label } from "./ui/label";
import { Badge } from "./ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "./ui/tabs";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "./ui/dialog";
import { Alert, AlertDescription } from "./ui/alert";
import { 
  Coins, 
  Video, 
  Image as ImageIcon, 
  FileText, 
  DollarSign,
  Calendar,
  TrendingUp,
  Settings,
  Save,
  Award,
  Crown,
  ShoppingCart,
  Gift
} from "lucide-react";

interface AdPointsConfig {
  video: { [key: string]: number };
  photo: { [key: string]: number };
  text: { [key: string]: number };
}

interface PremiumProduct {
  id: string;
  name: string;
  description: string;
  pointsCost: number;
  type: "boost" | "premium" | "feature";
  duration?: string;
  isActive: boolean;
}

export function AdminPointsManagement() {
  const [activeTab, setActiveTab] = useState("point-rates");

  // Points earned for viewing ads by type and duration
  const [adPointsConfig, setAdPointsConfig] = useState<AdPointsConfig>({
    video: { "7": 15, "14": 12, "30": 10, "60": 8 },
    photo: { "7": 8, "14": 6, "30": 5, "60": 4 },
    text: { "7": 5, "14": 4, "30": 3, "60": 2 }
  });

  // Premium products that can be purchased with points
  const [premiumProducts, setPremiumProducts] = useState<PremiumProduct[]>([
    {
      id: "profile-boost",
      name: "Profile Boost",
      description: "Boost your profile visibility for 24 hours",
      pointsCost: 500,
      type: "boost",
      duration: "24 hours",
      isActive: true
    },
    {
      id: "premium-week",
      name: "Premium Membership - 1 Week",
      description: "Unlock all premium features for 7 days",
      pointsCost: 2000,
      type: "premium",
      duration: "7 days",
      isActive: true
    },
    {
      id: "premium-month",
      name: "Premium Membership - 1 Month",
      description: "Unlock all premium features for 30 days",
      pointsCost: 6000,
      type: "premium",
      duration: "30 days",
      isActive: true
    },
    {
      id: "super-like",
      name: "Super Likes Pack",
      description: "10 Super Likes to stand out",
      pointsCost: 300,
      type: "feature",
      isActive: true
    },
    {
      id: "message-priority",
      name: "Priority Messaging",
      description: "Your messages appear first for 7 days",
      pointsCost: 800,
      type: "feature",
      duration: "7 days",
      isActive: true
    }
  ]);

  const [globalPointsMultiplier, setGlobalPointsMultiplier] = useState(1.0);
  const [bonusPointsActive, setBonusPointsActive] = useState(false);
  const [bonusMultiplier, setBonusMultiplier] = useState(2.0);

  const updateAdPoints = (type: keyof AdPointsConfig, duration: string, points: number) => {
    setAdPointsConfig(prev => ({
      ...prev,
      [type]: {
        ...prev[type],
        [duration]: points
      }
    }));
  };

  const updateProduct = (productId: string, updates: Partial<PremiumProduct>) => {
    setPremiumProducts(prev => 
      prev.map(product => 
        product.id === productId 
          ? { ...product, ...updates }
          : product
      )
    );
  };

  const addNewProduct = () => {
    const newProduct: PremiumProduct = {
      id: `product-${Date.now()}`,
      name: "New Product",
      description: "Description here",
      pointsCost: 100,
      type: "feature",
      isActive: true
    };
    setPremiumProducts(prev => [...prev, newProduct]);
  };

  const getTypeIcon = (type: string) => {
    switch (type) {
      case "video":
        return <Video className="w-4 h-4" />;
      case "photo":
        return <ImageIcon className="w-4 h-4" />;
      case "text":
        return <FileText className="w-4 h-4" />;
      default:
        return <FileText className="w-4 h-4" />;
    }
  };

  const getProductIcon = (type: string) => {
    switch (type) {
      case "boost":
        return <TrendingUp className="w-4 h-4" />;
      case "premium":
        return <Crown className="w-4 h-4" />;
      case "feature":
        return <Award className="w-4 h-4" />;
      default:
        return <Gift className="w-4 h-4" />;
    }
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-3xl font-bold">Points & Rewards Management</h2>
          <p className="text-muted-foreground">Configure point earning rates and premium products</p>
        </div>
        <div className="flex gap-2">
          <Button variant="outline">
            <TrendingUp className="w-4 h-4 mr-2" />
            Analytics
          </Button>
          <Button className="bg-orange-600 hover:bg-orange-700">
            <Save className="w-4 h-4 mr-2" />
            Save All Changes
          </Button>
        </div>
      </div>

      <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-6">
        <TabsList className="grid w-full grid-cols-4">
          <TabsTrigger value="point-rates">Point Rates</TabsTrigger>
          <TabsTrigger value="premium-products">Premium Store</TabsTrigger>
          <TabsTrigger value="global-settings">Global Settings</TabsTrigger>
          <TabsTrigger value="analytics">Analytics</TabsTrigger>
        </TabsList>

        {/* Point Rates Tab */}
        <TabsContent value="point-rates" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Coins className="w-5 h-5 text-orange-600" />
                Ad Viewing Point Rates
              </CardTitle>
              <CardDescription>
                Set how many points users earn for viewing different types of ads
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              {Object.entries(adPointsConfig).map(([type, durations]) => (
                <div key={type}>
                  <h4 className="font-semibold mb-3 flex items-center gap-2">
                    {getTypeIcon(type)}
                    {type.charAt(0).toUpperCase() + type.slice(1)} Ads
                    {type === "video" && (
                      <Badge variant="outline" className="text-xs">
                        Must watch to end
                      </Badge>
                    )}
                  </h4>
                  <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                    {Object.entries(durations).map(([duration, points]) => (
                      <div key={duration} className="space-y-2">
                        <Label className="text-sm">
                          {duration} Day Ad
                        </Label>
                        <div className="flex items-center gap-2">
                          <Input
                            type="number"
                            value={points}
                            onChange={(e) => updateAdPoints(type as keyof AdPointsConfig, duration, parseInt(e.target.value) || 0)}
                            min="0"
                            className="text-center"
                          />
                          <span className="text-sm text-muted-foreground">pts</span>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              ))}

              <Alert>
                <Coins className="w-4 h-4" />
                <AlertDescription>
                  <strong>Point Logic:</strong> Longer duration ads typically offer fewer points as they have more exposure time. 
                  Video ads offer the most points but require full viewing completion.
                </AlertDescription>
              </Alert>
            </CardContent>
          </Card>
        </TabsContent>

        {/* Premium Products Tab */}
        <TabsContent value="premium-products" className="space-y-6">
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <div>
                  <CardTitle className="flex items-center gap-2">
                    <ShoppingCart className="w-5 h-5 text-orange-600" />
                    Premium Store Products
                  </CardTitle>
                  <CardDescription>
                    Manage products users can purchase with points
                  </CardDescription>
                </div>
                <Button onClick={addNewProduct} className="bg-orange-600 hover:bg-orange-700">
                  Add Product
                </Button>
              </div>
            </CardHeader>
            <CardContent>
              <div className="grid gap-4">
                {premiumProducts.map((product) => (
                  <Card key={product.id} className="border-2">
                    <CardContent className="p-4">
                      <div className="flex items-center justify-between">
                        <div className="flex items-center gap-3">
                          <div className="p-2 bg-orange-100 dark:bg-orange-900/20 rounded-lg">
                            {getProductIcon(product.type)}
                          </div>
                          <div>
                            <h4 className="font-semibold">{product.name}</h4>
                            <p className="text-sm text-muted-foreground">{product.description}</p>
                            {product.duration && (
                              <p className="text-xs text-orange-600">Duration: {product.duration}</p>
                            )}
                          </div>
                        </div>
                        <div className="flex items-center gap-4">
                          <div className="text-right">
                            <div className="flex items-center gap-1">
                              <Input
                                type="number"
                                value={product.pointsCost}
                                onChange={(e) => updateProduct(product.id, { pointsCost: parseInt(e.target.value) || 0 })}
                                className="w-20 text-center"
                                min="0"
                              />
                              <Coins className="w-4 h-4 text-orange-600" />
                            </div>
                            <Badge 
                              variant={product.isActive ? "default" : "secondary"}
                              className="mt-1"
                            >
                              {product.isActive ? "Active" : "Inactive"}
                            </Badge>
                          </div>
                          <Button
                            variant="outline"
                            size="sm"
                            onClick={() => updateProduct(product.id, { isActive: !product.isActive })}
                          >
                            {product.isActive ? "Disable" : "Enable"}
                          </Button>
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        {/* Global Settings Tab */}
        <TabsContent value="global-settings" className="space-y-6">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Settings className="w-5 h-5" />
                  Point Multipliers
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <Label htmlFor="global-multiplier">Global Points Multiplier</Label>
                  <div className="flex items-center gap-2 mt-1">
                    <Input
                      id="global-multiplier"
                      type="number"
                      value={globalPointsMultiplier}
                      onChange={(e) => setGlobalPointsMultiplier(parseFloat(e.target.value) || 1.0)}
                      step="0.1"
                      min="0.1"
                      max="5.0"
                    />
                    <span className="text-sm text-muted-foreground">x</span>
                  </div>
                  <p className="text-xs text-muted-foreground mt-1">
                    Multiplies all point earnings globally
                  </p>
                </div>

                <div className="space-y-2">
                  <div className="flex items-center justify-between">
                    <Label htmlFor="bonus-points">Bonus Points Event</Label>
                    <Button
                      variant={bonusPointsActive ? "default" : "outline"}
                      size="sm"
                      onClick={() => setBonusPointsActive(!bonusPointsActive)}
                    >
                      {bonusPointsActive ? "Active" : "Inactive"}
                    </Button>
                  </div>
                  {bonusPointsActive && (
                    <div className="flex items-center gap-2">
                      <Input
                        type="number"
                        value={bonusMultiplier}
                        onChange={(e) => setBonusMultiplier(parseFloat(e.target.value) || 2.0)}
                        step="0.1"
                        min="1.1"
                        max="10.0"
                      />
                      <span className="text-sm text-muted-foreground">x bonus</span>
                    </div>
                  )}
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Award className="w-5 h-5" />
                  Point Conversion Rates
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <Label>Points to USD Conversion</Label>
                  <div className="text-sm text-muted-foreground mt-1">
                    1000 points = $1.00 USD value
                  </div>
                </div>
                <div>
                  <Label>Premium Membership Equivalent</Label>
                  <div className="text-sm text-muted-foreground mt-1">
                    6000 points = 1 month premium ($9.99)
                  </div>
                </div>
                <div>
                  <Label>Average Earning Rate</Label>
                  <div className="text-sm text-muted-foreground mt-1">
                    ~50-100 points per day with regular usage
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        {/* Analytics Tab */}
        <TabsContent value="analytics" className="space-y-6">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
            <Card>
              <CardContent className="p-6">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm text-muted-foreground">Total Points Distributed</p>
                    <p className="text-2xl font-bold">2,847,392</p>
                  </div>
                  <Coins className="w-8 h-8 text-orange-600" />
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardContent className="p-6">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm text-muted-foreground">Points Redeemed</p>
                    <p className="text-2xl font-bold">1,923,485</p>
                  </div>
                  <ShoppingCart className="w-8 h-8 text-green-600" />
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardContent className="p-6">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm text-muted-foreground">Avg Points/User</p>
                    <p className="text-2xl font-bold">1,247</p>
                  </div>
                  <TrendingUp className="w-8 h-8 text-blue-600" />
                </div>
              </CardContent>
            </Card>
          </div>

          <Card>
            <CardHeader>
              <CardTitle>Popular Premium Products</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                {premiumProducts
                  .filter(p => p.isActive)
                  .sort((a, b) => b.pointsCost - a.pointsCost)
                  .slice(0, 5)
                  .map((product) => (
                    <div key={product.id} className="flex items-center justify-between p-3 border rounded-lg">
                      <div className="flex items-center gap-2">
                        {getProductIcon(product.type)}
                        <span className="font-medium">{product.name}</span>
                      </div>
                      <div className="flex items-center gap-2">
                        <span className="text-sm text-muted-foreground">847 purchases</span>
                        <Badge variant="outline">{product.pointsCost} pts</Badge>
                      </div>
                    </div>
                  ))}
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
}