import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "./ui/card";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Input } from "./ui/input";
import { Label } from "./ui/label";
import { Textarea } from "./ui/textarea";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "./ui/dialog";
import { HelpCircle, Shield, Users, Edit, Eye, Save } from "lucide-react";

interface SupportPage {
  title: string;
  lastUpdated: string;
  content: string;
  status: string;
}

interface AdminSupportTabProps {
  supportPages: {
    helpCenter: SupportPage;
    safetyTips: SupportPage;
    communityGuidelines: SupportPage;
  };
  setSupportPages: React.Dispatch<React.SetStateAction<{
    helpCenter: SupportPage;
    safetyTips: SupportPage;
    communityGuidelines: SupportPage;
  }>>;
  onNavigate: (view: string) => void;
}

export function AdminSupportTab({ supportPages, setSupportPages, onNavigate }: AdminSupportTabProps) {
  return (
    <div className="space-y-6">
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <HelpCircle className="w-5 h-5" />
            Support Pages Management
          </CardTitle>
          <CardDescription>
            Manage Help Center, Safety Tips, and Community Guidelines content
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
            {/* Help Center */}
            <Card className="border-2">
              <CardHeader className="pb-3">
                <div className="flex items-center justify-between">
                  <CardTitle className="text-lg flex items-center gap-2">
                    <HelpCircle className="w-5 h-5 text-blue-600" />
                    Help Center
                  </CardTitle>
                  <Badge className="bg-green-100 text-green-800">
                    {supportPages.helpCenter.status}
                  </Badge>
                </div>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <p className="text-sm text-muted-foreground">
                    Last updated: {supportPages.helpCenter.lastUpdated}
                  </p>
                </div>
                <div className="space-y-2">
                  <Dialog>
                    <DialogTrigger asChild>
                      <Button className="w-full" size="sm">
                        <Edit className="w-4 h-4 mr-2" />
                        Edit Content
                      </Button>
                    </DialogTrigger>
                    <DialogContent className="max-w-4xl">
                      <DialogHeader>
                        <DialogTitle>Edit Help Center</DialogTitle>
                        <DialogDescription>
                          Update the help center content that users see
                        </DialogDescription>
                      </DialogHeader>
                      <div className="space-y-4">
                        <div>
                          <Label htmlFor="helpCenter-title">Title</Label>
                          <Input
                            id="helpCenter-title"
                            value={supportPages.helpCenter.title}
                            onChange={(e) => setSupportPages(prev => ({
                              ...prev,
                              helpCenter: { ...prev.helpCenter, title: e.target.value }
                            }))}
                          />
                        </div>
                        <div>
                          <Label htmlFor="helpCenter-content">Content</Label>
                          <Textarea
                            id="helpCenter-content"
                            value={supportPages.helpCenter.content}
                            onChange={(e) => setSupportPages(prev => ({
                              ...prev,
                              helpCenter: { ...prev.helpCenter, content: e.target.value }
                            }))}
                            rows={12}
                            className="font-mono text-sm"
                          />
                        </div>
                        <div className="flex gap-2 pt-4">
                          <Button 
                            onClick={() => {
                              setSupportPages(prev => ({
                                ...prev,
                                helpCenter: { ...prev.helpCenter, lastUpdated: new Date().toLocaleDateString() }
                              }));
                            }}
                            className="flex items-center gap-2"
                          >
                            <Save className="w-4 h-4" />
                            Save Changes
                          </Button>
                          <Button variant="outline" onClick={() => onNavigate("help-center")}>
                            <Eye className="w-4 h-4 mr-2" />
                            Preview
                          </Button>
                        </div>
                      </div>
                    </DialogContent>
                  </Dialog>
                  <Button variant="outline" size="sm" className="w-full" onClick={() => onNavigate("help-center")}>
                    <Eye className="w-4 h-4 mr-2" />
                    View Live Page
                  </Button>
                </div>
              </CardContent>
            </Card>

            {/* Safety Tips */}
            <Card className="border-2">
              <CardHeader className="pb-3">
                <div className="flex items-center justify-between">
                  <CardTitle className="text-lg flex items-center gap-2">
                    <Shield className="w-5 h-5 text-red-600" />
                    Safety Tips
                  </CardTitle>
                  <Badge className="bg-green-100 text-green-800">
                    {supportPages.safetyTips.status}
                  </Badge>
                </div>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <p className="text-sm text-muted-foreground">
                    Last updated: {supportPages.safetyTips.lastUpdated}
                  </p>
                </div>
                <div className="space-y-2">
                  <Dialog>
                    <DialogTrigger asChild>
                      <Button className="w-full" size="sm">
                        <Edit className="w-4 h-4 mr-2" />
                        Edit Content
                      </Button>
                    </DialogTrigger>
                    <DialogContent className="max-w-4xl">
                      <DialogHeader>
                        <DialogTitle>Edit Safety Tips</DialogTitle>
                        <DialogDescription>
                          Update the safety tips content that users see
                        </DialogDescription>
                      </DialogHeader>
                      <div className="space-y-4">
                        <div>
                          <Label htmlFor="safetyTips-title">Title</Label>
                          <Input
                            id="safetyTips-title"
                            value={supportPages.safetyTips.title}
                            onChange={(e) => setSupportPages(prev => ({
                              ...prev,
                              safetyTips: { ...prev.safetyTips, title: e.target.value }
                            }))}
                          />
                        </div>
                        <div>
                          <Label htmlFor="safetyTips-content">Content</Label>
                          <Textarea
                            id="safetyTips-content"
                            value={supportPages.safetyTips.content}
                            onChange={(e) => setSupportPages(prev => ({
                              ...prev,
                              safetyTips: { ...prev.safetyTips, content: e.target.value }
                            }))}
                            rows={12}
                            className="font-mono text-sm"
                          />
                        </div>
                        <div className="flex gap-2 pt-4">
                          <Button 
                            onClick={() => {
                              setSupportPages(prev => ({
                                ...prev,
                                safetyTips: { ...prev.safetyTips, lastUpdated: new Date().toLocaleDateString() }
                              }));
                            }}
                            className="flex items-center gap-2"
                          >
                            <Save className="w-4 h-4" />
                            Save Changes
                          </Button>
                          <Button variant="outline" onClick={() => onNavigate("safety-tips")}>
                            <Eye className="w-4 h-4 mr-2" />
                            Preview
                          </Button>
                        </div>
                      </div>
                    </DialogContent>
                  </Dialog>
                  <Button variant="outline" size="sm" className="w-full" onClick={() => onNavigate("safety-tips")}>
                    <Eye className="w-4 h-4 mr-2" />
                    View Live Page
                  </Button>
                </div>
              </CardContent>
            </Card>

            {/* Community Guidelines */}
            <Card className="border-2">
              <CardHeader className="pb-3">
                <div className="flex items-center justify-between">
                  <CardTitle className="text-lg flex items-center gap-2">
                    <Users className="w-5 h-5 text-purple-600" />
                    Community Guidelines
                  </CardTitle>
                  <Badge className="bg-green-100 text-green-800">
                    {supportPages.communityGuidelines.status}
                  </Badge>
                </div>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <p className="text-sm text-muted-foreground">
                    Last updated: {supportPages.communityGuidelines.lastUpdated}
                  </p>
                </div>
                <div className="space-y-2">
                  <Dialog>
                    <DialogTrigger asChild>
                      <Button className="w-full" size="sm">
                        <Edit className="w-4 h-4 mr-2" />
                        Edit Content
                      </Button>
                    </DialogTrigger>
                    <DialogContent className="max-w-4xl">
                      <DialogHeader>
                        <DialogTitle>Edit Community Guidelines</DialogTitle>
                        <DialogDescription>
                          Update the community guidelines content that users see
                        </DialogDescription>
                      </DialogHeader>
                      <div className="space-y-4">
                        <div>
                          <Label htmlFor="communityGuidelines-title">Title</Label>
                          <Input
                            id="communityGuidelines-title"
                            value={supportPages.communityGuidelines.title}
                            onChange={(e) => setSupportPages(prev => ({
                              ...prev,
                              communityGuidelines: { ...prev.communityGuidelines, title: e.target.value }
                            }))}
                          />
                        </div>
                        <div>
                          <Label htmlFor="communityGuidelines-content">Content</Label>
                          <Textarea
                            id="communityGuidelines-content"
                            value={supportPages.communityGuidelines.content}
                            onChange={(e) => setSupportPages(prev => ({
                              ...prev,
                              communityGuidelines: { ...prev.communityGuidelines, content: e.target.value }
                            }))}
                            rows={12}
                            className="font-mono text-sm"
                          />
                        </div>
                        <div className="flex gap-2 pt-4">
                          <Button 
                            onClick={() => {
                              setSupportPages(prev => ({
                                ...prev,
                                communityGuidelines: { ...prev.communityGuidelines, lastUpdated: new Date().toLocaleDateString() }
                              }));
                            }}
                            className="flex items-center gap-2"
                          >
                            <Save className="w-4 h-4" />
                            Save Changes
                          </Button>
                          <Button variant="outline" onClick={() => onNavigate("community-guidelines")}>
                            <Eye className="w-4 h-4 mr-2" />
                            Preview
                          </Button>
                        </div>
                      </div>
                    </DialogContent>
                  </Dialog>
                  <Button variant="outline" size="sm" className="w-full" onClick={() => onNavigate("community-guidelines")}>
                    <Eye className="w-4 h-4 mr-2" />
                    View Live Page
                  </Button>
                </div>
              </CardContent>
            </Card>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}