import { useState } from "react";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "./ui/dialog";
import { Button } from "./ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "./ui/card";
import { Badge } from "./ui/badge";
import { Input } from "./ui/input";
import { Label } from "./ui/label";
import { Textarea } from "./ui/textarea";
import { RadioGroup, RadioGroupItem } from "./ui/radio-group";
import { Separator } from "./ui/separator";
import { motion, AnimatePresence } from "motion/react";
import {
  Heart,
  X,
  AlertTriangle,
  Gift,
  Crown,
  Clock,
  Users,
  MessageCircle,
  Star,
  Zap,
  Shield,
  TrendingUp,
  Calendar,
  DollarSign,
  Sparkles,
  ArrowLeft,
  CheckCircle,
  Phone,
  Mail,
  HeartHandshake
} from "lucide-react";
import { useTranslation } from "./TranslationProvider";
import { toast } from "sonner@2.0.3";

interface CancellationModalProps {
  isOpen: boolean;
  onClose: () => void;
  currentPlan: {
    name: string;
    price: number;
    duration: string;
  };
  onConfirmCancellation: () => void;
  onRetentionOffer: (offer: string) => void;
}

interface CancellationReason {
  id: string;
  label: string;
  icon: React.ComponentType<any>;
  description: string;
}

interface RetentionOffer {
  id: string;
  title: string;
  description: string;
  discount?: number;
  duration?: string;
  features?: string[];
  icon: React.ComponentType<any>;
  color: string;
  value: string;
}

export function CancellationModal({ 
  isOpen, 
  onClose, 
  currentPlan, 
  onConfirmCancellation,
  onRetentionOffer 
}: CancellationModalProps) {
  const { t } = useTranslation();
  const [step, setStep] = useState<'reasons' | 'retention' | 'confirmation'>('reasons');
  const [selectedReason, setSelectedReason] = useState<string>('');
  const [otherReason, setOtherReason] = useState<string>('');
  const [selectedOffer, setSelectedOffer] = useState<string>('');

  const cancellationReasons: CancellationReason[] = [
    {
      id: 'cost',
      label: 'Too expensive',
      icon: DollarSign,
      description: 'The subscription cost is too high for my budget'
    },
    {
      id: 'not-using',
      label: 'Not using it enough',
      icon: Clock,
      description: 'I don\'t use the platform frequently enough'
    },
    {
      id: 'found-someone',
      label: 'Found someone special',
      icon: Heart,
      description: 'I\'ve found my match and no longer need the service'
    },
    {
      id: 'poor-matches',
      label: 'Poor quality matches',
      icon: Users,
      description: 'The matches I receive don\'t meet my expectations'
    },
    {
      id: 'technical-issues',
      label: 'Technical problems',
      icon: AlertTriangle,
      description: 'Experiencing bugs or technical difficulties'
    },
    {
      id: 'privacy-concerns',
      label: 'Privacy concerns',
      icon: Shield,
      description: 'Worried about privacy and data security'
    },
    {
      id: 'trying-other',
      label: 'Trying other platforms',
      icon: TrendingUp,
      description: 'Want to explore other dating platforms'
    },
    {
      id: 'temporary-break',
      label: 'Taking a break',
      icon: Calendar,
      description: 'Need a temporary break from online dating'
    }
  ];

  const retentionOffers: RetentionOffer[] = [
    {
      id: 'discount-50',
      title: '50% Off for 3 Months',
      description: 'Continue enjoying premium features at half the price',
      discount: 50,
      duration: '3 months',
      icon: Gift,
      color: 'bg-green-100 text-green-800',
      value: `$${(currentPlan.price * 0.5).toFixed(2)}${currentPlan.duration}`
    },
    {
      id: 'free-month',
      title: 'Free Month Extension',
      description: 'Get one month completely free to find your perfect match',
      duration: '1 month',
      icon: Calendar,
      color: 'bg-blue-100 text-blue-800',
      value: 'Completely Free'
    },
    {
      id: 'premium-features',
      title: 'Exclusive Premium Features',
      description: 'Access to VIP features and priority support',
      features: ['Priority matching', 'Exclusive events', 'Personal dating coach'],
      icon: Crown,
      color: 'bg-purple-100 text-purple-800',
      value: 'No extra cost'
    },
    {
      id: 'pause-subscription',
      title: 'Pause Your Subscription',
      description: 'Keep your profile and matches, pause billing for up to 3 months',
      duration: 'Up to 3 months',
      icon: Clock,
      color: 'bg-yellow-100 text-yellow-800',
      value: 'No charges'
    }
  ];

  const handleReasonSelect = (reasonId: string) => {
    setSelectedReason(reasonId);
  };

  const handleContinueToRetention = () => {
    if (selectedReason || otherReason.trim()) {
      setStep('retention');
    }
  };

  const handleAcceptOffer = (offerId: string) => {
    setSelectedOffer(offerId);
    const offer = retentionOffers.find(o => o.id === offerId);
    toast.success(`Great choice! ${offer?.title} has been applied to your account.`, {
      description: "You'll see the changes reflected in your next billing cycle.",
      duration: 5000,
    });
    onRetentionOffer(offerId);
    onClose();
  };

  const handleFinalCancellation = () => {
    setStep('confirmation');
  };

  const handleConfirmCancellation = () => {
    toast.error("Subscription cancelled", {
      description: "We're sorry to see you go. Your subscription will remain active until the end of your billing period.",
      duration: 5000,
    });
    onConfirmCancellation();
    onClose();
  };

  const resetModal = () => {
    setStep('reasons');
    setSelectedReason('');
    setOtherReason('');
    setSelectedOffer('');
  };

  const handleClose = () => {
    resetModal();
    onClose();
  };

  const selectedReasonData = cancellationReasons.find(r => r.id === selectedReason);

  return (
    <Dialog open={isOpen} onOpenChange={handleClose}>
      <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
        <AnimatePresence mode="wait">
          {step === 'reasons' && (
            <motion.div
              key="reasons"
              initial={{ opacity: 0, x: 20 }}
              animate={{ opacity: 1, x: 0 }}
              exit={{ opacity: 0, x: -20 }}
              transition={{ duration: 0.3 }}
            >
              <DialogHeader className="pb-6">
                <DialogTitle className="flex items-center gap-3 text-2xl">
                  <div className="p-2 bg-red-100 rounded-lg">
                    <Heart className="w-6 h-6 text-red-600" />
                  </div>
                  We're sorry to see you go!
                </DialogTitle>
                <DialogDescription>
                  Before you cancel, help us understand what went wrong so we can improve your experience.
                </DialogDescription>
              </DialogHeader>

              <div className="space-y-6">
                {/* Current Plan Info */}
                <Card className="border-yellow-200 bg-yellow-50 dark:bg-yellow-950">
                  <CardContent className="p-4">
                    <div className="flex items-center justify-between">
                      <div className="flex items-center gap-3">
                        <Crown className="w-5 h-5 text-yellow-600" />
                        <div>
                          <p className="font-medium">{currentPlan.name} Plan</p>
                          <p className="text-sm text-muted-foreground">
                            ${currentPlan.price}{currentPlan.duration}
                          </p>
                        </div>
                      </div>
                      <Badge className="bg-yellow-600 text-white">
                        Active
                      </Badge>
                    </div>
                  </CardContent>
                </Card>

                {/* Cancellation Reasons */}
                <div>
                  <Label className="text-base font-medium mb-4 block">
                    What's your primary reason for cancelling?
                  </Label>
                  <RadioGroup value={selectedReason} onValueChange={handleReasonSelect}>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                      {cancellationReasons.map((reason) => {
                        const Icon = reason.icon;
                        return (
                          <div key={reason.id} className="relative">
                            <RadioGroupItem
                              value={reason.id}
                              id={reason.id}
                              className="peer sr-only"
                            />
                            <Label
                              htmlFor={reason.id}
                              className="flex items-start gap-3 p-4 border-2 border-border rounded-lg cursor-pointer transition-all hover:border-primary peer-checked:border-primary peer-checked:bg-primary/5"
                            >
                              <Icon className="w-5 h-5 text-muted-foreground mt-0.5 flex-shrink-0" />
                              <div>
                                <p className="font-medium">{reason.label}</p>
                                <p className="text-sm text-muted-foreground">
                                  {reason.description}
                                </p>
                              </div>
                            </Label>
                          </div>
                        );
                      })}
                    </div>
                  </RadioGroup>
                </div>

                {/* Other Reason */}
                <div>
                  <Label htmlFor="other-reason" className="text-base font-medium">
                    Other reason (optional)
                  </Label>
                  <Textarea
                    id="other-reason"
                    value={otherReason}
                    onChange={(e) => setOtherReason(e.target.value)}
                    placeholder="Please tell us more about your reason for cancelling..."
                    className="mt-2"
                    rows={3}
                  />
                </div>

                {/* Action Buttons */}
                <div className="flex gap-3 pt-4">
                  <Button
                    variant="outline"
                    onClick={handleClose}
                    className="flex-1"
                  >
                    <ArrowLeft className="w-4 h-4 mr-2" />
                    Keep My Subscription
                  </Button>
                  <Button
                    onClick={handleContinueToRetention}
                    disabled={!selectedReason && !otherReason.trim()}
                    className="flex-1 bg-red-600 hover:bg-red-700"
                  >
                    Continue Cancellation
                  </Button>
                </div>
              </div>
            </motion.div>
          )}

          {step === 'retention' && (
            <motion.div
              key="retention"
              initial={{ opacity: 0, x: 20 }}
              animate={{ opacity: 1, x: 0 }}
              exit={{ opacity: 0, x: -20 }}
              transition={{ duration: 0.3 }}
            >
              <DialogHeader className="pb-6">
                <DialogTitle className="flex items-center gap-3 text-2xl">
                  <div className="p-2 bg-gradient-to-br from-purple-500 to-pink-600 rounded-lg">
                    <HeartHandshake className="w-6 h-6 text-white" />
                  </div>
                  Wait! We'd love to keep you
                </DialogTitle>
                <DialogDescription>
                  {selectedReasonData ? 
                    `We understand ${selectedReasonData.label.toLowerCase()} can be frustrating. Here are some special offers just for you:` :
                    'We have some special offers that might change your mind:'
                  }
                </DialogDescription>
              </DialogHeader>

              <div className="space-y-6">
                {/* Retention Offers */}
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  {retentionOffers.map((offer) => {
                    const Icon = offer.icon;
                    return (
                      <motion.div
                        key={offer.id}
                        initial={{ opacity: 0, y: 20 }}
                        animate={{ opacity: 1, y: 0 }}
                        transition={{ duration: 0.3, delay: 0.1 }}
                      >
                        <Card className="border-2 hover:border-primary/50 transition-colors cursor-pointer group">
                          <CardHeader className="pb-3">
                            <div className="flex items-start justify-between">
                              <div className="flex items-center gap-3">
                                <div className={`p-2 rounded-lg ${offer.color.replace('text-', 'text-').replace('bg-', 'bg-')}`}>
                                  <Icon className="w-5 h-5" />
                                </div>
                                <div>
                                  <CardTitle className="text-lg">{offer.title}</CardTitle>
                                  <Badge className={offer.color}>
                                    {offer.value}
                                  </Badge>
                                </div>
                              </div>
                            </div>
                          </CardHeader>
                          <CardContent className="space-y-4">
                            <p className="text-sm text-muted-foreground">
                              {offer.description}
                            </p>
                            
                            {offer.features && (
                              <div className="space-y-2">
                                {offer.features.map((feature, index) => (
                                  <div key={index} className="flex items-center gap-2">
                                    <CheckCircle className="w-4 h-4 text-green-600" />
                                    <span className="text-sm">{feature}</span>
                                  </div>
                                ))}
                              </div>
                            )}

                            <Button
                              onClick={() => handleAcceptOffer(offer.id)}
                              className="w-full bg-gradient-to-r from-purple-600 to-pink-600 hover:from-purple-700 hover:to-pink-700"
                            >
                              <Sparkles className="w-4 h-4 mr-2" />
                              Accept This Offer
                            </Button>
                          </CardContent>
                        </Card>
                      </motion.div>
                    );
                  })}
                </div>

                {/* Support Options */}
                <Card className="border-blue-200 bg-blue-50 dark:bg-blue-950">
                  <CardHeader>
                    <CardTitle className="flex items-center gap-2">
                      <Phone className="w-5 h-5 text-blue-600" />
                      Need Personal Help?
                    </CardTitle>
                  </CardHeader>
                  <CardContent>
                    <p className="text-sm text-muted-foreground mb-4">
                      Our customer success team is here to help you make the most of SinglesZonne.
                    </p>
                    <div className="flex gap-3">
                      <Button variant="outline" size="sm" className="flex-1">
                        <Phone className="w-4 h-4 mr-2" />
                        Schedule Call
                      </Button>
                      <Button variant="outline" size="sm" className="flex-1">
                        <Mail className="w-4 h-4 mr-2" />
                        Email Support
                      </Button>
                    </div>
                  </CardContent>
                </Card>

                {/* Action Buttons */}
                <div className="flex gap-3 pt-4">
                  <Button
                    variant="outline"
                    onClick={() => setStep('reasons')}
                    className="flex-1"
                  >
                    <ArrowLeft className="w-4 h-4 mr-2" />
                    Back
                  </Button>
                  <Button
                    variant="outline"
                    onClick={handleClose}
                    className="flex-1"
                  >
                    Keep My Subscription
                  </Button>
                  <Button
                    onClick={handleFinalCancellation}
                    variant="destructive"
                    className="flex-1"
                  >
                    Still Cancel
                  </Button>
                </div>
              </div>
            </motion.div>
          )}

          {step === 'confirmation' && (
            <motion.div
              key="confirmation"
              initial={{ opacity: 0, x: 20 }}
              animate={{ opacity: 1, x: 0 }}
              exit={{ opacity: 0, x: -20 }}
              transition={{ duration: 0.3 }}
            >
              <DialogHeader className="pb-6 text-center">
                <DialogTitle className="flex items-center justify-center gap-3 text-2xl">
                  <div className="p-3 bg-red-100 rounded-full">
                    <AlertTriangle className="w-6 h-6 text-red-600" />
                  </div>
                  Final Confirmation
                </DialogTitle>
                <DialogDescription>
                  Are you absolutely sure you want to cancel your subscription?
                </DialogDescription>
              </DialogHeader>

              <div className="space-y-6">
                {/* What You'll Lose */}
                <Card className="border-red-200 bg-red-50 dark:bg-red-950">
                  <CardHeader>
                    <CardTitle className="text-red-600">
                      You'll lose access to:
                    </CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                      {[
                        'Unlimited likes and matches',
                        'See who liked your profile',
                        'Premium profile boost',
                        'Advanced search filters',
                        'Read receipts',
                        'Ad-free experience',
                        'Priority customer support',
                        'Exclusive events access'
                      ].map((feature, index) => (
                        <div key={index} className="flex items-center gap-2">
                          <X className="w-4 h-4 text-red-500" />
                          <span className="text-sm">{feature}</span>
                        </div>
                      ))}
                    </div>
                  </CardContent>
                </Card>

                {/* Cancellation Terms */}
                <div className="text-sm text-muted-foreground space-y-2">
                  <p>• Your subscription will remain active until the end of your current billing period</p>
                  <p>• You can reactivate anytime, but special pricing may not be available</p>
                  <p>• Your profile will be downgraded to free tier features</p>
                  <p>• All matches and conversations will be preserved</p>
                </div>

                {/* Final Action Buttons */}
                <div className="flex gap-3 pt-4">
                  <Button
                    variant="outline"
                    onClick={() => setStep('retention')}
                    className="flex-1"
                  >
                    <ArrowLeft className="w-4 h-4 mr-2" />
                    Back to Offers
                  </Button>
                  <Button
                    onClick={handleConfirmCancellation}
                    variant="destructive"
                    className="flex-1"
                  >
                    <X className="w-4 h-4 mr-2" />
                    Yes, Cancel Subscription
                  </Button>
                </div>
              </div>
            </motion.div>
          )}
        </AnimatePresence>
      </DialogContent>
    </Dialog>
  );
}