import { useState } from 'react';
import { Button } from './ui/button';
import { Card } from './ui/card';
import { Label } from './ui/label';
import { Badge } from './ui/badge';
import { 
  Palette, 
  RotateCcw, 
  Check,
  Sparkles,
  Eye
} from 'lucide-react';

interface ColorPickerProps {
  label: string;
  value: string;
  onChange: (color: string) => void;
  className?: string;
}

export function ColorPicker({ label, value, onChange, className = '' }: ColorPickerProps) {
  const [isOpen, setIsOpen] = useState(false);

  // Predefined color options
  const colorOptions = [
    '#ef4444', // Red
    '#f97316', // Orange
    '#f59e0b', // Amber
    '#eab308', // Yellow
    '#84cc16', // Lime
    '#22c55e', // Green
    '#10b981', // Emerald
    '#06b6d4', // Cyan
    '#0ea5e9', // Sky
    '#3b82f6', // Blue
    '#6366f1', // Indigo
    '#8b5cf6', // Purple
    '#a855f7', // Violet
    '#d946ef', // Fuchsia
    '#ec4899', // Pink
    '#f43f5e', // Rose
  ];

  return (
    <div className={`space-y-2 ${className}`}>
      <Label className="text-sm font-medium">{label}</Label>
      
      <div className="flex items-center gap-2">
        {/* Color Display Button */}
        <Button
          variant="outline"
          onClick={() => setIsOpen(!isOpen)}
          className="w-12 h-12 p-0 border-2 overflow-hidden"
          style={{ backgroundColor: value }}
        >
          {!value && <Palette className="w-5 h-5" />}
        </Button>
        
        {/* Hex Input */}
        <div className="flex-1">
          <input
            type="text"
            value={value}
            onChange={(e) => onChange(e.target.value)}
            placeholder="#000000"
            className="w-full px-3 py-2 border border-input rounded-md bg-background text-foreground text-sm focus:outline-none focus:ring-2 focus:ring-ring"
            pattern="^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$"
          />
        </div>
        
        {/* Native Color Picker */}
        <input
          type="color"
          value={value}
          onChange={(e) => onChange(e.target.value)}
          className="w-12 h-12 border-2 border-input rounded-md cursor-pointer bg-background"
        />
      </div>

      {/* Color Palette */}
      {isOpen && (
        <Card className="p-4 mt-2 border shadow-lg">
          <div className="space-y-3">
            <div className="flex items-center justify-between">
              <span className="text-sm font-medium">Quick Colors</span>
              <Button
                variant="ghost"
                size="sm"
                onClick={() => setIsOpen(false)}
              >
                <Check className="w-4 h-4" />
              </Button>
            </div>
            
            <div className="grid grid-cols-8 gap-2">
              {colorOptions.map((color) => (
                <Button
                  key={color}
                  variant="outline"
                  onClick={() => {
                    onChange(color);
                    setIsOpen(false);
                  }}
                  className={`w-8 h-8 p-0 border-2 transition-all duration-200 hover:scale-110 ${
                    value === color ? 'ring-2 ring-primary ring-offset-2' : ''
                  }`}
                  style={{ backgroundColor: color }}
                  title={color}
                />
              ))}
            </div>
          </div>
        </Card>
      )}
    </div>
  );
}

interface ColorThemePresetProps {
  name: string;
  colors: {
    primary: string;
    secondary: string;
    accent: string;
  };
  isActive: boolean;
  onClick: () => void;
}

export function ColorThemePreset({ name, colors, isActive, onClick }: ColorThemePresetProps) {
  return (
    <Card 
      className={`p-4 cursor-pointer transition-all duration-200 hover:shadow-lg ${
        isActive ? 'ring-2 ring-primary border-primary' : 'hover:border-primary/50'
      }`}
      onClick={onClick}
    >
      <div className="space-y-3">
        <div className="flex items-center justify-between">
          <span className="font-medium text-sm">{name}</span>
          {isActive && (
            <Badge variant="default" className="text-xs">
              <Check className="w-3 h-3 mr-1" />
              Active
            </Badge>
          )}
        </div>
        
        <div className="flex space-x-2">
          <div 
            className="w-6 h-6 rounded-full border-2 border-white shadow-sm"
            style={{ backgroundColor: colors.primary }}
            title={`Primary: ${colors.primary}`}
          />
          <div 
            className="w-6 h-6 rounded-full border-2 border-white shadow-sm"
            style={{ backgroundColor: colors.secondary }}
            title={`Secondary: ${colors.secondary}`}
          />
          <div 
            className="w-6 h-6 rounded-full border-2 border-white shadow-sm"
            style={{ backgroundColor: colors.accent }}
            title={`Accent: ${colors.accent}`}
          />
        </div>
        
        {/* Preview gradient */}
        <div 
          className="w-full h-3 rounded-full"
          style={{
            background: `linear-gradient(135deg, ${colors.primary}, ${colors.secondary}, ${colors.accent})`
          }}
        />
      </div>
    </Card>
  );
}

interface ProfilePreviewProps {
  theme: {
    primary: string;
    secondary: string;
    accent: string;
    gradient1: string;
    gradient2: string;
    gradient3: string;
  };
}

export function ProfilePreview({ theme }: ProfilePreviewProps) {
  return (
    <Card className="p-6 bg-gradient-to-br from-background to-muted">
      <div className="space-y-4">
        <div className="flex items-center gap-2 mb-4">
          <Eye className="w-5 h-5 text-primary" />
          <h3 className="font-semibold">Profile Preview</h3>
        </div>
        
        {/* Mock About Me Section */}
        <Card 
          className="overflow-hidden border-0 shadow-lg"
          style={{
            background: `linear-gradient(135deg, ${theme.gradient1}15, ${theme.gradient2}15, ${theme.gradient3}15)`
          }}
        >
          <div className="p-4">
            <div className="flex items-center gap-3 mb-4">
              <div 
                className="p-2 rounded-full"
                style={{ background: `linear-gradient(135deg, ${theme.primary}, ${theme.secondary})` }}
              >
                <Sparkles className="w-4 h-4 text-white" />
              </div>
              <div>
                <h4 
                  className="font-bold bg-clip-text text-transparent"
                  style={{
                    background: `linear-gradient(135deg, ${theme.primary}, ${theme.secondary})`,
                    WebkitBackgroundClip: 'text',
                    WebkitTextFillColor: 'transparent'
                  }}
                >
                  About Me
                </h4>
                <p className="text-xs text-muted-foreground">Preview of your custom colors</p>
              </div>
            </div>
            
            <div className="relative">
              <div 
                className="absolute left-0 top-0 w-1 h-full rounded-full"
                style={{ backgroundColor: theme.primary }}
              />
              <div className="pl-4">
                <p className="text-sm text-foreground/80 mb-2">
                  This is how your profile will look with the selected colors.
                </p>
                
                <div className="flex flex-wrap gap-2">
                  <Badge 
                    variant="secondary" 
                    className="text-xs border"
                    style={{ 
                      backgroundColor: `${theme.primary}20`,
                      color: theme.primary,
                      borderColor: `${theme.primary}40`
                    }}
                  >
                    ✨ Creative
                  </Badge>
                  <Badge 
                    variant="secondary" 
                    className="text-xs border"
                    style={{ 
                      backgroundColor: `${theme.secondary}20`,
                      color: theme.secondary,
                      borderColor: `${theme.secondary}40`
                    }}
                  >
                    💫 Authentic
                  </Badge>
                  <Badge 
                    variant="secondary" 
                    className="text-xs border"
                    style={{ 
                      backgroundColor: `${theme.accent}20`,
                      color: theme.accent,
                      borderColor: `${theme.accent}40`
                    }}
                  >
                    🎯 Passionate
                  </Badge>
                </div>
              </div>
            </div>
          </div>
          
          {/* Bottom accent */}
          <div 
            className="h-2"
            style={{
              background: `linear-gradient(135deg, ${theme.gradient1}, ${theme.gradient2}, ${theme.gradient3})`
            }}
          />
        </Card>
      </div>
    </Card>
  );
}