import { useState } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "./ui/card";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Avatar } from "./ui/avatar";
import { Progress } from "./ui/progress";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "./ui/tabs";
import { 
  Trophy, 
  Heart, 
  Star, 
  Calendar, 
  Clock,
  Users,
  Crown,
  Vote,
  Gift,
  TrendingUp,
  Eye,
  Award,
  Timer,
  Sparkles,
  ChevronRight
} from "lucide-react";
import { motion } from "motion/react";
import { ImageWithFallback } from "./figma/ImageWithFallback";
import { useUser } from "./UserProvider";

interface Contest {
  id: string;
  title: string;
  description: string;
  category: 'beauty' | 'personality' | 'photo' | 'story';
  status: 'active' | 'upcoming' | 'ended';
  startDate: string;
  endDate: string;
  prize: string;
  prizeValue: number;
  maxParticipants: number;
  currentParticipants: number;
  isParticipating: boolean;
  canVote: boolean;
  totalVotes: number;
  userVotes: number;
  adminVotes: number;
  image: string;
  participants: ContestParticipant[];
  rules: string[];
  votingType: 'public' | 'admin' | 'mixed';
}

interface ContestParticipant {
  id: string;
  userId: string;
  username: string;
  avatar: string;
  photos: string[];
  story?: string;
  votes: number;
  adminVotes: number;
  totalScore: number;
  rank: number;
  isVerified: boolean;
  badges: string[];
}

interface ContestsPageProps {
  onNavigate?: (view: string) => void;
}

export function ContestsPage({ onNavigate }: ContestsPageProps) {
  const { user, isRole } = useUser();
  const [activeTab, setActiveTab] = useState("active");
  const [selectedContest, setSelectedContest] = useState<string | null>(null);
  const [votingContest, setVotingContest] = useState<string | null>(null);

  // Mock contest data
  const contests: Contest[] = [
    {
      id: "1",
      title: "💖 Most Charming Smile",
      description: "Show off your most captivating smile and win amazing prizes!",
      category: "beauty",
      status: "active",
      startDate: "2024-01-15",
      endDate: "2024-01-30",
      prize: "Premium Subscription + $500 Gift Card",
      prizeValue: 500,
      maxParticipants: 100,
      currentParticipants: 67,
      isParticipating: false,
      canVote: true,
      totalVotes: 2134,
      userVotes: 1820,
      adminVotes: 314,
      image: "/api/placeholder/400/300",
      votingType: "mixed",
      participants: [
        {
          id: "p1",
          userId: "u1",
          username: "SunnySmile",
          avatar: "/api/placeholder/80/80",
          photos: ["/api/placeholder/200/300", "/api/placeholder/200/300"],
          votes: 345,
          adminVotes: 50,
          totalScore: 395,
          rank: 1,
          isVerified: true,
          badges: ["verified", "premium"]
        },
        {
          id: "p2", 
          userId: "u2",
          username: "CharmingGlow",
          avatar: "/api/placeholder/80/80",
          photos: ["/api/placeholder/200/300"],
          votes: 298,
          adminVotes: 45,
          totalScore: 343,
          rank: 2,
          isVerified: true,
          badges: ["verified"]
        },
        {
          id: "p3",
          userId: "u3", 
          username: "RadiantVibes",
          avatar: "/api/placeholder/80/80",
          photos: ["/api/placeholder/200/300", "/api/placeholder/200/300"],
          votes: 267,
          adminVotes: 40,
          totalScore: 307,
          rank: 3,
          isVerified: false,
          badges: []
        }
      ],
      rules: [
        "Must submit 1-3 photos showing your smile",
        "Photos must be recent (within 6 months)",
        "No filters or heavy editing allowed",
        "Must be a verified member to participate",
        "Voting ends on contest end date",
        "Winners announced within 48 hours"
      ]
    },
    {
      id: "2",
      title: "✨ Best Love Story",
      description: "Share your most romantic dating experience or dream date story!",
      category: "story",
      status: "active",
      startDate: "2024-01-10",
      endDate: "2024-01-25",
      prize: "Romantic Getaway for Two + Premium Features",
      prizeValue: 1000,
      maxParticipants: 50,
      currentParticipants: 32,
      isParticipating: true,
      canVote: false,
      totalVotes: 1456,
      userVotes: 1201,
      adminVotes: 255,
      image: "/api/placeholder/400/300",
      votingType: "mixed",
      participants: [],
      rules: [
        "Submit a written story (500-1000 words)",
        "Story must be original and personal",
        "Appropriate content only",
        "Story must be in English",
        "One submission per participant"
      ]
    },
    {
      id: "3",
      title: "🌟 Rising Star",
      description: "New members showcase - show us what makes you special!",
      category: "personality",
      status: "upcoming",
      startDate: "2024-02-01",
      endDate: "2024-02-15",
      prize: "1 Year Premium + $300 Shopping Spree",
      prizeValue: 300,
      maxParticipants: 75,
      currentParticipants: 0,
      isParticipating: false,
      canVote: false,
      totalVotes: 0,
      userVotes: 0,
      adminVotes: 0,
      image: "/api/placeholder/400/300",
      votingType: "public",
      participants: [],
      rules: [
        "Must be a member for less than 3 months",
        "Submit photos and personality questions",
        "Complete profile required",
        "Active participation in community"
      ]
    }
  ];

  const getStatusBadge = (status: Contest['status']) => {
    switch (status) {
      case 'active':
        return <Badge className="bg-green-100 text-green-800 dark:bg-green-900/20 dark:text-green-400">🟢 Active</Badge>;
      case 'upcoming':
        return <Badge className="bg-blue-100 text-blue-800 dark:bg-blue-900/20 dark:text-blue-400">🔵 Upcoming</Badge>;
      case 'ended':
        return <Badge className="bg-gray-100 text-gray-800 dark:bg-gray-900/20 dark:text-gray-400">⚫ Ended</Badge>;
      default:
        return <Badge>{status}</Badge>;
    }
  };

  const getCategoryIcon = (category: Contest['category']) => {
    switch (category) {
      case 'beauty':
        return <Sparkles className="w-4 h-4" />;
      case 'personality':
        return <Heart className="w-4 h-4" />;
      case 'photo':
        return <Eye className="w-4 h-4" />;
      case 'story':
        return <Award className="w-4 h-4" />;
      default:
        return <Trophy className="w-4 h-4" />;
    }
  };

  const activeContests = contests.filter(c => c.status === 'active');
  const upcomingContests = contests.filter(c => c.status === 'upcoming');
  const endedContests = contests.filter(c => c.status === 'ended');

  const handleVote = (contestId: string, participantId: string) => {
    // Mock voting logic
    console.log(`Voting for participant ${participantId} in contest ${contestId}`);
    // In real app, this would call an API
  };

  const handleJoinContest = (contestId: string) => {
    // Mock join logic
    console.log(`Joining contest ${contestId}`);
    // In real app, this would call an API
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="text-center">
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <h2 className="text-3xl font-bold mb-4 flex items-center justify-center gap-3">
            <Trophy className="w-8 h-8 text-yellow-500" />
            Community Contests
          </h2>
          <p className="text-muted-foreground max-w-2xl mx-auto">
            Participate in exciting contests, showcase your best self, and win amazing prizes! Vote for your favorites and support fellow members.
          </p>
        </motion.div>
      </div>

      {/* Contest Tabs */}
      <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
        <TabsList className="grid w-full grid-cols-3">
          <TabsTrigger value="active" className="flex items-center gap-2">
            <Trophy className="w-4 h-4" />
            Active ({activeContests.length})
          </TabsTrigger>
          <TabsTrigger value="upcoming" className="flex items-center gap-2">
            <Clock className="w-4 h-4" />
            Upcoming ({upcomingContests.length})
          </TabsTrigger>
          <TabsTrigger value="ended" className="flex items-center gap-2">
            <Award className="w-4 h-4" />
            Past Winners ({endedContests.length})
          </TabsTrigger>
        </TabsList>

        {/* Active Contests */}
        <TabsContent value="active" className="space-y-6">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            {activeContests.map((contest, index) => (
              <motion.div
                key={contest.id}
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ duration: 0.5, delay: index * 0.1 }}
              >
                <Card className="overflow-hidden hover:shadow-lg transition-shadow">
                  <div className="relative">
                    <ImageWithFallback
                      src={contest.image}
                      alt={contest.title}
                      className="w-full h-48 object-cover"
                    />
                    <div className="absolute top-4 left-4 flex gap-2">
                      {getStatusBadge(contest.status)}
                      <Badge className="bg-white/90 text-gray-800">
                        {getCategoryIcon(contest.category)}
                        <span className="ml-1 capitalize">{contest.category}</span>
                      </Badge>
                    </div>
                    {contest.prizeValue && (
                      <div className="absolute top-4 right-4">
                        <Badge className="bg-gradient-to-r from-yellow-400 to-yellow-600 text-yellow-900">
                          <Gift className="w-3 h-3 mr-1" />
                          ${contest.prizeValue}
                        </Badge>
                      </div>
                    )}
                  </div>

                  <CardHeader>
                    <CardTitle className="flex items-start justify-between">
                      <span>{contest.title}</span>
                      <div className="flex items-center gap-1 text-sm text-muted-foreground">
                        <Timer className="w-4 h-4" />
                        <span>Ends {contest.endDate}</span>
                      </div>
                    </CardTitle>
                    <CardDescription>{contest.description}</CardDescription>
                  </CardHeader>

                  <CardContent className="space-y-4">
                    {/* Contest Stats */}
                    <div className="grid grid-cols-2 gap-4">
                      <div className="text-center p-3 bg-muted/50 rounded-lg">
                        <div className="flex items-center justify-center gap-1 mb-1">
                          <Users className="w-4 h-4 text-blue-600" />
                        </div>
                        <p className="text-sm font-medium">{contest.currentParticipants}/{contest.maxParticipants}</p>
                        <p className="text-xs text-muted-foreground">Participants</p>
                      </div>
                      <div className="text-center p-3 bg-muted/50 rounded-lg">
                        <div className="flex items-center justify-center gap-1 mb-1">
                          <Vote className="w-4 h-4 text-purple-600" />
                        </div>
                        <p className="text-sm font-medium">{contest.totalVotes.toLocaleString()}</p>
                        <p className="text-xs text-muted-foreground">Total Votes</p>
                      </div>
                    </div>

                    {/* Progress Bar */}
                    <div className="space-y-2">
                      <div className="flex justify-between text-sm">
                        <span>Contest Progress</span>
                        <span>{Math.round((contest.currentParticipants / contest.maxParticipants) * 100)}%</span>
                      </div>
                      <Progress 
                        value={(contest.currentParticipants / contest.maxParticipants) * 100}
                        className="h-2"
                      />
                    </div>

                    {/* Prize Info */}
                    <div className="p-3 bg-gradient-to-r from-yellow-50 to-orange-50 dark:from-yellow-900/20 dark:to-orange-900/20 rounded-lg">
                      <div className="flex items-center gap-2 mb-1">
                        <Trophy className="w-4 h-4 text-yellow-600" />
                        <span className="font-medium text-sm">Prize</span>
                      </div>
                      <p className="text-sm text-muted-foreground">{contest.prize}</p>
                    </div>

                    {/* Action Buttons */}
                    <div className="flex gap-2">
                      {!contest.isParticipating ? (
                        <Button 
                          className="flex-1"
                          onClick={() => handleJoinContest(contest.id)}
                          disabled={contest.currentParticipants >= contest.maxParticipants}
                        >
                          <Crown className="w-4 h-4 mr-2" />
                          Join Contest
                        </Button>
                      ) : (
                        <Button className="flex-1" variant="outline" disabled>
                          <Crown className="w-4 h-4 mr-2" />
                          Participating
                        </Button>
                      )}
                      <Button 
                        variant="outline"
                        onClick={() => setSelectedContest(contest.id)}
                      >
                        <Eye className="w-4 h-4 mr-2" />
                        View Details
                      </Button>
                      {contest.canVote && (
                        <Button 
                          variant="outline"
                          onClick={() => setVotingContest(contest.id)}
                        >
                          <Vote className="w-4 h-4 mr-2" />
                          Vote
                        </Button>
                      )}
                    </div>
                  </CardContent>
                </Card>
              </motion.div>
            ))}
          </div>
        </TabsContent>

        {/* Upcoming Contests */}
        <TabsContent value="upcoming" className="space-y-6">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            {upcomingContests.map((contest, index) => (
              <motion.div
                key={contest.id}
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ duration: 0.5, delay: index * 0.1 }}
              >
                <Card className="overflow-hidden">
                  <div className="relative">
                    <ImageWithFallback
                      src={contest.image}
                      alt={contest.title}
                      className="w-full h-48 object-cover grayscale-[30%]"
                    />
                    <div className="absolute inset-0 bg-gradient-to-t from-black/60 to-transparent" />
                    <div className="absolute top-4 left-4">
                      {getStatusBadge(contest.status)}
                    </div>
                    <div className="absolute bottom-4 left-4 text-white">
                      <p className="text-sm">Starts {contest.startDate}</p>
                    </div>
                  </div>

                  <CardHeader>
                    <CardTitle>{contest.title}</CardTitle>
                    <CardDescription>{contest.description}</CardDescription>
                  </CardHeader>

                  <CardContent className="space-y-4">
                    <div className="p-3 bg-muted/50 rounded-lg">
                      <div className="flex items-center gap-2 mb-2">
                        <Gift className="w-4 h-4 text-yellow-600" />
                        <span className="font-medium text-sm">Prize</span>
                      </div>
                      <p className="text-sm">{contest.prize}</p>
                    </div>

                    <Button className="w-full" disabled>
                      <Clock className="w-4 h-4 mr-2" />
                      Coming Soon
                    </Button>
                  </CardContent>
                </Card>
              </motion.div>
            ))}
          </div>
        </TabsContent>

        {/* Past Winners */}
        <TabsContent value="ended" className="space-y-6">
          <div className="text-center py-12">
            <Trophy className="w-16 h-16 text-yellow-500 mx-auto mb-4" />
            <h3 className="text-xl font-semibold mb-2">No Past Contests Yet</h3>
            <p className="text-muted-foreground">
              Check back soon to see our contest winners and their amazing entries!
            </p>
          </div>
        </TabsContent>
      </Tabs>

      {/* Contest Detail Modal would go here */}
      {selectedContest && (
        <div className="fixed inset-0 bg-black/50 flex items-center justify-center p-4 z-50">
          <div className="bg-background rounded-lg max-w-4xl w-full max-h-[90vh] overflow-y-auto">
            {/* Contest details modal content */}
            <div className="p-6">
              <div className="flex justify-between items-start mb-4">
                <h3 className="text-2xl font-bold">Contest Details</h3>
                <Button variant="ghost" onClick={() => setSelectedContest(null)}>×</Button>
              </div>
              {/* Detailed contest view would go here */}
            </div>
          </div>
        </div>
      )}

      {/* Voting Modal would go here */}
      {votingContest && (
        <div className="fixed inset-0 bg-black/50 flex items-center justify-center p-4 z-50">
          <div className="bg-background rounded-lg max-w-4xl w-full max-h-[90vh] overflow-y-auto">
            {/* Voting interface would go here */}
            <div className="p-6">
              <div className="flex justify-between items-start mb-4">
                <h3 className="text-2xl font-bold">Vote for Participants</h3>
                <Button variant="ghost" onClick={() => setVotingContest(null)}>×</Button>
              </div>
              {/* Participant voting grid would go here */}
            </div>
          </div>
        </div>
      )}
    </div>
  );
}