import { useState, useEffect } from "react";
import { Button } from "./ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "./ui/card";
import { Checkbox } from "./ui/checkbox";
import { Separator } from "./ui/separator";
import { Cookie, Settings, Check, X, Info, ChevronUp, ChevronDown } from "lucide-react";
import { motion, AnimatePresence } from "motion/react";

interface CookieConsentProps {
  isOpen: boolean;
  onClose: () => void;
  onNavigate?: (page: string) => void;
}

interface CookieCategory {
  id: string;
  name: string;
  description: string;
  required: boolean;
  enabled: boolean;
}

const cookieCategories: CookieCategory[] = [
  {
    id: "necessary",
    name: "Necessary Cookies",
    description: "Essential for the website to function properly. These cannot be disabled.",
    required: true,
    enabled: true,
  },
  {
    id: "functional",
    name: "Functional Cookies",
    description: "Enable enhanced functionality like chat, notifications, and personalized features.",
    required: false,
    enabled: false,
  },
  {
    id: "analytics",
    name: "Analytics Cookies",
    description: "Help us understand how visitors interact with our website to improve user experience.",
    required: false,
    enabled: false,
  },
  {
    id: "marketing",
    name: "Marketing Cookies",
    description: "Used to track visitors and display personalized advertisements and content.",
    required: false,
    enabled: false,
  },
];

export function CookieConsent({ isOpen, onClose, onNavigate }: CookieConsentProps) {
  const [categories, setCategories] = useState<CookieCategory[]>(cookieCategories);
  const [showDetails, setShowDetails] = useState(false);
  const [isExpanded, setIsExpanded] = useState(false);

  // Handle body padding when component is active
  useEffect(() => {
    if (isOpen) {
      if (isExpanded) {
        document.body.classList.add('cookie-consent-expanded');
        document.body.classList.remove('cookie-consent-active');
      } else {
        document.body.classList.add('cookie-consent-active');
        document.body.classList.remove('cookie-consent-expanded');
      }
    } else {
      document.body.classList.remove('cookie-consent-active', 'cookie-consent-expanded');
    }

    return () => {
      document.body.classList.remove('cookie-consent-active', 'cookie-consent-expanded');
    };
  }, [isOpen, isExpanded]);

  const updateCategory = (id: string, enabled: boolean) => {
    setCategories(prev => 
      prev.map(cat => 
        cat.id === id ? { ...cat, enabled } : cat
      )
    );
  };

  const acceptAll = () => {
    const updatedCategories = categories.map(cat => ({ ...cat, enabled: true }));
    setCategories(updatedCategories);
    saveCookiePreferences(updatedCategories);
    document.body.classList.remove('cookie-consent-active', 'cookie-consent-expanded');
    onClose();
  };

  const acceptNecessary = () => {
    const updatedCategories = categories.map(cat => ({ 
      ...cat, 
      enabled: cat.required 
    }));
    setCategories(updatedCategories);
    saveCookiePreferences(updatedCategories);
    document.body.classList.remove('cookie-consent-active', 'cookie-consent-expanded');
    onClose();
  };

  const saveCustom = () => {
    saveCookiePreferences(categories);
    document.body.classList.remove('cookie-consent-active', 'cookie-consent-expanded');
    onClose();
  };

  const saveCookiePreferences = (prefs: CookieCategory[]) => {
    const preferences = prefs.reduce((acc, cat) => {
      acc[cat.id] = cat.enabled;
      return acc;
    }, {} as Record<string, boolean>);
    
    localStorage.setItem('cookiePreferences', JSON.stringify(preferences));
    localStorage.setItem('cookieConsentGiven', 'true');
    
    // In a real app, you would apply these preferences to actual cookie handling
    console.log('Cookie preferences saved:', preferences);
  };

  if (!isOpen) return null;

  return (
    <AnimatePresence>
      <motion.div
        initial={{ y: "100%" }}
        animate={{ y: 0 }}
        exit={{ y: "100%" }}
        className="fixed bottom-0 left-0 right-0 z-50"
      >
        <Card className="border-t-2 border-x-0 border-b-0 rounded-t-2xl rounded-b-none bg-background/95 backdrop-blur-sm shadow-2xl">
          {/* Collapsed State - Tab Header */}
          {!isExpanded && (
            <div className="px-6 py-4">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-3">
                  <Cookie className="w-5 h-5 text-orange-600" />
                  <div>
                    <h4 className="text-sm font-medium">Cookie Settings</h4>
                    <p className="text-xs text-muted-foreground">
                      We use cookies to enhance your experience
                    </p>
                  </div>
                </div>
                
                <div className="flex items-center gap-2">
                  <Button onClick={acceptAll} size="sm" className="text-xs px-3">
                    <Check className="w-3 h-3 mr-1" />
                    Accept All
                  </Button>
                  <Button 
                    variant="outline" 
                    onClick={acceptNecessary}
                    size="sm"
                    className="text-xs px-3"
                  >
                    Essential Only
                  </Button>
                  <Button 
                    variant="ghost" 
                    onClick={() => setIsExpanded(true)}
                    size="sm"
                    className="text-xs px-2"
                  >
                    <ChevronUp className="w-4 h-4" />
                  </Button>
                  <Button 
                    variant="ghost" 
                    size="sm"
                    onClick={() => {
                      document.body.classList.remove('cookie-consent-active', 'cookie-consent-expanded');
                      onClose();
                    }}
                    className="text-xs px-2"
                  >
                    <X className="w-4 h-4" />
                  </Button>
                </div>
              </div>
            </div>
          )}

          {/* Expanded State - Full Content */}
          <AnimatePresence>
            {isExpanded && (
              <motion.div
                initial={{ height: 0, opacity: 0 }}
                animate={{ height: "auto", opacity: 1 }}
                exit={{ height: 0, opacity: 0 }}
                className="overflow-hidden"
              >
                <CardHeader className="space-y-4 pb-4">
                  <div className="flex items-center justify-between">
                    <CardTitle className="flex items-center gap-3">
                      <Cookie className="w-6 h-6 text-orange-600" />
                      Cookie Preferences
                    </CardTitle>
                    <div className="flex items-center gap-2">
                      <Button 
                        variant="ghost" 
                        size="sm"
                        onClick={() => setIsExpanded(false)}
                        className="h-8 w-8 p-0"
                      >
                        <ChevronDown className="w-4 h-4" />
                      </Button>
                      <Button 
                        variant="ghost" 
                        size="sm"
                        onClick={() => {
                          document.body.classList.remove('cookie-consent-active', 'cookie-consent-expanded');
                          onClose();
                        }}
                        className="h-8 w-8 p-0"
                      >
                        <X className="w-4 h-4" />
                      </Button>
                    </div>
                  </div>
                  <p className="text-muted-foreground">
                    We use cookies to enhance your experience on SinglesZonne. Choose which cookies you'd like to accept.
                  </p>
                </CardHeader>

                <CardContent className="space-y-6 max-h-[60vh] overflow-y-auto">
                  {/* Quick Actions */}
                  <div className="flex flex-wrap gap-3">
                    <Button onClick={acceptAll} className="flex-1 min-w-[120px]">
                      <Check className="w-4 h-4 mr-2" />
                      Accept All
                    </Button>
                    <Button 
                      variant="outline" 
                      onClick={acceptNecessary}
                      className="flex-1 min-w-[120px]"
                    >
                      Necessary Only
                    </Button>
                    <Button 
                      variant="ghost" 
                      onClick={() => setShowDetails(!showDetails)}
                      className="flex-1 min-w-[120px]"
                    >
                      <Settings className="w-4 h-4 mr-2" />
                      {showDetails ? "Hide Details" : "Customize"}
                    </Button>
                  </div>

                  <AnimatePresence>
                    {showDetails && (
                      <motion.div
                        initial={{ height: 0, opacity: 0 }}
                        animate={{ height: "auto", opacity: 1 }}
                        exit={{ height: 0, opacity: 0 }}
                        className="space-y-4 overflow-hidden"
                      >
                        <Separator />
                        
                        <div className="space-y-4">
                          <h4 className="font-semibold flex items-center gap-2">
                            <Info className="w-4 h-4" />
                            Cookie Categories
                          </h4>
                          
                          {categories.map((category) => (
                            <div key={category.id} className="space-y-3 p-4 border rounded-lg">
                              <div className="flex items-start justify-between gap-3">
                                <div className="flex-1 space-y-1">
                                  <div className="flex items-center gap-2">
                                    <h5 className="font-medium">{category.name}</h5>
                                    {category.required && (
                                      <span className="text-xs bg-green-100 dark:bg-green-900/30 text-green-700 dark:text-green-300 px-2 py-1 rounded">
                                        Required
                                      </span>
                                    )}
                                  </div>
                                  <p className="text-sm text-muted-foreground">
                                    {category.description}
                                  </p>
                                </div>
                                <Checkbox
                                  checked={category.enabled}
                                  disabled={category.required}
                                  onCheckedChange={(checked) => 
                                    updateCategory(category.id, checked as boolean)
                                  }
                                  className="mt-1"
                                />
                              </div>
                            </div>
                          ))}
                        </div>

                        <Separator />

                        <div className="flex gap-3">
                          <Button onClick={saveCustom} className="flex-1">
                            Save Preferences
                          </Button>
                          <Button 
                            variant="outline" 
                            onClick={() => setShowDetails(false)}
                            className="flex-1"
                          >
                            Cancel
                          </Button>
                        </div>
                      </motion.div>
                    )}
                  </AnimatePresence>

                  {/* Cookie Policy Link */}
                  <div className="text-center pt-4 border-t">
                    <p className="text-xs text-muted-foreground">
                      For more information, read our{" "}
                      <button 
                        className="text-primary hover:underline"
                        onClick={() => {
                          onNavigate?.('cookies');
                          document.body.classList.remove('cookie-consent-active', 'cookie-consent-expanded');
                          onClose();
                        }}
                      >
                        Cookie Policy
                      </button>
                      {" "}and{" "}
                      <button 
                        className="text-primary hover:underline"
                        onClick={() => {
                          onNavigate?.('privacy');
                          document.body.classList.remove('cookie-consent-active', 'cookie-consent-expanded');
                          onClose();
                        }}
                      >
                        Privacy Policy
                      </button>
                    </p>
                  </div>
                </CardContent>
              </motion.div>
            )}
          </AnimatePresence>
        </Card>
      </motion.div>
    </AnimatePresence>
  );
}

// Hook to manage cookie consent state
export function useCookieConsent() {
  const [showConsent, setShowConsent] = useState(false);

  useEffect(() => {
    const consentGiven = localStorage.getItem('cookieConsentGiven');
    if (!consentGiven) {
      // Delay showing consent to avoid blocking initial render
      const timer = setTimeout(() => setShowConsent(true), 1000);
      return () => clearTimeout(timer);
    }
  }, []);

  const openConsent = () => setShowConsent(true);
  const closeConsent = () => setShowConsent(false);

  return {
    showConsent,
    openConsent,
    closeConsent,
  };
}