import { useState } from "react";
import { Header } from "./Header";
import { Footer } from "./Footer";
import { Button } from "./ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "./ui/card";
import { ChevronLeft, Cookie, Shield, AlertCircle, Settings } from "lucide-react";
import { Badge } from "./ui/badge";
import { Separator } from "./ui/separator";
import { Alert, AlertDescription } from "./ui/alert";

interface CookiesPageProps {
  onBack: () => void;
  onNavigate?: (view: string) => void;
  onOpenCookieConsent?: () => void;
}

export function CookiesPage({ onBack, onNavigate, onOpenCookieConsent }: CookiesPageProps) {
  const [lastUpdated] = useState("December 20, 2024");

  return (
    <div className="min-h-screen bg-background">
      <Header currentView="cookies" onNavigate={onNavigate} />
      
      <div className="w-full px-6 py-8">
        <div className="max-w-4xl mx-auto">
          {/* Header */}
          <div className="mb-8">
            <Button
              variant="ghost"
              onClick={onBack}
              className="mb-4 flex items-center gap-2"
            >
              <ChevronLeft className="w-4 h-4" />
              Back
            </Button>
            
            <div className="flex items-center gap-4 mb-4">
              <div className="p-3 bg-orange-100 dark:bg-orange-900/20 rounded-full">
                <Cookie className="w-8 h-8 text-orange-600" />
              </div>
              <div>
                <h1 className="text-3xl font-bold">Cookie Policy</h1>
                <p className="text-muted-foreground">
                  How SinglesZonne uses cookies and similar technologies
                </p>
              </div>
            </div>
            
            <div className="flex items-center gap-4">
              <Badge variant="secondary">
                Last updated: {lastUpdated}
              </Badge>
              <Button 
                variant="outline" 
                size="sm"
                onClick={onOpenCookieConsent}
                className="flex items-center gap-2"
              >
                <Settings className="w-4 h-4" />
                Cookie Preferences
              </Button>
            </div>
          </div>

          {/* Cookie Consent Alert */}
          <Alert className="mb-8 border-orange-200 bg-orange-50 dark:border-orange-800 dark:bg-orange-900/20">
            <AlertCircle className="w-4 h-4" />
            <AlertDescription>
              <strong>Cookie Preferences:</strong> You can manage your cookie settings at any time by clicking the "Cookie Preferences" button above or in our footer.
            </AlertDescription>
          </Alert>

          <div className="space-y-8">
            {/* What Are Cookies */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Shield className="w-5 h-5 text-blue-600" />
                  What Are Cookies?
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <p>
                  Cookies are small text files that are placed on your device when you visit our website. 
                  They help us provide you with a better experience by remembering your preferences and 
                  understanding how you use our platform.
                </p>
                <p>
                  We use both session cookies (which expire when you close your browser) and persistent 
                  cookies (which remain on your device until they expire or you delete them).
                </p>
              </CardContent>
            </Card>

            {/* Types of Cookies */}
            <Card>
              <CardHeader>
                <CardTitle>Types of Cookies We Use</CardTitle>
              </CardHeader>
              <CardContent className="space-y-6">
                <div className="space-y-4">
                  <div className="border-l-4 border-green-500 pl-4">
                    <h3 className="font-semibold text-green-700 dark:text-green-400">Essential Cookies</h3>
                    <p className="text-sm text-muted-foreground mt-1">
                      These cookies are necessary for the website to function properly. They enable basic 
                      features like page navigation, authentication, and security features.
                    </p>
                    <div className="mt-2">
                      <Badge variant="outline" className="text-xs">Required</Badge>
                    </div>
                  </div>

                  <Separator />

                  <div className="border-l-4 border-blue-500 pl-4">
                    <h3 className="font-semibold text-blue-700 dark:text-blue-400">Functional Cookies</h3>
                    <p className="text-sm text-muted-foreground mt-1">
                      These cookies allow us to remember your preferences (like language settings, 
                      theme preferences) and provide enhanced features.
                    </p>
                    <div className="mt-2">
                      <Badge variant="outline" className="text-xs">Optional</Badge>
                    </div>
                  </div>

                  <Separator />

                  <div className="border-l-4 border-purple-500 pl-4">
                    <h3 className="font-semibold text-purple-700 dark:text-purple-400">Analytics Cookies</h3>
                    <p className="text-sm text-muted-foreground mt-1">
                      These cookies help us understand how visitors interact with our website by 
                      collecting and reporting information anonymously.
                    </p>
                    <div className="mt-2">
                      <Badge variant="outline" className="text-xs">Optional</Badge>
                    </div>
                  </div>

                  <Separator />

                  <div className="border-l-4 border-orange-500 pl-4">
                    <h3 className="font-semibold text-orange-700 dark:text-orange-400">Marketing Cookies</h3>
                    <p className="text-sm text-muted-foreground mt-1">
                      These cookies track your activity across websites to deliver more relevant 
                      advertisements and measure the effectiveness of advertising campaigns.
                    </p>
                    <div className="mt-2">
                      <Badge variant="outline" className="text-xs">Optional</Badge>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Specific Cookies */}
            <Card>
              <CardHeader>
                <CardTitle>Specific Cookies We Use</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="overflow-x-auto">
                  <table className="w-full text-sm">
                    <thead>
                      <tr className="border-b">
                        <th className="text-left p-2 font-medium">Cookie Name</th>
                        <th className="text-left p-2 font-medium">Purpose</th>
                        <th className="text-left p-2 font-medium">Type</th>
                        <th className="text-left p-2 font-medium">Duration</th>
                      </tr>
                    </thead>
                    <tbody className="text-sm">
                      <tr className="border-b">
                        <td className="p-2 font-mono">auth_token</td>
                        <td className="p-2">Maintains your login session</td>
                        <td className="p-2"><Badge variant="outline" className="text-xs">Essential</Badge></td>
                        <td className="p-2">30 days</td>
                      </tr>
                      <tr className="border-b">
                        <td className="p-2 font-mono">theme_preference</td>
                        <td className="p-2">Remembers your light/dark theme choice</td>
                        <td className="p-2"><Badge variant="outline" className="text-xs">Functional</Badge></td>
                        <td className="p-2">1 year</td>
                      </tr>
                      <tr className="border-b">
                        <td className="p-2 font-mono">language_setting</td>
                        <td className="p-2">Stores your preferred language</td>
                        <td className="p-2"><Badge variant="outline" className="text-xs">Functional</Badge></td>
                        <td className="p-2">1 year</td>
                      </tr>
                      <tr className="border-b">
                        <td className="p-2 font-mono">cookie_consent</td>
                        <td className="p-2">Remembers your cookie preferences</td>
                        <td className="p-2"><Badge variant="outline" className="text-xs">Essential</Badge></td>
                        <td className="p-2">1 year</td>
                      </tr>
                      <tr className="border-b">
                        <td className="p-2 font-mono">_ga</td>
                        <td className="p-2">Google Analytics - distinguishes users</td>
                        <td className="p-2"><Badge variant="outline" className="text-xs">Analytics</Badge></td>
                        <td className="p-2">2 years</td>
                      </tr>
                      <tr className="border-b">
                        <td className="p-2 font-mono">_gid</td>
                        <td className="p-2">Google Analytics - distinguishes users</td>
                        <td className="p-2"><Badge variant="outline" className="text-xs">Analytics</Badge></td>
                        <td className="p-2">24 hours</td>
                      </tr>
                    </tbody>
                  </table>
                </div>
              </CardContent>
            </Card>

            {/* Third-Party Cookies */}
            <Card>
              <CardHeader>
                <CardTitle>Third-Party Cookies</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <p>
                  We may also use third-party services that set their own cookies. These include:
                </p>
                <ul className="list-disc list-inside space-y-2 text-sm">
                  <li><strong>Google Analytics:</strong> For website analytics and performance measurement</li>
                  <li><strong>Google ReCAPTCHA:</strong> For security and spam protection</li>
                  <li><strong>Social Media Platforms:</strong> For social login and sharing features</li>
                  <li><strong>Payment Processors:</strong> For secure payment processing</li>
                </ul>
              </CardContent>
            </Card>

            {/* Managing Cookies */}
            <Card>
              <CardHeader>
                <CardTitle>Managing Your Cookie Preferences</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <h4 className="font-medium">On SinglesZonne:</h4>
                    <p className="text-sm text-muted-foreground">
                      Use our Cookie Preferences center to manage your settings for optional cookies.
                    </p>
                    <Button 
                      variant="outline" 
                      size="sm"
                      onClick={onOpenCookieConsent}
                      className="flex items-center gap-2"
                    >
                      <Settings className="w-4 h-4" />
                      Manage Preferences
                    </Button>
                  </div>
                  
                  <div className="space-y-2">
                    <h4 className="font-medium">In Your Browser:</h4>
                    <p className="text-sm text-muted-foreground">
                      You can also manage cookies through your browser settings. Note that disabling 
                      certain cookies may affect website functionality.
                    </p>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Local Storage */}
            <Card>
              <CardHeader>
                <CardTitle>Local Storage and Similar Technologies</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <p>
                  In addition to cookies, we may use other storage technologies such as:
                </p>
                <ul className="list-disc list-inside space-y-2 text-sm">
                  <li><strong>Local Storage:</strong> To store user preferences and application state</li>
                  <li><strong>Session Storage:</strong> For temporary data during your browsing session</li>
                  <li><strong>Web Beacons:</strong> Small images used to track email interactions</li>
                </ul>
              </CardContent>
            </Card>

            {/* Updates */}
            <Card>
              <CardHeader>
                <CardTitle>Updates to This Policy</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <p>
                  We may update this Cookie Policy from time to time to reflect changes in our 
                  practices or for other operational, legal, or regulatory reasons. When we do, 
                  we will update the "Last updated" date at the top of this policy.
                </p>
                <p>
                  We encourage you to review this policy periodically to stay informed about 
                  how we use cookies and similar technologies.
                </p>
              </CardContent>
            </Card>

            {/* Contact */}
            <Card>
              <CardHeader>
                <CardTitle>Contact Us</CardTitle>
              </CardHeader>
              <CardContent>
                <p className="mb-4">
                  If you have any questions about our use of cookies or this Cookie Policy, 
                  please contact us:
                </p>
                <div className="bg-muted p-4 rounded-lg text-sm">
                  <p><strong>Email:</strong> privacy@singleszonne.com</p>
                  <p><strong>Address:</strong> SinglesZonne Privacy Team</p>
                  <p>123 Dating Street</p>
                  <p>New York, NY 10001</p>
                  <p>United States</p>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
      
      <Footer onNavigate={onNavigate} onOpenCookieConsent={onOpenCookieConsent} />
    </div>
  );
}