import React, { useState } from 'react';
import { 
  Video, 
  Mic, 
  MapPin, 
  Calendar, 
  Clock, 
  Users, 
  DollarSign, 
  Star,
  Search,
  Filter,
  Heart,
  Award,
  CheckCircle,
  Play
} from 'lucide-react';
import { Button } from './ui/button';
import { Input } from './ui/input';
import { Card } from './ui/card';
import { Badge } from './ui/badge';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription } from './ui/dialog';
import { Tabs, TabsContent, TabsList, TabsTrigger } from './ui/tabs';
import { Avatar, AvatarImage, AvatarFallback } from './ui/avatar';
import { Textarea } from './ui/textarea';
import { Label } from './ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from './ui/select';
import { toast } from 'sonner@2.0.3';

interface CoachSession {
  id: string;
  coachId: string;
  coachName: string;
  coachPhoto: string;
  title: string;
  description: string;
  sessionType: 'video' | 'audio';
  date: string;
  time: string;
  duration: number; // in minutes
  maxParticipants: number;
  currentParticipants: number;
  price: number; // 0 for free
  location: string;
  country: string;
  category: string;
  rating: number;
  reviewCount: number;
  isVerified: boolean;
  tags: string[];
}

interface Coach {
  id: string;
  name: string;
  photo: string;
  bio: string;
  specialties: string[];
  rating: number;
  totalSessions: number;
  reviewCount: number;
  location: string;
  country: string;
  isVerified: boolean;
  yearsExperience: number;
}

// Mock data
const mockSessions: CoachSession[] = [
  {
    id: '1',
    coachId: 'c1',
    coachName: 'Dr. Emily Roberts',
    coachPhoto: 'https://images.unsplash.com/photo-1494790108377-be9c29b29330?w=400',
    title: 'First Date Confidence Masterclass',
    description: 'Learn how to overcome first date anxiety and make a great impression. We\'ll cover conversation starters, body language, and building genuine connections.',
    sessionType: 'video',
    date: '2025-10-25',
    time: '18:00',
    duration: 60,
    maxParticipants: 20,
    currentParticipants: 12,
    price: 25,
    location: 'New York',
    country: 'USA',
    category: 'Confidence Building',
    rating: 4.8,
    reviewCount: 156,
    isVerified: true,
    tags: ['First Dates', 'Confidence', 'Communication']
  },
  {
    id: '2',
    coachId: 'c2',
    coachName: 'Marcus Johnson',
    coachPhoto: 'https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?w=400',
    title: 'Online Dating Profile Optimization',
    description: 'Transform your dating profile into a magnet! Get personalized feedback and learn proven strategies to stand out.',
    sessionType: 'video',
    date: '2025-10-23',
    time: '19:30',
    duration: 90,
    maxParticipants: 15,
    currentParticipants: 8,
    price: 0,
    location: 'London',
    country: 'UK',
    category: 'Profile Tips',
    rating: 4.9,
    reviewCount: 203,
    isVerified: true,
    tags: ['Profile', 'Photos', 'Bio Writing']
  },
  {
    id: '3',
    coachId: 'c3',
    coachName: 'Sofia Martinez',
    coachPhoto: 'https://images.unsplash.com/photo-1438761681033-6461ffad8d80?w=400',
    title: 'Building Emotional Intelligence in Relationships',
    description: 'Audio session focused on developing emotional awareness and communication skills for deeper connections.',
    sessionType: 'audio',
    date: '2025-10-24',
    time: '20:00',
    duration: 45,
    maxParticipants: 30,
    currentParticipants: 22,
    price: 15,
    location: 'Madrid',
    country: 'Spain',
    category: 'Emotional Intelligence',
    rating: 4.7,
    reviewCount: 89,
    isVerified: true,
    tags: ['Emotions', 'Communication', 'Self-Awareness']
  },
  {
    id: '4',
    coachId: 'c4',
    coachName: 'David Chen',
    coachPhoto: 'https://images.unsplash.com/photo-1500648767791-00dcc994a43e?w=400',
    title: 'Red Flags & Green Flags: What to Look For',
    description: 'Learn to identify healthy relationship patterns and warning signs early on. Interactive Q&A session included.',
    sessionType: 'video',
    date: '2025-10-26',
    time: '17:00',
    duration: 75,
    maxParticipants: 25,
    currentParticipants: 18,
    price: 30,
    location: 'Singapore',
    country: 'Singapore',
    category: 'Relationship Advice',
    rating: 4.9,
    reviewCount: 178,
    isVerified: true,
    tags: ['Red Flags', 'Dating Safety', 'Compatibility']
  }
];

const mockCoaches: Coach[] = [
  {
    id: 'c1',
    name: 'Dr. Emily Roberts',
    photo: 'https://images.unsplash.com/photo-1494790108377-be9c29b29330?w=400',
    bio: 'Licensed therapist and dating coach with over 10 years of experience helping singles build confidence and find meaningful connections.',
    specialties: ['Confidence Building', 'First Dates', 'Communication Skills'],
    rating: 4.8,
    totalSessions: 234,
    reviewCount: 156,
    location: 'New York',
    country: 'USA',
    isVerified: true,
    yearsExperience: 10
  },
  {
    id: 'c2',
    name: 'Marcus Johnson',
    photo: 'https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?w=400',
    bio: 'Former matchmaker turned dating coach. Specializing in online dating strategies and profile optimization.',
    specialties: ['Profile Tips', 'Online Dating', 'First Messages'],
    rating: 4.9,
    totalSessions: 312,
    reviewCount: 203,
    location: 'London',
    country: 'UK',
    isVerified: true,
    yearsExperience: 8
  }
];

const categories = [
  'All Categories',
  'Confidence Building',
  'Profile Tips',
  'First Dates',
  'Communication',
  'Emotional Intelligence',
  'Relationship Advice',
  'Dating Safety'
];

export default function DatingCoachesPage() {
  const [activeTab, setActiveTab] = useState<'sessions' | 'coaches'>('sessions');
  const [searchQuery, setSearchQuery] = useState('');
  const [selectedCategory, setSelectedCategory] = useState('All Categories');
  const [sessionType, setSessionType] = useState<'all' | 'video' | 'audio'>('all');
  const [priceFilter, setPriceFilter] = useState<'all' | 'free' | 'paid'>('all');
  const [selectedSession, setSelectedSession] = useState<CoachSession | null>(null);
  const [selectedCoach, setSelectedCoach] = useState<Coach | null>(null);
  const [showRSVPModal, setShowRSVPModal] = useState(false);
  const [showApplicationModal, setShowApplicationModal] = useState(false);
  const [applicationForm, setApplicationForm] = useState({
    fullName: '',
    email: '',
    phone: '',
    location: '',
    country: '',
    yearsExperience: '',
    specialties: '',
    bio: '',
    certifications: '',
    why: ''
  });

  const filteredSessions = mockSessions.filter(session => {
    const matchesSearch = session.title.toLowerCase().includes(searchQuery.toLowerCase()) ||
                         session.coachName.toLowerCase().includes(searchQuery.toLowerCase()) ||
                         session.location.toLowerCase().includes(searchQuery.toLowerCase());
    const matchesCategory = selectedCategory === 'All Categories' || session.category === selectedCategory;
    const matchesType = sessionType === 'all' || session.sessionType === sessionType;
    const matchesPrice = priceFilter === 'all' || 
                        (priceFilter === 'free' && session.price === 0) ||
                        (priceFilter === 'paid' && session.price > 0);
    
    return matchesSearch && matchesCategory && matchesType && matchesPrice;
  });

  const filteredCoaches = mockCoaches.filter(coach => {
    const matchesSearch = coach.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
                         coach.location.toLowerCase().includes(searchQuery.toLowerCase()) ||
                         coach.specialties.some(s => s.toLowerCase().includes(searchQuery.toLowerCase()));
    
    return matchesSearch;
  });

  const handleRSVP = (session: CoachSession) => {
    setSelectedSession(session);
    setShowRSVPModal(true);
  };

  const confirmRSVP = () => {
    if (selectedSession) {
      if (selectedSession.price > 0) {
        // In real app, process payment
        toast.success(`Payment of $${selectedSession.price} processed. You're registered for "${selectedSession.title}"!`);
      } else {
        toast.success(`You're registered for "${selectedSession.title}"!`);
      }
      setShowRSVPModal(false);
      setSelectedSession(null);
    }
  };

  const handleApplicationSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    // In real app, send to admin for approval
    toast.success('Your application has been submitted! Our team will review it within 2-3 business days.');
    setShowApplicationModal(false);
    setApplicationForm({
      fullName: '',
      email: '',
      phone: '',
      location: '',
      country: '',
      yearsExperience: '',
      specialties: '',
      bio: '',
      certifications: '',
      why: ''
    });
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-pink-50 via-purple-50 to-blue-50 dark:from-gray-900 dark:via-purple-900/20 dark:to-gray-900">
      <div className="max-w-7xl mx-auto px-4 py-8">
        {/* Header */}
        <div className="mb-8">
          <div className="flex items-center justify-between mb-4">
            <div>
              <h1 className="text-pink-600 dark:text-pink-400 mb-2">Dating Coaches</h1>
              <p className="text-gray-600 dark:text-gray-400">
                Learn from certified dating experts and transform your love life
              </p>
            </div>
            <Button 
              onClick={() => setShowApplicationModal(true)}
              className="bg-gradient-to-r from-pink-500 to-purple-500 hover:from-pink-600 hover:to-purple-600"
            >
              <Award className="w-4 h-4 mr-2" />
              Become a Coach
            </Button>
          </div>

          {/* Stats Bar */}
          <div className="grid grid-cols-4 gap-4 mb-6">
            <Card className="p-4 bg-white/80 dark:bg-gray-800/80 backdrop-blur-sm">
              <div className="flex items-center gap-3">
                <div className="w-10 h-10 rounded-full bg-pink-100 dark:bg-pink-900/30 flex items-center justify-center">
                  <Users className="w-5 h-5 text-pink-600 dark:text-pink-400" />
                </div>
                <div>
                  <div className="text-gray-500 dark:text-gray-400 text-sm">Active Coaches</div>
                  <div>127</div>
                </div>
              </div>
            </Card>
            <Card className="p-4 bg-white/80 dark:bg-gray-800/80 backdrop-blur-sm">
              <div className="flex items-center gap-3">
                <div className="w-10 h-10 rounded-full bg-purple-100 dark:bg-purple-900/30 flex items-center justify-center">
                  <Calendar className="w-5 h-5 text-purple-600 dark:text-purple-400" />
                </div>
                <div>
                  <div className="text-gray-500 dark:text-gray-400 text-sm">Upcoming Sessions</div>
                  <div>248</div>
                </div>
              </div>
            </Card>
            <Card className="p-4 bg-white/80 dark:bg-gray-800/80 backdrop-blur-sm">
              <div className="flex items-center gap-3">
                <div className="w-10 h-10 rounded-full bg-blue-100 dark:bg-blue-900/30 flex items-center justify-center">
                  <Star className="w-5 h-5 text-blue-600 dark:text-blue-400" />
                </div>
                <div>
                  <div className="text-gray-500 dark:text-gray-400 text-sm">Avg Rating</div>
                  <div>4.8</div>
                </div>
              </div>
            </Card>
            <Card className="p-4 bg-white/80 dark:bg-gray-800/80 backdrop-blur-sm">
              <div className="flex items-center gap-3">
                <div className="w-10 h-10 rounded-full bg-green-100 dark:bg-green-900/30 flex items-center justify-center">
                  <Heart className="w-5 h-5 text-green-600 dark:text-green-400" />
                </div>
                <div>
                  <div className="text-gray-500 dark:text-gray-400 text-sm">Success Stories</div>
                  <div>1,834</div>
                </div>
              </div>
            </Card>
          </div>

          {/* Search and Filters */}
          <Card className="p-6 bg-white/80 dark:bg-gray-800/80 backdrop-blur-sm">
            <div className="grid grid-cols-1 md:grid-cols-5 gap-4">
              <div className="md:col-span-2">
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
                  <Input
                    placeholder="Search by topic, coach, or location..."
                    value={searchQuery}
                    onChange={(e) => setSearchQuery(e.target.value)}
                    className="pl-10"
                  />
                </div>
              </div>
              <Select value={selectedCategory} onValueChange={setSelectedCategory}>
                <SelectTrigger>
                  <SelectValue placeholder="Category" />
                </SelectTrigger>
                <SelectContent>
                  {categories.map(cat => (
                    <SelectItem key={cat} value={cat}>{cat}</SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <Select value={sessionType} onValueChange={(v: any) => setSessionType(v)}>
                <SelectTrigger>
                  <SelectValue placeholder="Session Type" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Types</SelectItem>
                  <SelectItem value="video">Video Only</SelectItem>
                  <SelectItem value="audio">Audio Only</SelectItem>
                </SelectContent>
              </Select>
              <Select value={priceFilter} onValueChange={(v: any) => setPriceFilter(v)}>
                <SelectTrigger>
                  <SelectValue placeholder="Price" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Sessions</SelectItem>
                  <SelectItem value="free">Free Only</SelectItem>
                  <SelectItem value="paid">Paid Only</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </Card>
        </div>

        {/* Tabs */}
        <Tabs value={activeTab} onValueChange={(v: any) => setActiveTab(v)} className="space-y-6">
          <TabsList className="grid w-full max-w-md grid-cols-2 bg-white/80 dark:bg-gray-800/80 backdrop-blur-sm">
            <TabsTrigger value="sessions">Coaching Sessions</TabsTrigger>
            <TabsTrigger value="coaches">Browse Coaches</TabsTrigger>
          </TabsList>

          {/* Sessions Tab */}
          <TabsContent value="sessions" className="space-y-4">
            {filteredSessions.length === 0 ? (
              <Card className="p-12 text-center bg-white/80 dark:bg-gray-800/80 backdrop-blur-sm">
                <Calendar className="w-16 h-16 mx-auto mb-4 text-gray-400" />
                <h3 className="mb-2 text-gray-600 dark:text-gray-400">No sessions found</h3>
                <p className="text-gray-500 dark:text-gray-500">Try adjusting your filters</p>
              </Card>
            ) : (
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                {filteredSessions.map(session => (
                  <Card key={session.id} className="overflow-hidden bg-white/80 dark:bg-gray-800/80 backdrop-blur-sm hover:shadow-xl transition-all">
                    <div className="p-6">
                      {/* Coach Info */}
                      <div className="flex items-start gap-4 mb-4">
                        <Avatar className="w-14 h-14 ring-2 ring-pink-200 dark:ring-pink-800">
                          <AvatarImage src={session.coachPhoto} alt={session.coachName} />
                          <AvatarFallback>{session.coachName[0]}</AvatarFallback>
                        </Avatar>
                        <div className="flex-1">
                          <div className="flex items-center gap-2 mb-1">
                            <span className="font-medium">{session.coachName}</span>
                            {session.isVerified && (
                              <CheckCircle className="w-4 h-4 text-blue-500" />
                            )}
                          </div>
                          <div className="flex items-center gap-3 text-sm text-gray-600 dark:text-gray-400">
                            <div className="flex items-center gap-1">
                              <Star className="w-4 h-4 text-yellow-500 fill-yellow-500" />
                              <span>{session.rating}</span>
                              <span className="text-gray-400">({session.reviewCount})</span>
                            </div>
                            <div className="flex items-center gap-1">
                              <MapPin className="w-4 h-4" />
                              <span>{session.location}, {session.country}</span>
                            </div>
                          </div>
                        </div>
                      </div>

                      {/* Session Details */}
                      <div className="mb-4">
                        <h3 className="mb-2">{session.title}</h3>
                        <p className="text-gray-600 dark:text-gray-400 text-sm mb-3">
                          {session.description}
                        </p>
                        
                        {/* Tags */}
                        <div className="flex flex-wrap gap-2 mb-4">
                          {session.tags.map(tag => (
                            <Badge key={tag} variant="secondary" className="text-xs">
                              {tag}
                            </Badge>
                          ))}
                        </div>

                        {/* Session Info Grid */}
                        <div className="grid grid-cols-2 gap-3 p-3 bg-gray-50 dark:bg-gray-900/50 rounded-lg">
                          <div className="flex items-center gap-2 text-sm">
                            {session.sessionType === 'video' ? (
                              <Video className="w-4 h-4 text-purple-500" />
                            ) : (
                              <Mic className="w-4 h-4 text-blue-500" />
                            )}
                            <span className="capitalize">{session.sessionType} Session</span>
                          </div>
                          <div className="flex items-center gap-2 text-sm">
                            <Calendar className="w-4 h-4 text-pink-500" />
                            <span>{new Date(session.date).toLocaleDateString()}</span>
                          </div>
                          <div className="flex items-center gap-2 text-sm">
                            <Clock className="w-4 h-4 text-orange-500" />
                            <span>{session.time} ({session.duration} min)</span>
                          </div>
                          <div className="flex items-center gap-2 text-sm">
                            <Users className="w-4 h-4 text-green-500" />
                            <span>{session.currentParticipants}/{session.maxParticipants} joined</span>
                          </div>
                        </div>
                      </div>

                      {/* Footer */}
                      <div className="flex items-center justify-between pt-4 border-t dark:border-gray-700">
                        <div className="flex items-center gap-2">
                          {session.price === 0 ? (
                            <Badge className="bg-green-500 text-white">Free</Badge>
                          ) : (
                            <div className="flex items-center gap-1">
                              <DollarSign className="w-5 h-5 text-gray-600 dark:text-gray-400" />
                              <span className="text-xl font-semibold">{session.price}</span>
                            </div>
                          )}
                        </div>
                        <Button
                          onClick={() => handleRSVP(session)}
                          disabled={session.currentParticipants >= session.maxParticipants}
                          className="bg-gradient-to-r from-pink-500 to-purple-500 hover:from-pink-600 hover:to-purple-600"
                        >
                          {session.currentParticipants >= session.maxParticipants ? 'Full' : 'Reserve Spot'}
                        </Button>
                      </div>
                    </div>
                  </Card>
                ))}
              </div>
            )}
          </TabsContent>

          {/* Coaches Tab */}
          <TabsContent value="coaches" className="space-y-4">
            {filteredCoaches.length === 0 ? (
              <Card className="p-12 text-center bg-white/80 dark:bg-gray-800/80 backdrop-blur-sm">
                <Users className="w-16 h-16 mx-auto mb-4 text-gray-400" />
                <h3 className="mb-2 text-gray-600 dark:text-gray-400">No coaches found</h3>
                <p className="text-gray-500 dark:text-gray-500">Try adjusting your search</p>
              </Card>
            ) : (
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                {filteredCoaches.map(coach => (
                  <Card 
                    key={coach.id} 
                    className="overflow-hidden bg-white/80 dark:bg-gray-800/80 backdrop-blur-sm hover:shadow-xl transition-all cursor-pointer"
                    onClick={() => setSelectedCoach(coach)}
                  >
                    <div className="p-6">
                      <div className="text-center mb-4">
                        <Avatar className="w-24 h-24 mx-auto mb-3 ring-4 ring-pink-200 dark:ring-pink-800">
                          <AvatarImage src={coach.photo} alt={coach.name} />
                          <AvatarFallback>{coach.name[0]}</AvatarFallback>
                        </Avatar>
                        <div className="flex items-center justify-center gap-2 mb-1">
                          <h3>{coach.name}</h3>
                          {coach.isVerified && (
                            <CheckCircle className="w-5 h-5 text-blue-500" />
                          )}
                        </div>
                        <div className="flex items-center justify-center gap-1 text-sm text-gray-600 dark:text-gray-400 mb-2">
                          <MapPin className="w-4 h-4" />
                          <span>{coach.location}, {coach.country}</span>
                        </div>
                        <div className="flex items-center justify-center gap-4 text-sm">
                          <div className="flex items-center gap-1">
                            <Star className="w-4 h-4 text-yellow-500 fill-yellow-500" />
                            <span>{coach.rating}</span>
                            <span className="text-gray-400">({coach.reviewCount})</span>
                          </div>
                          <div>
                            <span className="font-medium">{coach.totalSessions}</span>
                            <span className="text-gray-400 ml-1">sessions</span>
                          </div>
                        </div>
                      </div>

                      <p className="text-sm text-gray-600 dark:text-gray-400 mb-4 text-center">
                        {coach.bio}
                      </p>

                      <div className="space-y-3">
                        <div>
                          <div className="text-sm text-gray-500 dark:text-gray-500 mb-2">Specialties</div>
                          <div className="flex flex-wrap gap-2">
                            {coach.specialties.map(specialty => (
                              <Badge key={specialty} variant="secondary" className="text-xs">
                                {specialty}
                              </Badge>
                            ))}
                          </div>
                        </div>
                        <div className="flex items-center gap-2 text-sm text-gray-600 dark:text-gray-400">
                          <Award className="w-4 h-4" />
                          <span>{coach.yearsExperience} years experience</span>
                        </div>
                      </div>

                      <Button 
                        className="w-full mt-4 bg-gradient-to-r from-pink-500 to-purple-500 hover:from-pink-600 hover:to-purple-600"
                        onClick={(e) => {
                          e.stopPropagation();
                          setSelectedCoach(coach);
                        }}
                      >
                        View Sessions
                      </Button>
                    </div>
                  </Card>
                ))}
              </div>
            )}
          </TabsContent>
        </Tabs>
      </div>

      {/* RSVP Modal */}
      <Dialog open={showRSVPModal} onOpenChange={setShowRSVPModal}>
        <DialogContent className="max-w-lg">
          <DialogHeader>
            <DialogTitle>Reserve Your Spot</DialogTitle>
            <DialogDescription>
              Confirm your registration for this coaching session
            </DialogDescription>
          </DialogHeader>
          {selectedSession && (
            <div className="space-y-4">
              <div className="flex items-start gap-4 p-4 bg-gray-50 dark:bg-gray-900/50 rounded-lg">
                <Avatar className="w-12 h-12">
                  <AvatarImage src={selectedSession.coachPhoto} alt={selectedSession.coachName} />
                  <AvatarFallback>{selectedSession.coachName[0]}</AvatarFallback>
                </Avatar>
                <div className="flex-1">
                  <div className="font-medium mb-1">{selectedSession.title}</div>
                  <div className="text-sm text-gray-600 dark:text-gray-400">{selectedSession.coachName}</div>
                  <div className="flex items-center gap-4 mt-2 text-sm">
                    <div className="flex items-center gap-1">
                      <Calendar className="w-4 h-4" />
                      <span>{new Date(selectedSession.date).toLocaleDateString()}</span>
                    </div>
                    <div className="flex items-center gap-1">
                      <Clock className="w-4 h-4" />
                      <span>{selectedSession.time}</span>
                    </div>
                  </div>
                </div>
              </div>

              {selectedSession.price > 0 && (
                <div className="p-4 bg-blue-50 dark:bg-blue-900/20 rounded-lg">
                  <div className="flex items-center justify-between mb-2">
                    <span className="text-gray-600 dark:text-gray-400">Session Fee</span>
                    <span className="text-2xl font-semibold">${selectedSession.price}</span>
                  </div>
                  <p className="text-sm text-gray-600 dark:text-gray-400">
                    Payment will be processed securely. You'll receive a confirmation email with the session link.
                  </p>
                </div>
              )}

              <div className="bg-yellow-50 dark:bg-yellow-900/20 p-4 rounded-lg">
                <p className="text-sm text-yellow-800 dark:text-yellow-200">
                  <strong>Important:</strong> You'll receive the video/audio link 30 minutes before the session starts. Please ensure you have a stable internet connection.
                </p>
              </div>

              <div className="flex gap-3">
                <Button
                  variant="outline"
                  onClick={() => setShowRSVPModal(false)}
                  className="flex-1"
                >
                  Cancel
                </Button>
                <Button
                  onClick={confirmRSVP}
                  className="flex-1 bg-gradient-to-r from-pink-500 to-purple-500 hover:from-pink-600 hover:to-purple-600"
                >
                  {selectedSession.price > 0 ? `Pay $${selectedSession.price} & Register` : 'Register Free'}
                </Button>
              </div>
            </div>
          )}
        </DialogContent>
      </Dialog>

      {/* Coach Application Modal */}
      <Dialog open={showApplicationModal} onOpenChange={setShowApplicationModal}>
        <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>Apply to Become a Dating Coach</DialogTitle>
            <DialogDescription>
              Share your expertise and help singles find love. Our team reviews all applications within 2-3 business days.
            </DialogDescription>
          </DialogHeader>
          <form onSubmit={handleApplicationSubmit} className="space-y-4">
            <div className="grid grid-cols-2 gap-4">
              <div>
                <Label htmlFor="fullName">Full Name *</Label>
                <Input
                  id="fullName"
                  required
                  value={applicationForm.fullName}
                  onChange={(e) => setApplicationForm({...applicationForm, fullName: e.target.value})}
                />
              </div>
              <div>
                <Label htmlFor="email">Email *</Label>
                <Input
                  id="email"
                  type="email"
                  required
                  value={applicationForm.email}
                  onChange={(e) => setApplicationForm({...applicationForm, email: e.target.value})}
                />
              </div>
            </div>

            <div className="grid grid-cols-2 gap-4">
              <div>
                <Label htmlFor="phone">Phone Number *</Label>
                <Input
                  id="phone"
                  required
                  value={applicationForm.phone}
                  onChange={(e) => setApplicationForm({...applicationForm, phone: e.target.value})}
                />
              </div>
              <div>
                <Label htmlFor="yearsExperience">Years of Experience *</Label>
                <Input
                  id="yearsExperience"
                  type="number"
                  required
                  value={applicationForm.yearsExperience}
                  onChange={(e) => setApplicationForm({...applicationForm, yearsExperience: e.target.value})}
                />
              </div>
            </div>

            <div className="grid grid-cols-2 gap-4">
              <div>
                <Label htmlFor="location">City/Location *</Label>
                <Input
                  id="location"
                  required
                  value={applicationForm.location}
                  onChange={(e) => setApplicationForm({...applicationForm, location: e.target.value})}
                />
              </div>
              <div>
                <Label htmlFor="country">Country *</Label>
                <Input
                  id="country"
                  required
                  value={applicationForm.country}
                  onChange={(e) => setApplicationForm({...applicationForm, country: e.target.value})}
                />
              </div>
            </div>

            <div>
              <Label htmlFor="specialties">Areas of Expertise *</Label>
              <Input
                id="specialties"
                placeholder="e.g., First Dates, Communication, Profile Optimization"
                required
                value={applicationForm.specialties}
                onChange={(e) => setApplicationForm({...applicationForm, specialties: e.target.value})}
              />
              <p className="text-sm text-gray-500 dark:text-gray-500 mt-1">Separate multiple areas with commas</p>
            </div>

            <div>
              <Label htmlFor="certifications">Certifications & Credentials</Label>
              <Textarea
                id="certifications"
                placeholder="List any relevant certifications, degrees, or credentials"
                rows={3}
                value={applicationForm.certifications}
                onChange={(e) => setApplicationForm({...applicationForm, certifications: e.target.value})}
              />
            </div>

            <div>
              <Label htmlFor="bio">Professional Bio *</Label>
              <Textarea
                id="bio"
                placeholder="Tell us about your background and approach to dating coaching (min 100 characters)"
                rows={4}
                required
                value={applicationForm.bio}
                onChange={(e) => setApplicationForm({...applicationForm, bio: e.target.value})}
              />
            </div>

            <div>
              <Label htmlFor="why">Why do you want to be a dating coach on SinglesZonne? *</Label>
              <Textarea
                id="why"
                placeholder="Share your motivation and what makes you unique"
                rows={4}
                required
                value={applicationForm.why}
                onChange={(e) => setApplicationForm({...applicationForm, why: e.target.value})}
              />
            </div>

            <div className="bg-blue-50 dark:bg-blue-900/20 p-4 rounded-lg">
              <h4 className="font-medium mb-2">What happens next?</h4>
              <ul className="text-sm text-gray-600 dark:text-gray-400 space-y-1">
                <li>• Our team reviews your application (2-3 business days)</li>
                <li>• If approved, you'll receive an email with setup instructions</li>
                <li>• Complete your coach profile and schedule your first session</li>
                <li>• Start helping singles find love!</li>
              </ul>
            </div>

            <div className="flex gap-3">
              <Button
                type="button"
                variant="outline"
                onClick={() => setShowApplicationModal(false)}
                className="flex-1"
              >
                Cancel
              </Button>
              <Button
                type="submit"
                className="flex-1 bg-gradient-to-r from-pink-500 to-purple-500 hover:from-pink-600 hover:to-purple-600"
              >
                Submit Application
              </Button>
            </div>
          </form>
        </DialogContent>
      </Dialog>

      {/* Coach Details Modal */}
      <Dialog open={selectedCoach !== null} onOpenChange={() => setSelectedCoach(null)}>
        <DialogContent className="max-w-3xl max-h-[90vh] overflow-y-auto">
          {selectedCoach && (
            <div className="space-y-6">
              <div className="flex items-start gap-6">
                <Avatar className="w-24 h-24 ring-4 ring-pink-200 dark:ring-pink-800">
                  <AvatarImage src={selectedCoach.photo} alt={selectedCoach.name} />
                  <AvatarFallback>{selectedCoach.name[0]}</AvatarFallback>
                </Avatar>
                <div className="flex-1">
                  <div className="flex items-center gap-2 mb-2">
                    <h2>{selectedCoach.name}</h2>
                    {selectedCoach.isVerified && (
                      <CheckCircle className="w-6 h-6 text-blue-500" />
                    )}
                  </div>
                  <div className="flex items-center gap-4 mb-3">
                    <div className="flex items-center gap-1">
                      <Star className="w-5 h-5 text-yellow-500 fill-yellow-500" />
                      <span className="font-medium">{selectedCoach.rating}</span>
                      <span className="text-gray-400">({selectedCoach.reviewCount} reviews)</span>
                    </div>
                    <div className="flex items-center gap-1 text-gray-600 dark:text-gray-400">
                      <MapPin className="w-4 h-4" />
                      <span>{selectedCoach.location}, {selectedCoach.country}</span>
                    </div>
                  </div>
                  <div className="flex items-center gap-6 text-sm">
                    <div>
                      <span className="text-gray-500 dark:text-gray-500">Sessions: </span>
                      <span className="font-medium">{selectedCoach.totalSessions}</span>
                    </div>
                    <div>
                      <span className="text-gray-500 dark:text-gray-500">Experience: </span>
                      <span className="font-medium">{selectedCoach.yearsExperience} years</span>
                    </div>
                  </div>
                </div>
              </div>

              <div>
                <h3 className="mb-2">About</h3>
                <p className="text-gray-600 dark:text-gray-400">{selectedCoach.bio}</p>
              </div>

              <div>
                <h3 className="mb-3">Specialties</h3>
                <div className="flex flex-wrap gap-2">
                  {selectedCoach.specialties.map(specialty => (
                    <Badge key={specialty} className="bg-gradient-to-r from-pink-500 to-purple-500 text-white">
                      {specialty}
                    </Badge>
                  ))}
                </div>
              </div>

              <div>
                <h3 className="mb-3">Upcoming Sessions</h3>
                <div className="space-y-3">
                  {mockSessions
                    .filter(s => s.coachId === selectedCoach.id)
                    .map(session => (
                      <Card key={session.id} className="p-4 bg-gray-50 dark:bg-gray-900/50">
                        <div className="flex items-start justify-between">
                          <div className="flex-1">
                            <h4 className="mb-2">{session.title}</h4>
                            <div className="flex items-center gap-4 text-sm text-gray-600 dark:text-gray-400">
                              <div className="flex items-center gap-1">
                                <Calendar className="w-4 h-4" />
                                <span>{new Date(session.date).toLocaleDateString()}</span>
                              </div>
                              <div className="flex items-center gap-1">
                                <Clock className="w-4 h-4" />
                                <span>{session.time}</span>
                              </div>
                              <div className="flex items-center gap-1">
                                <Users className="w-4 h-4" />
                                <span>{session.currentParticipants}/{session.maxParticipants}</span>
                              </div>
                            </div>
                          </div>
                          <Button
                            size="sm"
                            onClick={() => {
                              setSelectedCoach(null);
                              handleRSVP(session);
                            }}
                            className="bg-gradient-to-r from-pink-500 to-purple-500 hover:from-pink-600 hover:to-purple-600"
                          >
                            {session.price === 0 ? 'Join Free' : `$${session.price}`}
                          </Button>
                        </div>
                      </Card>
                    ))}
                </div>
              </div>
            </div>
          )}
        </DialogContent>
      </Dialog>
    </div>
  );
}
