import { useState } from "react";
import { Header } from "./Header";
import { Footer } from "./Footer";
import { Card } from "./ui/card";
import { Badge } from "./ui/badge";
import { Button } from "./ui/button";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "./ui/tabs";
import { 
  Heart, 
  Shield, 
  Users, 
  AlertTriangle, 
  CheckCircle, 
  Lock, 
  Eye, 
  MessageCircle,
  Phone,
  Mail,
  FileText,
  UserCheck,
  Ban,
  Flag,
  Trophy,
  Award,
  Gamepad2
} from "lucide-react";
import { motion } from "motion/react";
import { ImageWithFallback } from "./figma/ImageWithFallback";
import { ContestsPage } from "./ContestsPage";
import DatingCoachesPage from "./DatingCoachesPage";
import { DatingGamesPage } from "./DatingGamesPage";

interface DiscoverPageProps {
  onNavigate: (view: string) => void;
  onOpenCookieConsent?: () => void;
}

export function DiscoverPage({ onNavigate, onOpenCookieConsent }: DiscoverPageProps) {
  const [activeTab, setActiveTab] = useState("contests");

  const helpCenterItems = [
    {
      title: "Getting Started",
      description: "Learn the basics of using Hotzonne to find your perfect match",
      icon: Heart,
      topics: ["Creating your profile", "Uploading photos", "Setting preferences", "Understanding matches"]
    },
    {
      title: "Profile & Photos",
      description: "Tips for creating an attractive and authentic profile",
      icon: UserCheck,
      topics: ["Writing a great bio", "Choosing the best photos", "Profile verification", "Privacy settings"]
    },
    {
      title: "Messaging & Communication",
      description: "How to start conversations and connect with matches",
      icon: MessageCircle,
      topics: ["Starting conversations", "Message etiquette", "Video chat features", "Blocking users"]
    },
    {
      title: "Account & Billing",
      description: "Manage your subscription and account settings",
      icon: FileText,
      topics: ["Subscription plans", "Payment methods", "Canceling subscription", "Account deletion"]
    }
  ];

  const safetyTips = [
    {
      title: "Protect Your Personal Information",
      description: "Never share sensitive details like your home address, workplace, or financial information until you've built trust.",
      icon: Lock,
      tips: [
        "Use Hotzonne's messaging system initially",
        "Don't share your full name too early",
        "Avoid giving out phone numbers immediately",
        "Be cautious about social media connections"
      ]
    },
    {
      title: "Meet Safely",
      description: "When you're ready to meet in person, choose public places and let someone know your plans.",
      icon: Shield,
      tips: [
        "Meet in busy, public locations",
        "Tell a friend about your date plans",
        "Drive yourself or use your own transportation",
        "Trust your instincts if something feels wrong"
      ]
    },
    {
      title: "Recognize Red Flags",
      description: "Be aware of warning signs that might indicate someone isn't who they claim to be.",
      icon: AlertTriangle,
      tips: [
        "They refuse to video chat or meet",
        "They ask for money or financial help",
        "Their photos seem too professional",
        "They push for personal information quickly"
      ]
    },
    {
      title: "Report Suspicious Activity",
      description: "Help keep our community safe by reporting inappropriate behavior or suspicious profiles.",
      icon: Flag,
      tips: [
        "Report fake profiles immediately",
        "Block users who make you uncomfortable",
        "Report harassment or inappropriate messages",
        "Contact support for serious safety concerns"
      ]
    }
  ];

  const communityGuidelines = [
    {
      title: "Be Authentic",
      description: "Use real, recent photos and honest information about yourself",
      icon: UserCheck,
      rules: [
        "Upload genuine photos of yourself",
        "Provide accurate age and location",
        "Be honest about your intentions",
        "Don't impersonate others"
      ]
    },
    {
      title: "Respect Others",
      description: "Treat all members with kindness and respect",
      icon: Heart,
      rules: [
        "Be polite in all communications",
        "Respect others' boundaries and decisions",
        "Don't send inappropriate content",
        "Accept rejection gracefully"
      ]
    },
    {
      title: "Prohibited Content",
      description: "Content that violates our standards will result in account suspension",
      icon: Ban,
      rules: [
        "No nudity or sexually explicit content",
        "No harassment or hate speech",
        "No spam or promotional content",
        "No illegal activities or content"
      ]
    },
    {
      title: "Privacy & Safety",
      description: "Protect yourself and others in the community",
      icon: Eye,
      rules: [
        "Don't share others' personal information",
        "Report suspicious or harmful behavior",
        "Use our platform's safety features",
        "Follow local laws and regulations"
      ]
    }
  ];

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <Header currentView="discover" onNavigate={onNavigate} />
      
      {/* Discover Banner */}
      <div className="bg-gradient-to-r from-purple-600 via-purple-700 to-purple-800 dark:from-purple-700 dark:via-purple-800 dark:to-purple-900 text-white py-11 relative overflow-hidden">
        {/* Moving gradient overlay */}
        <motion.div 
          className="absolute inset-0 bg-gradient-to-r from-transparent via-purple-300/30 to-transparent opacity-40"
          animate={{
            x: ['-100%', '100%']
          }}
          transition={{
            duration: 8,
            repeat: Infinity,
            ease: "linear",
            repeatDelay: 2
          }}
          style={{
            width: '200%',
            background: 'linear-gradient(90deg, transparent 0%, rgba(255,255,255,0.15) 25%, rgba(255,255,255,0.3) 50%, rgba(255,255,255,0.15) 75%, transparent 100%)'
          }}
        />
        
        {/* Additional gradient overlay */}
        <div className="absolute inset-0 bg-gradient-to-r from-transparent via-purple-600/20 to-purple-800/40 dark:from-transparent dark:via-purple-700/20 dark:to-purple-900/40"></div>
        
        <div className="max-w-7xl mx-auto px-6 text-center relative z-10">
          <motion.h1 
            className="text-4xl font-bold mb-4"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ 
              duration: 0.8,
              ease: "easeOut"
            }}
          >
            <motion.span
              animate={{ 
                scale: [1, 1.05, 1],
              }}
              transition={{
                duration: 3,
                repeat: Infinity,
                ease: "easeInOut",
                repeatDelay: 0.5
              }}
            >
              💜 Discover & Learn
            </motion.span>
          </motion.h1>
          <motion.p 
            className="text-lg text-purple-100 dark:text-purple-200"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ 
              duration: 0.8,
              delay: 0.2,
              ease: "easeOut"
            }}
          >
            Everything you need to know for a safe and successful dating experience
          </motion.p>
        </div>
      </div>

      {/* Main Content */}
      <div className="max-w-6xl mx-auto px-6 py-12">
        <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
          {/* Horizontal Tabs */}
          <TabsList className="grid w-full grid-cols-6 mb-8 h-auto p-1">
            <TabsTrigger 
              value="contests" 
              className="flex items-center gap-2 px-4 py-3 data-[state=active]:bg-purple-600 data-[state=active]:text-white"
            >
              <Trophy className="w-4 h-4" />
              <span className="hidden sm:inline">Contests</span>
            </TabsTrigger>
            <TabsTrigger 
              value="dating-coaches" 
              className="flex items-center gap-2 px-4 py-3 data-[state=active]:bg-purple-600 data-[state=active]:text-white"
            >
              <Award className="w-4 h-4" />
              <span className="hidden sm:inline">Dating Coaches</span>
            </TabsTrigger>
            <TabsTrigger 
              value="dating-games" 
              className="flex items-center gap-2 px-4 py-3 data-[state=active]:bg-purple-600 data-[state=active]:text-white"
            >
              <Gamepad2 className="w-4 h-4" />
              <span className="hidden sm:inline">Dating Games</span>
            </TabsTrigger>
            <TabsTrigger 
              value="help-center" 
              className="flex items-center gap-2 px-4 py-3 data-[state=active]:bg-purple-600 data-[state=active]:text-white"
            >
              <Heart className="w-4 h-4" />
              <span className="hidden sm:inline">Help Center</span>
            </TabsTrigger>
            <TabsTrigger 
              value="safety-tips" 
              className="flex items-center gap-2 px-4 py-3 data-[state=active]:bg-purple-600 data-[state=active]:text-white"
            >
              <Shield className="w-4 h-4" />
              <span className="hidden sm:inline">Safety Tips</span>
            </TabsTrigger>
            <TabsTrigger 
              value="community-guidelines" 
              className="flex items-center gap-2 px-4 py-3 data-[state=active]:bg-purple-600 data-[state=active]:text-white"
            >
              <Users className="w-4 h-4" />
              <span className="hidden sm:inline">Community Guidelines</span>
            </TabsTrigger>
          </TabsList>

          {/* Contests Content */}
          <TabsContent value="contests" className="space-y-6">
            <ContestsPage onNavigate={onNavigate} />
          </TabsContent>

          {/* Dating Coaches Content */}
          <TabsContent value="dating-coaches" className="space-y-6">
            <DatingCoachesPage />
          </TabsContent>

          {/* Dating Games Content */}
          <TabsContent value="dating-games" className="space-y-6">
            <DatingGamesPage />
          </TabsContent>

          {/* Help Center Content */}
          <TabsContent value="help-center" className="space-y-6">
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.5 }}
            >
              <div className="text-center mb-8">
                <h2 className="text-2xl font-bold mb-4">How can we help you?</h2>
                <p className="text-muted-foreground">Find answers to common questions and learn how to make the most of Hotzonne</p>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                {helpCenterItems.map((item, index) => {
                  const Icon = item.icon;
                  return (
                    <motion.div
                      key={item.title}
                      initial={{ opacity: 0, y: 20 }}
                      animate={{ opacity: 1, y: 0 }}
                      transition={{ duration: 0.3, delay: index * 0.1 }}
                    >
                      <Card className="p-6 hover:shadow-lg transition-shadow">
                        <div className="flex items-start gap-4">
                          <div className="bg-purple-100 dark:bg-purple-900/20 p-3 rounded-lg">
                            <Icon className="w-6 h-6 text-purple-600" />
                          </div>
                          <div className="flex-1">
                            <h3 className="font-semibold mb-2">{item.title}</h3>
                            <p className="text-muted-foreground text-sm mb-4">{item.description}</p>
                            <div className="space-y-2">
                              {item.topics.map((topic) => (
                                <div key={topic} className="flex items-center gap-2 text-sm">
                                  <CheckCircle className="w-4 h-4 text-green-600" />
                                  <span>{topic}</span>
                                </div>
                              ))}
                            </div>
                          </div>
                        </div>
                      </Card>
                    </motion.div>
                  );
                })}
              </div>

              {/* Contact Support */}
              <Card className="p-6 bg-gradient-to-r from-purple-50 to-purple-100 dark:from-purple-900/20 dark:to-purple-800/20 mt-8">
                <div className="text-center">
                  <h3 className="font-semibold mb-4">Still need help?</h3>
                  <p className="text-muted-foreground mb-6">Our support team is here to help you with any questions</p>
                  <div className="flex flex-col sm:flex-row gap-4 justify-center">
                    <Button className="bg-purple-600 hover:bg-purple-700">
                      <MessageCircle className="w-4 h-4 mr-2" />
                      Live Chat
                    </Button>
                    <Button variant="outline">
                      <Mail className="w-4 h-4 mr-2" />
                      Email Support
                    </Button>
                  </div>
                </div>
              </Card>
            </motion.div>
          </TabsContent>

          {/* Safety Tips Content */}
          <TabsContent value="safety-tips" className="space-y-6">
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.5 }}
            >
              <div className="text-center mb-8">
                <h2 className="text-2xl font-bold mb-4">Your Safety is Our Priority</h2>
                <p className="text-muted-foreground">Follow these guidelines to ensure a safe and positive dating experience</p>
              </div>

              <div className="space-y-6">
                {safetyTips.map((tip, index) => {
                  const Icon = tip.icon;
                  return (
                    <motion.div
                      key={tip.title}
                      initial={{ opacity: 0, y: 20 }}
                      animate={{ opacity: 1, y: 0 }}
                      transition={{ duration: 0.3, delay: index * 0.1 }}
                    >
                      <Card className="p-6">
                        <div className="flex items-start gap-4">
                          <div className="bg-green-100 dark:bg-green-900/20 p-3 rounded-lg">
                            <Icon className="w-6 h-6 text-green-600" />
                          </div>
                          <div className="flex-1">
                            <h3 className="font-semibold mb-2">{tip.title}</h3>
                            <p className="text-muted-foreground mb-4">{tip.description}</p>
                            <div className="grid grid-cols-1 sm:grid-cols-2 gap-2">
                              {tip.tips.map((item) => (
                                <div key={item} className="flex items-start gap-2 text-sm">
                                  <CheckCircle className="w-4 h-4 text-green-600 mt-0.5 flex-shrink-0" />
                                  <span>{item}</span>
                                </div>
                              ))}
                            </div>
                          </div>
                        </div>
                      </Card>
                    </motion.div>
                  );
                })}
              </div>

              {/* Emergency Contact */}
              <Card className="p-6 bg-gradient-to-r from-red-50 to-red-100 dark:from-red-900/20 dark:to-red-800/20">
                <div className="text-center">
                  <AlertTriangle className="w-8 h-8 text-red-600 mx-auto mb-4" />
                  <h3 className="font-semibold mb-4">In Case of Emergency</h3>
                  <p className="text-muted-foreground mb-6">If you feel unsafe or threatened, contact local authorities immediately</p>
                  <Button className="bg-red-600 hover:bg-red-700">
                    <Phone className="w-4 h-4 mr-2" />
                    Emergency: 911
                  </Button>
                </div>
              </Card>
            </motion.div>
          </TabsContent>

          {/* Community Guidelines Content */}
          <TabsContent value="community-guidelines" className="space-y-6">
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.5 }}
            >
              <div className="text-center mb-8">
                <h2 className="text-2xl font-bold mb-4">Community Guidelines</h2>
                <p className="text-muted-foreground">Help us maintain a respectful and safe environment for everyone</p>
              </div>

              <div className="space-y-6">
                {communityGuidelines.map((guideline, index) => {
                  const Icon = guideline.icon;
                  return (
                    <motion.div
                      key={guideline.title}
                      initial={{ opacity: 0, y: 20 }}
                      animate={{ opacity: 1, y: 0 }}
                      transition={{ duration: 0.3, delay: index * 0.1 }}
                    >
                      <Card className="p-6">
                        <div className="flex items-start gap-4">
                          <div className="bg-blue-100 dark:bg-blue-900/20 p-3 rounded-lg">
                            <Icon className="w-6 h-6 text-blue-600" />
                          </div>
                          <div className="flex-1">
                            <h3 className="font-semibold mb-2">{guideline.title}</h3>
                            <p className="text-muted-foreground mb-4">{guideline.description}</p>
                            <div className="space-y-2">
                              {guideline.rules.map((rule) => (
                                <div key={rule} className="flex items-start gap-2 text-sm">
                                  <CheckCircle className="w-4 h-4 text-blue-600 mt-0.5 flex-shrink-0" />
                                  <span>{rule}</span>
                                </div>
                              ))}
                            </div>
                          </div>
                        </div>
                      </Card>
                    </motion.div>
                  );
                })}
              </div>

              {/* Reporting */}
              <Card className="p-6 bg-gradient-to-r from-yellow-50 to-yellow-100 dark:from-yellow-900/20 dark:to-yellow-800/20">
                <div className="text-center">
                  <Flag className="w-8 h-8 text-yellow-600 mx-auto mb-4" />
                  <h3 className="font-semibold mb-4">Report Violations</h3>
                  <p className="text-muted-foreground mb-6">Help us keep the community safe by reporting any violations of these guidelines</p>
                  <Button className="bg-yellow-600 hover:bg-yellow-700">
                    <Flag className="w-4 h-4 mr-2" />
                    Report a User
                  </Button>
                </div>
              </Card>
            </motion.div>
          </TabsContent>
        </Tabs>
      </div>
      
      {/* Footer */}
      <Footer onNavigate={onNavigate} onOpenCookieConsent={onOpenCookieConsent} />
    </div>
  );
}