import { useState, useEffect } from "react";
import { ImageWithFallback } from "./figma/ImageWithFallback";

interface DynamicBackgroundProps {
  className?: string;
}

const backgroundImages = [
  {
    url: "https://images.unsplash.com/photo-1600472680277-e0660504339b?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxkaXZlcnNlJTIwY291cGxlcyUyMGhhcHB5JTIwaW50ZXJuYXRpb25hbHxlbnwxfHx8fDE3NTgzNzU2MTV8MA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
    alt: "Diverse couples celebrating love internationally",
    overlay: "bg-black/40 dark:bg-black/60"
  },
  {
    url: "https://images.unsplash.com/photo-1708876953466-9d47b2b7cda3?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxyb21hbnRpYyUyMHN1bnNldCUyMHRyYXZlbCUyMGNvdXBsZXxlbnwxfHx8fDE3NTgzNzU2MTh8MA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
    alt: "Romantic sunset travel couple adventure",
    overlay: "bg-black/35 dark:bg-black/55"
  },
  {
    url: "https://images.unsplash.com/photo-1668687524808-ef90e2a62429?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxtdWx0aWN1bHR1cmFsJTIwbG92ZSUyMGNlbGVicmF0aW9ufGVufDF8fHx8MTc1ODM3NTYyMXww&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
    alt: "Multicultural love celebration and joy",
    overlay: "bg-black/30 dark:bg-black/50"
  },
  {
    url: "https://images.unsplash.com/photo-1758146928286-eb515c0c5968?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxiZWF1dGlmdWwlMjB3b3JsZCUyMGRlc3RpbmF0aW9ucyUyMHJvbWFudGljfGVufDF8fHx8MTc1ODM3NTYyM3ww&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
    alt: "Beautiful world destinations for romantic getaways",
    overlay: "bg-black/45 dark:bg-black/65"
  },
  {
    url: "https://images.unsplash.com/photo-1525353297422-7504381dce2d?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxnbG9iYWwlMjBjb25uZWN0aW9ucyUyMGxvdmUlMjBoZWFydHN8ZW58MXx8fHwxNzU4Mzc1NjI3fDA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
    alt: "Global connections and love hearts",
    overlay: "bg-black/40 dark:bg-black/60"
  },
  {
    url: "https://images.unsplash.com/photo-1755810392482-3f1d4ea988f1?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHx3YW5kZXJsdXN0JTIwdHJhdmVsJTIwcm9tYW5jZSUyMGFkdmVudHVyZXxlbnwxfHx8fDE3NTgzNzU2MzB8MA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
    alt: "Wanderlust travel romance and adventure",
    overlay: "bg-black/35 dark:bg-black/55"
  }
];

export function DynamicBackground({ className }: DynamicBackgroundProps) {
  const [currentImageIndex, setCurrentImageIndex] = useState(0);
  const [nextImageIndex, setNextImageIndex] = useState(1);
  const [isTransitioning, setIsTransitioning] = useState(false);

  useEffect(() => {
    // Initial random selection
    const randomIndex = Math.floor(Math.random() * backgroundImages.length);
    setCurrentImageIndex(randomIndex);
    setNextImageIndex((randomIndex + 1) % backgroundImages.length);

    // Set up rotation timer
    const interval = setInterval(() => {
      setIsTransitioning(true);
      
      setTimeout(() => {
        setCurrentImageIndex(nextImageIndex);
        setNextImageIndex((nextImageIndex + 1) % backgroundImages.length);
        setIsTransitioning(false);
      }, 500); // Half second for smooth transition
    }, 8000); // Change image every 8 seconds

    return () => clearInterval(interval);
  }, [nextImageIndex]);

  const currentImage = backgroundImages[currentImageIndex];
  const nextImage = backgroundImages[nextImageIndex];

  return (
    <div className={`relative overflow-hidden ${className}`}>
      {/* Current Background Image */}
      <div 
        className={`absolute inset-0 transition-opacity duration-500 ${
          isTransitioning ? 'opacity-0' : 'opacity-100'
        }`}
      >
        <ImageWithFallback
          src={currentImage.url}
          alt={currentImage.alt}
          className="w-full h-full object-cover"
        />
        <div className={`absolute inset-0 ${currentImage.overlay}`}></div>
      </div>

      {/* Next Background Image (for smooth transition) */}
      <div 
        className={`absolute inset-0 transition-opacity duration-500 ${
          isTransitioning ? 'opacity-100' : 'opacity-0'
        }`}
      >
        <ImageWithFallback
          src={nextImage.url}
          alt={nextImage.alt}
          className="w-full h-full object-cover"
        />
        <div className={`absolute inset-0 ${nextImage.overlay}`}></div>
      </div>

      {/* Subtle animation overlay for visual interest */}
      <div className="absolute inset-0 bg-gradient-to-br from-transparent via-transparent to-black/20 dark:to-black/40"></div>
      
      {/* Romantic bokeh effect overlay */}
      <div className="absolute inset-0 opacity-20">
        <div className="absolute top-1/4 left-1/4 w-32 h-32 bg-pink-300 rounded-full blur-3xl animate-pulse"></div>
        <div className="absolute top-3/4 right-1/4 w-24 h-24 bg-purple-300 rounded-full blur-2xl animate-pulse delay-1000"></div>
        <div className="absolute bottom-1/4 left-1/3 w-20 h-20 bg-red-300 rounded-full blur-2xl animate-pulse delay-2000"></div>
      </div>
    </div>
  );
}