import { useEffect, useRef, useState } from 'react';
import { Button } from './ui/button';
import { MapPin, Plus, Minus, Maximize2, Navigation } from 'lucide-react';
import { GoogleMapsPlaceholder } from './GoogleMapsConfig';
import { GOOGLE_MAPS_CONFIG, shouldEnableMaps } from './GoogleMapsApiConfig';

interface GoogleMapsProps {
  center: {
    lat: number;
    lng: number;
  };
  zoom?: number;
  markers?: Array<{
    position: { lat: number; lng: number };
    title: string;
    info?: string;
  }>;
  className?: string;
  showControls?: boolean;
  onMapClick?: (coordinates: { lat: number; lng: number }) => void;
}

declare global {
  interface Window {
    google: any;
    initMap: () => void;
  }
}

export function GoogleMaps({ 
  center, 
  zoom = 13, 
  markers = [], 
  className = "w-full h-96",
  showControls = true,
  onMapClick 
}: GoogleMapsProps) {
  const mapRef = useRef<HTMLDivElement>(null);
  const [map, setMap] = useState<any>(null);
  const [isLoaded, setIsLoaded] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const markersRef = useRef<any[]>([]);
  const infoWindowRef = useRef<any>(null);

  useEffect(() => {
    // Check if API key is configured
    if (!shouldEnableMaps()) {
      setError(GOOGLE_MAPS_CONFIG.placeholderMessage);
      return;
    }
    loadGoogleMaps();
  }, []);

  useEffect(() => {
    if (map && isLoaded) {
      updateMapData();
    }
  }, [center, zoom, markers, map, isLoaded]);

  const loadGoogleMaps = () => {
    // Check if Google Maps is already loaded
    if (window.google && window.google.maps) {
      setIsLoaded(true);
      initializeMap();
      return;
    }

    // Check if script is already being loaded
    if (document.querySelector('script[src*="maps.googleapis.com"]')) {
      // Wait for it to load
      const checkLoaded = setInterval(() => {
        if (window.google && window.google.maps) {
          clearInterval(checkLoaded);
          setIsLoaded(true);
          initializeMap();
        }
      }, 100);
      return;
    }

    // Load Google Maps script
    const script = document.createElement('script');
    script.src = `https://maps.googleapis.com/maps/api/js?key=${GOOGLE_MAPS_CONFIG.API_KEY}&libraries=${GOOGLE_MAPS_CONFIG.LIBRARIES.join(',')}`;
    script.async = true;
    script.defer = true;
    
    script.onload = () => {
      setIsLoaded(true);
      initializeMap();
    };
    
    script.onerror = () => {
      // Silently handle script loading errors
      setError(GOOGLE_MAPS_CONFIG.placeholderMessage);
    };
    
    document.head.appendChild(script);
  };

  const initializeMap = () => {
    if (!mapRef.current || !window.google) return;

    try {
      const mapInstance = new window.google.maps.Map(mapRef.current, {
        center: center,
        zoom: zoom,
        mapTypeControl: false,
        streetViewControl: false,
        fullscreenControl: false,
        zoomControl: false, // We'll use custom controls
        styles: GOOGLE_MAPS_CONFIG.MAP_STYLES
      });

      // Add click listener
      if (onMapClick) {
        mapInstance.addListener('click', (event: any) => {
          onMapClick({
            lat: event.latLng.lat(),
            lng: event.latLng.lng()
          });
        });
      }

      setMap(mapInstance);
      
      // Create info window
      infoWindowRef.current = new window.google.maps.InfoWindow();
      
    } catch (err) {
      setError('Failed to initialize Google Maps');
      console.error('Google Maps initialization error:', err);
    }
  };

  const updateMapData = () => {
    if (!map || !window.google) return;

    // Clear existing markers
    markersRef.current.forEach(marker => marker.setMap(null));
    markersRef.current = [];

    // Update map center and zoom
    map.setCenter(center);
    map.setZoom(zoom);

    // Add markers
    markers.forEach((markerData) => {
      const marker = new window.google.maps.Marker({
        position: markerData.position,
        map: map,
        title: markerData.title,
        icon: {
          url: 'data:image/svg+xml;charset=UTF-8,' + encodeURIComponent(`
            <svg width="32" height="32" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
              <path d="M12 2C8.13 2 5 5.13 5 9c0 5.25 7 13 7 13s7-7.75 7-13c0-3.87-3.13-7-7-7zm0 9.5c-1.38 0-2.5-1.12-2.5-2.5s1.12-2.5 2.5-2.5 2.5 1.12 2.5 2.5-1.12 2.5-2.5 2.5z" fill="#ef4444"/>
            </svg>
          `),
          scaledSize: new window.google.maps.Size(32, 32),
          anchor: new window.google.maps.Point(16, 32)
        }
      });

      // Add click listener for info window
      if (markerData.info) {
        marker.addListener('click', () => {
          infoWindowRef.current.setContent(`
            <div style="padding: 8px; max-width: 200px;">
              <h3 style="margin: 0 0 8px 0; font-weight: bold;">${markerData.title}</h3>
              <p style="margin: 0; color: #666;">${markerData.info}</p>
            </div>
          `);
          infoWindowRef.current.open(map, marker);
        });
      }

      markersRef.current.push(marker);
    });
  };

  const handleZoomIn = () => {
    if (map) {
      map.setZoom(map.getZoom() + 1);
    }
  };

  const handleZoomOut = () => {
    if (map) {
      map.setZoom(map.getZoom() - 1);
    }
  };

  const handleRecenter = () => {
    if (map) {
      map.setCenter(center);
      map.setZoom(zoom);
    }
  };

  const handleFullscreen = () => {
    if (mapRef.current) {
      if (mapRef.current.requestFullscreen) {
        mapRef.current.requestFullscreen();
      }
    }
  };

  if (error) {
    return <GoogleMapsPlaceholder error={error} />;
  }

  if (!isLoaded) {
    return (
      <div className={`${className} bg-muted flex items-center justify-center rounded-lg border`}>
        <div className="text-center p-6">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary mx-auto mb-4"></div>
          <p className="text-muted-foreground">Loading Map...</p>
        </div>
      </div>
    );
  }

  return (
    <div className={`${className} relative rounded-lg overflow-hidden border`}>
      <div ref={mapRef} className="w-full h-full" />
      
      {/* Custom Map Controls */}
      {showControls && (
        <>
          {/* Zoom Controls */}
          <div className="absolute top-4 right-4 flex flex-col gap-2">
            <Button
              size="sm"
              variant="secondary"
              className="w-8 h-8 p-0 bg-white/95 hover:bg-white shadow-lg"
              onClick={handleZoomIn}
            >
              <Plus className="w-4 h-4" />
            </Button>
            <Button
              size="sm"
              variant="secondary"
              className="w-8 h-8 p-0 bg-white/95 hover:bg-white shadow-lg"
              onClick={handleZoomOut}
            >
              <Minus className="w-4 h-4" />
            </Button>
          </div>

          {/* Additional Controls */}
          <div className="absolute bottom-4 right-4 flex gap-2">
            <Button
              size="sm"
              variant="secondary"
              className="bg-white/95 hover:bg-white shadow-lg"
              onClick={handleRecenter}
            >
              <Navigation className="w-4 h-4 mr-1" />
              Recenter
            </Button>
            <Button
              size="sm"
              variant="secondary"
              className="bg-white/95 hover:bg-white shadow-lg"
              onClick={handleFullscreen}
            >
              <Maximize2 className="w-4 h-4" />
            </Button>
          </div>
        </>
      )}
    </div>
  );
}

// Hook for using Google Maps Places Autocomplete
export function useGooglePlacesAutocomplete(inputRef: React.RefObject<HTMLInputElement>) {
  const [autocomplete, setAutocomplete] = useState<any>(null);
  const [selectedPlace, setSelectedPlace] = useState<any>(null);

  useEffect(() => {
    // Skip if Google Maps is not available or API key not configured
    if (!shouldEnableMaps() || !window.google || !inputRef.current) return;

    try {
      const autocompleteInstance = new window.google.maps.places.Autocomplete(
        inputRef.current,
        {
          types: ['(cities)'],
          fields: ['place_id', 'geometry', 'name', 'formatted_address']
        }
      );

      autocompleteInstance.addListener('place_changed', () => {
        const place = autocompleteInstance.getPlace();
        setSelectedPlace(place);
      });

      setAutocomplete(autocompleteInstance);

      return () => {
        if (autocompleteInstance) {
          window.google.maps.event.clearInstanceListeners(autocompleteInstance);
        }
      };
    } catch (error) {
      // Silently handle any autocomplete initialization errors
      return;
    }
  }, [inputRef]);

  return { autocomplete, selectedPlace };
}

// Utility function to get user's current location
export function getCurrentLocation(): Promise<{ lat: number; lng: number }> {
  return new Promise((resolve, reject) => {
    if (!navigator.geolocation) {
      reject(new Error('Geolocation is not supported by this browser'));
      return;
    }

    navigator.geolocation.getCurrentPosition(
      (position) => {
        resolve({
          lat: position.coords.latitude,
          lng: position.coords.longitude
        });
      },
      (error) => {
        reject(error);
      },
      {
        enableHighAccuracy: true,
        timeout: 10000,
        maximumAge: 600000 // 10 minutes
      }
    );
  });
}

// Utility function to calculate distance between two points
export function calculateDistance(
  point1: { lat: number; lng: number },
  point2: { lat: number; lng: number }
): number {
  if (!window.google || !shouldEnableMaps()) {
    // Fallback calculation using Haversine formula
    const R = 6371; // Earth's radius in km
    const dLat = (point2.lat - point1.lat) * Math.PI / 180;
    const dLng = (point2.lng - point1.lng) * Math.PI / 180;
    const a = Math.sin(dLat/2) * Math.sin(dLat/2) + 
              Math.cos(point1.lat * Math.PI / 180) * Math.cos(point2.lat * Math.PI / 180) *
              Math.sin(dLng/2) * Math.sin(dLng/2);
    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
    return Math.round(R * c);
  }
  
  try {
    const distance = window.google.maps.geometry.spherical.computeDistanceBetween(
      new window.google.maps.LatLng(point1.lat, point1.lng),
      new window.google.maps.LatLng(point2.lat, point2.lng)
    );
    return Math.round(distance / 1000); // Return distance in kilometers
  } catch (error) {
    // Fallback to basic calculation
    return 0;
  }
}