import { useEffect, useState } from 'react';
import { GOOGLE_MAPS_CONFIG, shouldEnableMaps } from './GoogleMapsApiConfig';

interface GoogleMapsLoaderProps {
  children: React.ReactNode;
}

export function GoogleMapsLoader({ children }: GoogleMapsLoaderProps) {
  const [isLoaded, setIsLoaded] = useState(false);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    // Skip loading if API key is not configured - components will show placeholders
    if (!shouldEnableMaps()) {
      setIsLoaded(false);
      return;
    }

    // Check if Google Maps is already loaded
    if (window.google && window.google.maps) {
      setIsLoaded(true);
      return;
    }

    // Check if script is already being loaded
    if (document.querySelector('script[src*="maps.googleapis.com"]')) {
      // Wait for it to load
      const checkLoaded = setInterval(() => {
        if (window.google && window.google.maps) {
          clearInterval(checkLoaded);
          setIsLoaded(true);
        }
      }, 100);

      // Timeout after 10 seconds
      setTimeout(() => {
        clearInterval(checkLoaded);
        if (!window.google || !window.google.maps) {
          setError('Google Maps loading timeout');
        }
      }, 10000);
      
      return;
    }

    // Load Google Maps script
    const script = document.createElement('script');
    script.src = `https://maps.googleapis.com/maps/api/js?key=${GOOGLE_MAPS_CONFIG.API_KEY}&libraries=${GOOGLE_MAPS_CONFIG.LIBRARIES.join(',')}`;
    script.async = true;
    script.defer = true;
    
    script.onload = () => {
      setIsLoaded(true);
    };
    
    script.onerror = () => {
      // Silently handle the error - components will show placeholders
      setError(null);
      setIsLoaded(false);
    };
    
    document.head.appendChild(script);

    // Cleanup function
    return () => {
      // Note: We don't remove the script as it might be needed by other components
    };
  }, []);

  // Always render children - individual map components will handle the missing API key gracefully
  return <>{children}</>;
}

// Environment variable setup instructions component
export function GoogleMapsSetupInstructions() {
  return (
    <div className="bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-800 rounded-lg p-4 m-4">
      <h3 className="font-semibold text-yellow-800 dark:text-yellow-200 mb-2">
        Google Maps Setup Required
      </h3>
      <div className="text-sm text-yellow-700 dark:text-yellow-300 space-y-2">
        <p>To enable Google Maps functionality, you need to:</p>
        <ol className="list-decimal list-inside space-y-1 ml-4">
          <li>Get a Google Maps API key from the Google Cloud Console</li>
          <li>Enable the Maps JavaScript API and Places API</li>
          <li>Add your API key to the environment variables:</li>
        </ol>
        <div className="bg-yellow-100 dark:bg-yellow-800/30 p-2 rounded font-mono text-xs mt-2">
          REACT_APP_GOOGLE_MAPS_API_KEY=your_api_key_here
        </div>
        <p className="text-xs mt-2">
          <strong>Note:</strong> In production, make sure to restrict your API key to your domain for security.
        </p>
      </div>
    </div>
  );
}