import { Card } from './ui/card';
import { Button } from './ui/button';
import { Alert, AlertDescription } from './ui/alert';
import { MapPin, ExternalLink, Copy, CheckCircle } from 'lucide-react';
import { useState } from 'react';

export function GoogleMapsSetupGuide() {
  const [copiedStep, setCopiedStep] = useState<number | null>(null);

  const copyToClipboard = (text: string, step: number) => {
    navigator.clipboard.writeText(text).then(() => {
      setCopiedStep(step);
      setTimeout(() => setCopiedStep(null), 2000);
    });
  };

  const codeExamples = {
    loader: `// In /components/GoogleMapsLoader.tsx (line ~13)
const GOOGLE_MAPS_API_KEY = "AIzaSyBxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx";`,
    
    component: `// In /components/GoogleMaps.tsx (line ~45)  
const GOOGLE_MAPS_API_KEY = "AIzaSyBxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx";`
  };

  return (
    <div className="max-w-4xl mx-auto p-6">
      <Card className="p-6">
        <div className="flex items-center gap-3 mb-6">
          <div className="w-12 h-12 bg-blue-100 dark:bg-blue-900 rounded-lg flex items-center justify-center">
            <MapPin className="w-6 h-6 text-blue-600 dark:text-blue-400" />
          </div>
          <div>
            <h2 className="text-2xl font-bold">Google Maps Setup Guide</h2>
            <p className="text-muted-foreground">Enable interactive maps and location features</p>
          </div>
        </div>

        <Alert className="mb-6">
          <MapPin className="h-4 w-4" />
          <AlertDescription>
            Follow these steps to enable Google Maps functionality in SinglesZonne. This will activate
            location selection, interactive user profiles, and location-based search features.
          </AlertDescription>
        </Alert>

        <div className="space-y-6">
          {/* Step 1 */}
          <div className="border rounded-lg p-4">
            <div className="flex items-start gap-3">
              <div className="w-6 h-6 bg-primary text-primary-foreground rounded-full flex items-center justify-center text-sm font-bold">
                1
              </div>
              <div className="flex-1">
                <h4 className="font-medium mb-2">Get Google Maps API Key</h4>
                <p className="text-sm text-muted-foreground mb-3">
                  Visit the Google Cloud Console to create a new project and API key
                </p>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => window.open('https://console.cloud.google.com/apis/credentials', '_blank')}
                >
                  <ExternalLink className="w-4 h-4 mr-2" />
                  Open Google Cloud Console
                </Button>
              </div>
            </div>
          </div>

          {/* Step 2 */}
          <div className="border rounded-lg p-4">
            <div className="flex items-start gap-3">
              <div className="w-6 h-6 bg-primary text-primary-foreground rounded-full flex items-center justify-center text-sm font-bold">
                2
              </div>
              <div className="flex-1">
                <h4 className="font-medium mb-2">Enable Required APIs</h4>
                <p className="text-sm text-muted-foreground mb-3">
                  In your Google Cloud project, enable these APIs:
                </p>
                <ul className="space-y-1 text-sm text-muted-foreground">
                  <li>• Maps JavaScript API</li>
                  <li>• Places API</li>
                  <li>• Geocoding API</li>
                </ul>
              </div>
            </div>
          </div>

          {/* Step 3 */}
          <div className="border rounded-lg p-4">
            <div className="flex items-start gap-3">
              <div className="w-6 h-6 bg-primary text-primary-foreground rounded-full flex items-center justify-center text-sm font-bold">
                3
              </div>
              <div className="flex-1">
                <h4 className="font-medium mb-2">Update GoogleMapsLoader.tsx</h4>
                <p className="text-sm text-muted-foreground mb-3">
                  Replace the API key placeholder in GoogleMapsLoader.tsx:
                </p>
                <div className="bg-muted p-3 rounded-md relative">
                  <code className="text-sm whitespace-pre-wrap">{codeExamples.loader}</code>
                  <Button
                    variant="ghost"
                    size="sm"
                    className="absolute top-1 right-1"
                    onClick={() => copyToClipboard(codeExamples.loader, 3)}
                  >
                    {copiedStep === 3 ? <CheckCircle className="w-3 h-3" /> : <Copy className="w-3 h-3" />}
                  </Button>
                </div>
              </div>
            </div>
          </div>

          {/* Step 4 */}
          <div className="border rounded-lg p-4">
            <div className="flex items-start gap-3">
              <div className="w-6 h-6 bg-primary text-primary-foreground rounded-full flex items-center justify-center text-sm font-bold">
                4
              </div>
              <div className="flex-1">
                <h4 className="font-medium mb-2">Update GoogleMaps.tsx</h4>
                <p className="text-sm text-muted-foreground mb-3">
                  Replace the API key placeholder in GoogleMaps.tsx:
                </p>
                <div className="bg-muted p-3 rounded-md relative">
                  <code className="text-sm whitespace-pre-wrap">{codeExamples.component}</code>
                  <Button
                    variant="ghost"
                    size="sm"
                    className="absolute top-1 right-1"
                    onClick={() => copyToClipboard(codeExamples.component, 4)}
                  >
                    {copiedStep === 4 ? <CheckCircle className="w-3 h-3" /> : <Copy className="w-3 h-3" />}
                  </Button>
                </div>
              </div>
            </div>
          </div>

          {/* Step 5 */}
          <div className="border rounded-lg p-4">
            <div className="flex items-start gap-3">
              <div className="w-6 h-6 bg-primary text-primary-foreground rounded-full flex items-center justify-center text-sm font-bold">
                5
              </div>
              <div className="flex-1">
                <h4 className="font-medium mb-2">Secure Your API Key (Production)</h4>
                <p className="text-sm text-muted-foreground mb-3">
                  For production deployment, restrict your API key to your domain:
                </p>
                <div className="bg-muted p-3 rounded-md">
                  <code className="text-sm">
                    yourdomain.com/*<br />
                    *.yourdomain.com/*
                  </code>
                </div>
              </div>
            </div>
          </div>
        </div>

        {/* Development Tips */}
        <div className="bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-800 rounded-lg p-4 mt-6">
          <h4 className="font-medium text-yellow-800 dark:text-yellow-200 mb-2">
            💡 Development Tips
          </h4>
          <ul className="space-y-1 text-sm text-yellow-700 dark:text-yellow-300">
            <li>• Start with an unrestricted API key for development</li>
            <li>• Monitor usage in Google Cloud Console</li>
            <li>• Enable billing to avoid quota limits</li>
            <li>• Test with different locations and devices</li>
          </ul>
        </div>

        {/* Features Preview */}
        <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4 mt-4">
          <h4 className="font-medium text-blue-800 dark:text-blue-200 mb-2">
            🎯 Features Enabled After Setup
          </h4>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-2 text-sm text-blue-700 dark:text-blue-300">
            <div>• Interactive user profile maps</div>
            <div>• Location-based member search</div>
            <div>• City and address autocomplete</div>
            <div>• Current location detection</div>
            <div>• Distance calculations</div>
            <div>• Event venue mapping</div>
          </div>
        </div>
      </Card>
    </div>
  );
}