import { Heart, ChevronDown, MessageCircle, Users, Search, Flame, Zap, Megaphone, Crown, Shield, UserCheck, Settings2, Coins, ShoppingCart } from "lucide-react";
import { Button } from "./ui/button";
import { Avatar, AvatarFallback, AvatarImage } from "./ui/avatar";
import { Badge } from "./ui/badge";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "./ui/dropdown-menu";
import { ThemeToggle } from "./ThemeToggle";
import { LanguageSelector } from "./LanguageSelector";
import { useTranslation } from "./TranslationProvider";
import { useUser, useRoleSwitcher } from "./UserProvider";
import { usePoints } from "./PointsProvider";
import { PointsStore } from "./PointsStore";
import { useState } from "react";

interface HeaderProps {
  currentView: string;
  onNavigate: (view: string) => void;
}

export function Header({ currentView, onNavigate }: HeaderProps) {
  const { t } = useTranslation();
  const { user, isRole, getEffectiveSubscriptionPlan } = useUser();
  const { switchToRole } = useRoleSwitcher();
  const { points } = usePoints();
  const [isPointsStoreOpen, setIsPointsStoreOpen] = useState(false);
  
  const effectiveSubscriptionPlan = getEffectiveSubscriptionPlan();
  
  const navigationItems = [
    { id: "search", label: t('nav.search'), icon: Search },
    { id: "messages", label: t('nav.messages'), icon: MessageCircle },
    { id: "groups", label: "Groups & Events", icon: Users },
  ];

  const discoverSubItems = [
    { id: "hot-or-not", label: "Hot or Not", icon: Flame },
    { id: "matches", label: "Matches", icon: Heart },
    { id: "ads", label: "Ads", icon: Megaphone },
  ];

  return (
    <>
      <header className="bg-background border-b border-border px-6 py-[1.15rem] w-full">
        <div className="flex items-center justify-between w-full">
          {/* Logo */}
          <div className="flex items-center gap-2 cursor-pointer" onClick={() => onNavigate("landing")}>
            <Heart className="w-6 h-6 text-purple-600 fill-purple-600" />
            <h1 className="text-xl font-semibold text-foreground">SinglesZonne</h1>
          </div>

          {/* Navigation */}
          <nav className="hidden md:flex items-center gap-8">
            {currentView === "landing" ? (
              // Landing page navigation
              <>
                <Button
                  variant="ghost"
                  onClick={() => onNavigate("search")}
                  className="text-muted-foreground hover:text-foreground"
                >
                  Browse Profiles
                </Button>
                <Button
                  variant="ghost"
                  onClick={() => onNavigate("login")}
                  className="text-muted-foreground hover:text-foreground"
                >
                  Sign In
                </Button>
                <Button
                  onClick={() => onNavigate("register")}
                  className="bg-purple-600 hover:bg-purple-700 text-white"
                >
                  Join Free
                </Button>
              </>
            ) : (
              // Regular app navigation
              <>
            {navigationItems.map((item) => {
              const Icon = item.icon;
              return (
                <button
                  key={item.id}
                  onClick={() => onNavigate(item.id)}
                  className={`flex items-center gap-2 px-3 py-2 rounded-md transition-colors ${
                    currentView === item.id
                      ? "text-purple-600 bg-purple-50 dark:text-purple-400 dark:bg-purple-900/20"
                      : "text-muted-foreground hover:text-foreground hover:bg-accent"
                  }`}
                >
                  <Icon className="w-4 h-4" />
                  {item.label}
                </button>
              );
            })}
            
            {/* Discover Dropdown Menu */}
            <DropdownMenu>
              <DropdownMenuTrigger asChild>
                <button
                  className={`flex items-center gap-2 px-3 py-2 rounded-md transition-colors ${
                    ["discover", "hot-or-not", "matches", "ads"].includes(currentView)
                      ? "text-purple-600 bg-purple-50 dark:text-purple-400 dark:bg-purple-900/20"
                      : "text-muted-foreground hover:text-foreground hover:bg-accent"
                  }`}
                >
                  <Heart className="w-4 h-4" />
                  {t('nav.discover')}
                  <ChevronDown className="w-3 h-3" />
                </button>
              </DropdownMenuTrigger>
              <DropdownMenuContent align="start" className="w-48">
                {discoverSubItems.map((item) => {
                  const Icon = item.icon;
                  return (
                    <DropdownMenuItem
                      key={item.id}
                      onClick={() => onNavigate(item.id)}
                      className={`flex items-center gap-2 cursor-pointer ${
                        currentView === item.id
                          ? "text-purple-600 bg-purple-50 dark:text-purple-400 dark:bg-purple-900/20"
                          : ""
                      }`}
                    >
                      <Icon className="w-4 h-4" />
                      {item.label}
                    </DropdownMenuItem>
                  );
                })}
              </DropdownMenuContent>
            </DropdownMenu>
            
            {/* Points Display */}
            <Button
              onClick={() => setIsPointsStoreOpen(true)}
              variant="outline"
              className="border-orange-200 hover:bg-orange-50 dark:border-orange-800 dark:hover:bg-orange-900/20"
              size="sm"
            >
              <Coins className="w-4 h-4 mr-2 text-orange-600" />
              <span className="font-semibold text-orange-600">{points.toLocaleString()}</span>
            </Button>

            {/* Premium Button */}
            <Button
              onClick={() => onNavigate("subscription")}
              className={`bg-gradient-to-r from-yellow-500 to-yellow-600 hover:from-yellow-600 hover:to-yellow-700 text-white shadow-lg ${
                currentView === "subscription" 
                  ? "ring-2 ring-yellow-400 ring-offset-2 ring-offset-background" 
                  : ""
              }`}
              size="sm"
            >
              <Crown className="w-4 h-4 mr-2" />
              {t('subscription.premium')}
            </Button>
            </>
            )}
          </nav>

          {/* Theme Toggle and User Menu */}
          <div className="flex items-center gap-2">
            <LanguageSelector variant="icon" />
            <ThemeToggle />
            {currentView !== "landing" && (
              <>
                {/* Avatar - Click to view profile */}
                <div 
                  className="flex items-center gap-3 cursor-pointer"
                  onClick={() => onNavigate("user-profile")}
                >
                  <Avatar className="w-8 h-8">
                    <AvatarImage src="https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?w=100&h=100&fit=crop&crop=face" />
                    <AvatarFallback>{user?.name.charAt(0) || 'U'}</AvatarFallback>
                  </Avatar>
                  <div className="hidden sm:block">
                    <div className="flex items-center gap-2">
                      <span className="text-sm font-medium text-foreground hover:text-purple-600 transition-colors">
                        {user?.name || 'User'}
                      </span>
                      {user?.role === 'super-admin' && (
                        <Badge className="bg-gradient-to-r from-red-500 to-pink-600 text-white text-xs">
                          <Crown className="w-2 h-2 mr-1" />
                          Super
                        </Badge>
                      )}
                      {user?.role === 'admin' && (
                        <Badge className="bg-purple-100 text-purple-800 text-xs">
                          <Shield className="w-2 h-2 mr-1" />
                          Admin
                        </Badge>
                      )}
                      {user?.role === 'moderator' && (
                        <Badge className="bg-blue-100 text-blue-800 text-xs">
                          <UserCheck className="w-2 h-2 mr-1" />
                          Mod
                        </Badge>
                      )}
                    </div>
                  </div>
                </div>
                
                <DropdownMenu>
                  <DropdownMenuTrigger asChild>
                    <Button variant="ghost" size="sm" className="p-1">
                      <ChevronDown className="w-4 h-4 text-muted-foreground" />
                    </Button>
                  </DropdownMenuTrigger>
                  <DropdownMenuContent align="end" className="w-56">
                    {/* Super Admin Panel */}
                    {isRole('super-admin') && (
                      <DropdownMenuItem 
                        onClick={() => onNavigate("super-admin")}
                        className="flex items-center gap-2 text-red-600 bg-red-50 dark:text-red-400 dark:bg-red-900/20"
                      >
                        <Crown className="w-4 h-4" />
                        Super Admin Panel
                      </DropdownMenuItem>
                    )}
                    
                    {/* Admin Control Panel */}
                    {isRole(['admin', 'moderator']) && (
                      <DropdownMenuItem 
                        onClick={() => onNavigate("admin-panel")}
                        className="flex items-center gap-2 text-purple-600 bg-purple-50 dark:text-purple-400 dark:bg-purple-900/20"
                      >
                        <Shield className="w-4 h-4" />
                        {isRole('admin') ? 'Admin Control Panel' : 'Moderation Panel'}
                      </DropdownMenuItem>
                    )}
                    
                    {(isRole(['super-admin', 'admin', 'moderator'])) && <DropdownMenuSeparator />}
                    
                    <DropdownMenuItem onClick={() => onNavigate("user-profile")}>
                      View My Profile
                    </DropdownMenuItem>
                    <DropdownMenuItem onClick={() => onNavigate("profile-settings")}>
                      Profile Settings
                    </DropdownMenuItem>
                    <DropdownMenuItem onClick={() => onNavigate("privacy")}>
                      Privacy
                    </DropdownMenuItem>
                    
                    <DropdownMenuSeparator />
                    
                    {/* Role Switcher for Testing */}
                    <div className="px-2 py-1">
                      <p className="text-xs text-muted-foreground mb-1">Switch Role (Testing):</p>
                      <div className="grid grid-cols-2 gap-1">
                        <Button size="sm" variant="ghost" className="h-6 text-xs" onClick={() => switchToRole('super-admin')}>
                          Super
                        </Button>
                        <Button size="sm" variant="ghost" className="h-6 text-xs" onClick={() => switchToRole('admin')}>
                          Admin
                        </Button>
                        <Button size="sm" variant="ghost" className="h-6 text-xs" onClick={() => switchToRole('moderator')}>
                          Mod
                        </Button>
                        <Button size="sm" variant="ghost" className="h-6 text-xs" onClick={() => switchToRole('user')}>
                          User
                        </Button>
                      </div>
                    </div>
                    
                    <DropdownMenuSeparator />
                    
                    <DropdownMenuItem onClick={() => setIsPointsStoreOpen(true)}>
                      <ShoppingCart className="w-4 h-4 mr-2" />
                      Points Store
                    </DropdownMenuItem>
                    <DropdownMenuItem>Help & Support</DropdownMenuItem>
                    <DropdownMenuItem className="text-red-600">Sign Out</DropdownMenuItem>
                  </DropdownMenuContent>
                </DropdownMenu>
              </>
            )}
          </div>
        </div>
      </header>
      
      {/* Points Store Modal */}
      <PointsStore 
        isOpen={isPointsStoreOpen} 
        onClose={() => setIsPointsStoreOpen(false)} 
      />
    </>
  );
}