import { useState } from "react";
import { ArrowLeft, Search, Phone, Mail, MessageCircle, Book, Shield, Heart, Users } from "lucide-react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "./ui/card";
import { Input } from "./ui/input";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Accordion, AccordionContent, AccordionItem, AccordionTrigger } from "./ui/accordion";

interface HelpCenterPageProps {
  onBack: () => void;
}

export function HelpCenterPage({ onBack }: HelpCenterPageProps) {
  const [searchQuery, setSearchQuery] = useState("");

  const helpCategories = [
    {
      id: "getting-started",
      title: "Getting Started",
      icon: Heart,
      description: "Learn the basics of using SinglesZonne",
      articles: [
        "How to create your profile",
        "Setting up your photos",
        "Writing an attractive bio",
        "Understanding verification"
      ]
    },
    {
      id: "account-settings",
      title: "Account & Settings",
      icon: Users,
      description: "Manage your account preferences",
      articles: [
        "Changing your password",
        "Privacy settings explained",
        "Notification preferences",
        "Deleting your account"
      ]
    },
    {
      id: "messaging",
      title: "Messaging & Matches",
      icon: MessageCircle,
      description: "Connect with other members",
      articles: [
        "How matching works",
        "Sending your first message",
        "Message safety tips",
        "Reporting inappropriate behavior"
      ]
    },
    {
      id: "safety",
      title: "Safety & Security",
      icon: Shield,
      description: "Stay safe while dating online",
      articles: [
        "Meeting someone for the first time",
        "Recognizing red flags",
        "Blocking and reporting users",
        "Identity verification process"
      ]
    }
  ];

  const faqs = [
    {
      question: "How do I create a profile?",
      answer: "Creating a profile is easy! Click 'Sign Up' and follow our step-by-step process. You'll need to provide basic information, upload photos, and write a brief bio about yourself."
    },
    {
      question: "Is SinglesZonne free to use?",
      answer: "Yes! SinglesZonne offers a free basic membership. Premium features are available through our subscription plans for enhanced matching and messaging capabilities."
    },
    {
      question: "How does matching work?",
      answer: "Our matching algorithm considers your preferences, location, interests, and activity patterns to suggest compatible profiles. You can also browse and discover profiles manually."
    },
    {
      question: "How do I report inappropriate behavior?",
      answer: "You can report any user by clicking the three dots on their profile and selecting 'Report'. Our moderation team reviews all reports within 24 hours."
    },
    {
      question: "Can I hide my profile temporarily?",
      answer: "Yes! In your privacy settings, you can set your profile to 'Private' which will hide it from search results while keeping your account active."
    },
    {
      question: "How do I verify my account?",
      answer: "Account verification helps build trust. You can verify through photo verification, phone number, or email. Verified accounts get higher visibility in search results."
    }
  ];

  const filteredCategories = helpCategories.filter(category =>
    category.title.toLowerCase().includes(searchQuery.toLowerCase()) ||
    category.description.toLowerCase().includes(searchQuery.toLowerCase())
  );

  const filteredFAQs = faqs.filter(faq =>
    faq.question.toLowerCase().includes(searchQuery.toLowerCase()) ||
    faq.answer.toLowerCase().includes(searchQuery.toLowerCase())
  );

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <div className="bg-gradient-to-r from-purple-600 to-pink-600 text-white">
        <div className="max-w-6xl mx-auto px-6 py-8">
          <div className="flex items-center gap-4 mb-6">
            <Button
              variant="ghost"
              size="sm"
              onClick={onBack}
              className="text-white hover:bg-white/20"
            >
              <ArrowLeft className="w-4 h-4 mr-2" />
              Back
            </Button>
          </div>
          
          <div className="text-center max-w-2xl mx-auto">
            <h1 className="text-4xl font-bold mb-4">Help Center</h1>
            <p className="text-purple-100 text-lg mb-8">
              Find answers to your questions and get the help you need
            </p>
            
            {/* Search */}
            <div className="relative">
              <Search className="absolute left-4 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
              <Input
                placeholder="Search for help articles..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="pl-12 pr-4 py-3 text-lg bg-white/10 border-white/20 text-white placeholder:text-white/60"
              />
            </div>
          </div>
        </div>
      </div>

      {/* Content */}
      <div className="max-w-6xl mx-auto px-6 py-12">
        {/* Quick Contact */}
        <div className="grid md:grid-cols-3 gap-6 mb-12">
          <Card className="text-center">
            <CardHeader>
              <Phone className="w-8 h-8 text-purple-600 mx-auto mb-2" />
              <CardTitle>Call Us</CardTitle>
              <CardDescription>Speak with our support team</CardDescription>
            </CardHeader>
            <CardContent>
              <p className="font-medium text-lg mb-2">+1 (555) 123-4567</p>
              <p className="text-sm text-muted-foreground">Mon-Fri 9AM-6PM EST</p>
            </CardContent>
          </Card>

          <Card className="text-center">
            <CardHeader>
              <Mail className="w-8 h-8 text-purple-600 mx-auto mb-2" />
              <CardTitle>Email Support</CardTitle>
              <CardDescription>Get help via email</CardDescription>
            </CardHeader>
            <CardContent>
              <p className="font-medium text-lg mb-2">support@singleszonne.com</p>
              <p className="text-sm text-muted-foreground">Response within 24 hours</p>
            </CardContent>
          </Card>

          <Card className="text-center">
            <CardHeader>
              <MessageCircle className="w-8 h-8 text-purple-600 mx-auto mb-2" />
              <CardTitle>Live Chat</CardTitle>
              <CardDescription>Chat with us in real-time</CardDescription>
            </CardHeader>
            <CardContent>
              <Button className="w-full bg-purple-600 hover:bg-purple-700">
                Start Chat
              </Button>
              <p className="text-sm text-muted-foreground mt-2">Available 24/7</p>
            </CardContent>
          </Card>
        </div>

        {/* Help Categories */}
        <div className="mb-12">
          <h2 className="text-3xl font-bold mb-6">Browse by Category</h2>
          <div className="grid md:grid-cols-2 gap-6">
            {filteredCategories.map((category) => {
              const IconComponent = category.icon;
              return (
                <Card key={category.id} className="hover:shadow-lg transition-shadow cursor-pointer">
                  <CardHeader>
                    <div className="flex items-center gap-3">
                      <div className="p-2 bg-purple-100 dark:bg-purple-900/20 rounded-lg">
                        <IconComponent className="w-6 h-6 text-purple-600" />
                      </div>
                      <div>
                        <CardTitle>{category.title}</CardTitle>
                        <CardDescription>{category.description}</CardDescription>
                      </div>
                    </div>
                  </CardHeader>
                  <CardContent>
                    <ul className="space-y-2">
                      {category.articles.map((article, index) => (
                        <li key={index} className="flex items-center gap-2 text-sm text-muted-foreground hover:text-foreground transition-colors cursor-pointer">
                          <Book className="w-4 h-4" />
                          {article}
                        </li>
                      ))}
                    </ul>
                  </CardContent>
                </Card>
              );
            })}
          </div>
        </div>

        {/* FAQs */}
        <div>
          <div className="flex items-center gap-3 mb-6">
            <h2 className="text-3xl font-bold">Frequently Asked Questions</h2>
            <Badge variant="secondary">{filteredFAQs.length} questions</Badge>
          </div>
          
          <Card>
            <CardContent className="pt-6">
              <Accordion type="single" collapsible className="w-full">
                {filteredFAQs.map((faq, index) => (
                  <AccordionItem key={index} value={`item-${index}`}>
                    <AccordionTrigger className="text-left">
                      {faq.question}
                    </AccordionTrigger>
                    <AccordionContent className="text-muted-foreground leading-relaxed">
                      {faq.answer}
                    </AccordionContent>
                  </AccordionItem>
                ))}
              </Accordion>
            </CardContent>
          </Card>
        </div>

        {/* Still Need Help */}
        <div className="mt-12 text-center">
          <Card className="bg-gradient-to-r from-purple-50 to-pink-50 dark:from-purple-900/20 dark:to-pink-900/20 border-purple-200 dark:border-purple-800">
            <CardContent className="pt-6">
              <h3 className="text-2xl font-bold mb-3">Still need help?</h3>
              <p className="text-muted-foreground mb-6">
                Can't find what you're looking for? Our support team is here to help you.
              </p>
              <Button size="lg" className="bg-purple-600 hover:bg-purple-700">
                Contact Support
              </Button>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}