import { useState } from "react";
import { Header } from "./Header";
import { Footer } from "./Footer";
import { Button } from "./ui/button";
import { Card } from "./ui/card";
import { Heart, X, RotateCcw } from "lucide-react";
import { motion } from "motion/react";
import { ImageWithFallback } from "./figma/ImageWithFallback";
import { useColorTheme } from "./ColorThemeProvider";

interface HotOrNotPageProps {
  onNavigate: (view: string) => void;
  onOpenCookieConsent?: () => void;
}

export function HotOrNotPage({ onNavigate, onOpenCookieConsent }: HotOrNotPageProps) {
  const { theme } = useColorTheme();

  // Sample profile themes
  const profileThemes = [
    { primary: '#ec4899', secondary: '#a855f7', accent: '#6366f1', gradient1: '#ec4899', gradient2: '#a855f7', gradient3: '#6366f1' },
    { primary: '#f59e0b', secondary: '#ef4444', accent: '#ec4899', gradient1: '#f59e0b', gradient2: '#ef4444', gradient3: '#ec4899' },
    { primary: '#0ea5e9', secondary: '#06b6d4', accent: '#10b981', gradient1: '#0ea5e9', gradient2: '#06b6d4', gradient3: '#10b981' },
    { primary: '#10b981', secondary: '#22c55e', accent: '#84cc16', gradient1: '#10b981', gradient2: '#22c55e', gradient3: '#84cc16' }
  ];

  const [currentProfile, setCurrentProfile] = useState({
    id: 1,
    name: "Sarah",
    age: 28,
    location: "New York, NY",
    image: "https://images.unsplash.com/photo-1544005313-94ddf0286df2?w=400&h=600&fit=crop&crop=face",
    colorTheme: profileThemes[0]
  });
  
  const [votingStats, setVotingStats] = useState({
    hotVotes: 0,
    notVotes: 0,
    totalVotes: 0
  });

  const handleVote = (isHot: boolean) => {
    setVotingStats(prev => ({
      hotVotes: prev.hotVotes + (isHot ? 1 : 0),
      notVotes: prev.notVotes + (isHot ? 0 : 1),
      totalVotes: prev.totalVotes + 1
    }));
    
    // In a real app, this would send the vote to the backend
    console.log(`Voted ${isHot ? 'HOT' : 'NOT'} for ${currentProfile.name}`);
    
    // Load next profile (simulate)
    setTimeout(() => {
      const nextId = currentProfile.id + 1;
      setCurrentProfile({
        id: nextId,
        name: `Person ${nextId}`,
        age: Math.floor(Math.random() * 20) + 20,
        location: "Sample City",
        image: `https://images.unsplash.com/photo-${1500000000000 + Math.random() * 100000000}?w=400&h=600&fit=crop&crop=face`,
        colorTheme: profileThemes[nextId % profileThemes.length]
      });
    }, 300);
  };

  const handleSkip = () => {
    // Load next profile without voting
    const nextId = currentProfile.id + 1;
    setCurrentProfile({
      id: nextId,
      name: `Person ${nextId}`,
      age: Math.floor(Math.random() * 20) + 20,
      location: "Sample City",
      image: `https://images.unsplash.com/photo-${1500000000000 + Math.random() * 100000000}?w=400&h=600&fit=crop&crop=face`,
      colorTheme: profileThemes[nextId % profileThemes.length]
    });
  };

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <Header currentView="hot-or-not" onNavigate={onNavigate} />
      
      {/* Hot or Not Banner */}
      <div className="bg-gradient-to-r from-red-600 via-red-700 to-red-800 dark:from-red-700 dark:via-red-800 dark:to-red-900 text-white py-11 relative overflow-hidden">
        {/* Moving gradient overlay */}
        <motion.div 
          className="absolute inset-0 bg-gradient-to-r from-transparent via-red-300/30 to-transparent opacity-40"
          animate={{
            x: ['-100%', '100%']
          }}
          transition={{
            duration: 8,
            repeat: Infinity,
            ease: "linear",
            repeatDelay: 2
          }}
          style={{
            width: '200%',
            background: 'linear-gradient(90deg, transparent 0%, rgba(255,255,255,0.15) 25%, rgba(255,255,255,0.3) 50%, rgba(255,255,255,0.15) 75%, transparent 100%)'
          }}
        />
        
        {/* Additional gradient overlay */}
        <div className="absolute inset-0 bg-gradient-to-r from-transparent via-red-600/20 to-red-800/40 dark:from-transparent dark:via-red-700/20 dark:to-red-900/40"></div>
        
        <div className="max-w-7xl mx-auto px-6 text-center relative z-10">
          <motion.h1 
            className="text-4xl font-bold mb-4"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ 
              duration: 0.8,
              ease: "easeOut"
            }}
          >
            <motion.span
              animate={{ 
                scale: [1, 1.05, 1],
              }}
              transition={{
                duration: 3,
                repeat: Infinity,
                ease: "easeInOut",
                repeatDelay: 0.5
              }}
            >
              🔥 Hot or Not
            </motion.span>
          </motion.h1>
          <motion.p 
            className="text-lg text-red-100 dark:text-red-200"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ 
              duration: 0.8,
              delay: 0.2,
              ease: "easeOut"
            }}
          >
            Rate profiles and see how others rate you!
          </motion.p>
        </div>
      </div>

      {/* Main Content */}
      <div className="max-w-4xl mx-auto px-6 py-12">
        {/* Stats */}
        <div className="text-center mb-8">
          <div className="flex justify-center gap-8">
            <div className="text-center">
              <div 
                className="text-2xl font-bold" 
                style={{ color: currentProfile.colorTheme.primary }}
              >
                {votingStats.hotVotes}
              </div>
              <div className="text-muted-foreground">Hot Votes</div>
            </div>
            <div className="text-center">
              <div 
                className="text-2xl font-bold" 
                style={{ color: currentProfile.colorTheme.secondary }}
              >
                {votingStats.notVotes}
              </div>
              <div className="text-muted-foreground">Not Votes</div>
            </div>
            <div className="text-center">
              <div 
                className="text-2xl font-bold" 
                style={{ color: currentProfile.colorTheme.accent }}
              >
                {votingStats.totalVotes}
              </div>
              <div className="text-muted-foreground">Total Votes</div>
            </div>
          </div>
        </div>

        {/* Profile Card */}
        <motion.div
          key={currentProfile.id}
          initial={{ opacity: 0, scale: 0.9 }}
          animate={{ opacity: 1, scale: 1 }}
          transition={{ duration: 0.3 }}
          className="flex justify-center"
        >
          <Card 
            className="w-full max-w-md p-6 text-center border-0 shadow-lg"
            style={{
              background: `linear-gradient(135deg, ${currentProfile.colorTheme.gradient1}10, ${currentProfile.colorTheme.gradient2}10, ${currentProfile.colorTheme.gradient3}10)`
            }}
          >
            <div className="mb-6">
              <div 
                className="w-64 h-80 rounded-lg mx-auto mb-4 p-1"
                style={{
                  background: `linear-gradient(135deg, ${currentProfile.colorTheme.gradient1}, ${currentProfile.colorTheme.gradient2}, ${currentProfile.colorTheme.gradient3})`
                }}
              >
                <ImageWithFallback
                  src={currentProfile.image}
                  alt={currentProfile.name}
                  className="w-full h-full object-cover rounded-md"
                />
              </div>
              <h3 
                className="text-xl font-semibold mb-1"
                style={{ 
                  background: `linear-gradient(135deg, ${currentProfile.colorTheme.primary}, ${currentProfile.colorTheme.secondary})`,
                  WebkitBackgroundClip: 'text',
                  WebkitTextFillColor: 'transparent'
                }}
              >
                {currentProfile.name}, {currentProfile.age}
              </h3>
              <p className="text-muted-foreground">{currentProfile.location}</p>
            </div>

            {/* Voting Buttons */}
            <div className="flex justify-center gap-4">
              <Button
                onClick={() => handleVote(false)}
                variant="outline"
                size="lg"
                className="bg-white/80 hover:bg-white border"
                style={{ 
                  borderColor: currentProfile.colorTheme.secondary,
                  color: currentProfile.colorTheme.secondary
                }}
              >
                <X className="w-5 h-5 mr-2" />
                Not
              </Button>
              
              <Button
                onClick={handleSkip}
                variant="outline"
                size="lg"
                className="px-3 bg-white/80 hover:bg-white border"
                style={{ 
                  borderColor: currentProfile.colorTheme.accent,
                  color: currentProfile.colorTheme.accent
                }}
              >
                <RotateCcw className="w-5 h-5" />
              </Button>
              
              <Button
                onClick={() => handleVote(true)}
                size="lg"
                className="text-white border-0"
                style={{
                  background: `linear-gradient(135deg, ${currentProfile.colorTheme.primary}, ${currentProfile.colorTheme.gradient2})`
                }}
              >
                <Heart className="w-5 h-5 mr-2" />
                Hot
              </Button>
            </div>
          </Card>
        </motion.div>

        {/* Instructions */}
        <div className="text-center mt-8 text-muted-foreground">
          <p>Rate profiles by clicking "Hot" or "Not". Skip if you're unsure!</p>
        </div>
      </div>
      
      {/* Footer */}
      <Footer onNavigate={onNavigate} onOpenCookieConsent={onOpenCookieConsent} />
    </div>
  );
}