import { useState } from "react";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "./ui/dialog";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Alert, AlertDescription } from "./ui/alert";
import { Avatar, AvatarFallback, AvatarImage } from "./ui/avatar";
import { Card, CardContent } from "./ui/card";
import { 
  LogIn, 
  ArrowLeft, 
  Shield, 
  Crown, 
  UserCheck, 
  AlertTriangle,
  Eye,
  User as UserIcon
} from "lucide-react";
import { useUser, UserRole } from "./UserProvider";
import { toast } from "sonner@2.0.3";

interface LoginSwitcherProps {
  isOpen: boolean;
  onClose: () => void;
  targetUser: {
    id: string;
    name: string;
    email: string;
    role: UserRole;
    avatar?: string;
    subscriptionPlan: string;
  };
  onConfirm: () => void;
}

export function LoginSwitcher({ isOpen, onClose, targetUser, onConfirm }: LoginSwitcherProps) {
  const { user: currentUser, isRole } = useUser();
  const [isLoggingIn, setIsLoggingIn] = useState(false);

  const canLoginAs = () => {
    if (!currentUser) return false;
    
    // Super admin can login as anyone except other super admins
    if (isRole('super-admin')) {
      return targetUser.role !== 'super-admin';
    }
    
    // Admins can login as moderators and regular users
    if (isRole('admin')) {
      return ['moderator', 'premium', 'user'].includes(targetUser.role);
    }
    
    // Moderators can login as regular users
    if (isRole('moderator')) {
      return ['premium', 'user'].includes(targetUser.role);
    }
    
    return false;
  };

  const getRoleBadge = (role: UserRole) => {
    switch (role) {
      case 'super-admin':
        return (
          <Badge className="bg-gradient-to-r from-red-500 to-pink-600 text-white">
            <Crown className="w-3 h-3 mr-1" />
            Super Admin
          </Badge>
        );
      case 'admin':
        return (
          <Badge className="bg-purple-100 text-purple-800">
            <Shield className="w-3 h-3 mr-1" />
            Admin
          </Badge>
        );
      case 'moderator':
        return (
          <Badge className="bg-blue-100 text-blue-800">
            <UserCheck className="w-3 h-3 mr-1" />
            Moderator
          </Badge>
        );
      case 'premium':
        return (
          <Badge className="bg-yellow-100 text-yellow-800">
            <Crown className="w-3 h-3 mr-1" />
            Premium
          </Badge>
        );
      case 'user':
        return (
          <Badge className="bg-gray-100 text-gray-800">
            <UserIcon className="w-3 h-3 mr-1" />
            User
          </Badge>
        );
      default:
        return <Badge variant="outline">{role}</Badge>;
    }
  };

  const handleLoginAs = async () => {
    if (!canLoginAs()) {
      toast.error("You don't have permission to login as this user");
      return;
    }

    setIsLoggingIn(true);
    
    try {
      // Simulate login process
      await new Promise(resolve => setTimeout(resolve, 2000));
      
      // In a real app, this would:
      // 1. Log the action for audit purposes
      // 2. Switch the user session
      // 3. Redirect to the user's view
      
      toast.success(`Successfully logged in as ${targetUser.name}`, {
        description: "You are now viewing the platform from their perspective",
        duration: 5000,
      });
      
      onConfirm();
      onClose();
    } catch (error) {
      toast.error("Failed to login as user");
    } finally {
      setIsLoggingIn(false);
    }
  };

  if (!canLoginAs()) {
    return (
      <Dialog open={isOpen} onOpenChange={onClose}>
        <DialogContent className="max-w-md">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2 text-red-600">
              <AlertTriangle className="w-5 h-5" />
              Access Denied
            </DialogTitle>
            <DialogDescription>
              You don't have permission to login as this user
            </DialogDescription>
          </DialogHeader>
          
          <Alert className="border-red-200 bg-red-50">
            <AlertTriangle className="w-4 h-4" />
            <AlertDescription>
              Only Super Admins can login as other administrators. Admins can login as moderators and regular users.
            </AlertDescription>
          </Alert>

          <div className="flex gap-2 pt-4">
            <Button variant="outline" onClick={onClose} className="flex-1">
              <ArrowLeft className="w-4 h-4 mr-2" />
              Back
            </Button>
          </div>
        </DialogContent>
      </Dialog>
    );
  }

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-md">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <LogIn className="w-5 h-5 text-blue-600" />
            Login as User
          </DialogTitle>
          <DialogDescription>
            You are about to login as another user. This action will be logged for security purposes.
          </DialogDescription>
        </DialogHeader>

        <div className="space-y-4">
          {/* Current User */}
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center gap-3">
                <Avatar className="w-10 h-10">
                  <AvatarFallback>{currentUser?.name.charAt(0)}</AvatarFallback>
                </Avatar>
                <div>
                  <p className="font-medium text-sm">Currently logged in as:</p>
                  <div className="flex items-center gap-2">
                    <span className="text-sm">{currentUser?.name}</span>
                    {currentUser && getRoleBadge(currentUser.role)}
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Arrow */}
          <div className="flex justify-center">
            <div className="p-2 bg-blue-100 rounded-full">
              <LogIn className="w-4 h-4 text-blue-600" />
            </div>
          </div>

          {/* Target User */}
          <Card className="border-blue-200 bg-blue-50">
            <CardContent className="p-4">
              <div className="flex items-center gap-3">
                <Avatar className="w-10 h-10">
                  <AvatarImage src={targetUser.avatar} />
                  <AvatarFallback>{targetUser.name.charAt(0)}</AvatarFallback>
                </Avatar>
                <div>
                  <p className="font-medium text-sm">Switching to:</p>
                  <div className="flex items-center gap-2 mb-1">
                    <span className="text-sm font-medium">{targetUser.name}</span>
                    {getRoleBadge(targetUser.role)}
                  </div>
                  <p className="text-xs text-muted-foreground">{targetUser.email}</p>
                  <div className="flex items-center gap-2 mt-1">
                    <Badge variant="outline" className="text-xs">
                      {targetUser.subscriptionPlan}
                    </Badge>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Warning */}
          <Alert className="border-yellow-200 bg-yellow-50">
            <Eye className="w-4 h-4" />
            <AlertDescription className="text-sm">
              <strong>Important:</strong> This action will be logged for audit purposes. You will have access to all of {targetUser.name}'s account data and functionality.
            </AlertDescription>
          </Alert>

          {/* Action Buttons */}
          <div className="flex gap-2 pt-4">
            <Button 
              variant="outline" 
              onClick={onClose} 
              className="flex-1"
              disabled={isLoggingIn}
            >
              <ArrowLeft className="w-4 h-4 mr-2" />
              Cancel
            </Button>
            <Button 
              onClick={handleLoginAs} 
              className="flex-1 bg-blue-600 hover:bg-blue-700"
              disabled={isLoggingIn}
            >
              {isLoggingIn ? (
                <>
                  <div className="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2" />
                  Logging in...
                </>
              ) : (
                <>
                  <LogIn className="w-4 h-4 mr-2" />
                  Confirm Login
                </>
              )}
            </Button>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}