import { useState, useRef, useEffect } from 'react';
import { GoogleMaps, useGooglePlacesAutocomplete, getCurrentLocation } from './GoogleMaps';
import { Button } from './ui/button';
import { Input } from './ui/input';
import { Label } from './ui/label';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription } from './ui/dialog';
import { MapPin, Navigation, Search } from 'lucide-react';
import { Card } from './ui/card';
import { GoogleMapsPlaceholder } from './GoogleMapsConfig';
import { shouldEnableMaps } from './GoogleMapsApiConfig';

interface MapLocationPickerProps {
  isOpen: boolean;
  onClose: () => void;
  onLocationSelect: (location: {
    address: string;
    city: string;
    country: string;
    coordinates: { lat: number; lng: number };
  }) => void;
  initialLocation?: {
    address: string;
    coordinates: { lat: number; lng: number };
  };
  title?: string;
  description?: string;
}

export function MapLocationPicker({
  isOpen,
  onClose,
  onLocationSelect,
  initialLocation,
  title = "Select Your Location",
  description = "Choose your location on the map or search for a specific address"
}: MapLocationPickerProps) {
  const [selectedLocation, setSelectedLocation] = useState<{
    address: string;
    city: string;
    country: string;
    coordinates: { lat: number; lng: number };
  } | null>(null);
  
  const [mapCenter, setMapCenter] = useState(
    initialLocation?.coordinates || { lat: 40.7128, lng: -74.0060 } // Default to NYC
  );
  
  const [isGettingLocation, setIsGettingLocation] = useState(false);
  const [searchValue, setSearchValue] = useState(initialLocation?.address || '');
  
  const searchInputRef = useRef<HTMLInputElement>(null);
  const { selectedPlace } = useGooglePlacesAutocomplete(searchInputRef);

  // Handle place selection from autocomplete
  useEffect(() => {
    if (!shouldEnableMaps() || !selectedPlace || !selectedPlace.geometry) return;
    
    try {
      const location = selectedPlace.geometry.location;
      const coordinates = {
        lat: location.lat(),
        lng: location.lng()
      };
      
      setMapCenter(coordinates);
      
      // Extract address components
      const addressComponents = selectedPlace.address_components || [];
      let city = '';
      let country = '';
      
      addressComponents.forEach((component: any) => {
        if (component.types.includes('locality')) {
          city = component.long_name;
        } else if (component.types.includes('administrative_area_level_1') && !city) {
          city = component.long_name;
        } else if (component.types.includes('country')) {
          country = component.long_name;
        }
      });
      
      setSelectedLocation({
        address: selectedPlace.formatted_address || selectedPlace.name,
        city: city || 'Unknown City',
        country: country || 'Unknown Country',
        coordinates
      });
    } catch (error) {
      // Silently handle any place selection errors
      return;
    }
  }, [selectedPlace]);

  const handleMapClick = (coordinates: { lat: number; lng: number }) => {
    // Reverse geocoding would go here in a real implementation
    // For now, we'll create a basic location object
    setSelectedLocation({
      address: `${coordinates.lat.toFixed(4)}, ${coordinates.lng.toFixed(4)}`,
      city: 'Selected Location',
      country: 'Unknown',
      coordinates
    });
    setMapCenter(coordinates);
  };

  const handleGetCurrentLocation = async () => {
    setIsGettingLocation(true);
    try {
      const position = await getCurrentLocation();
      setMapCenter(position);
      
      // In a real app, you'd use reverse geocoding here
      setSelectedLocation({
        address: 'Current Location',
        city: 'Current City', 
        country: 'Current Country',
        coordinates: position
      });
    } catch (error) {
      // Silently handle geolocation errors
      // In a real app, you could show a toast notification
    } finally {
      setIsGettingLocation(false);
    }
  };

  const handleConfirmLocation = () => {
    if (selectedLocation) {
      onLocationSelect(selectedLocation);
      onClose();
    }
  };

  const handleCancel = () => {
    setSelectedLocation(null);
    setSearchValue(initialLocation?.address || '');
    onClose();
  };

  return (
    <Dialog open={isOpen} onOpenChange={handleCancel}>
      <DialogContent className="max-w-4xl max-h-[90vh] p-0">
        <DialogHeader className="p-6 pb-0">
          <DialogTitle>{title}</DialogTitle>
          <DialogDescription>{description}</DialogDescription>
        </DialogHeader>
        
        <div className="p-6 pt-4 space-y-4">
          {/* Search Input */}
          <div className="space-y-2">
            <Label htmlFor="location-search">Search for a location</Label>
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-muted-foreground" />
              <Input
                id="location-search"
                ref={searchInputRef}
                placeholder="Enter city, address, or landmark..."
                value={searchValue}
                onChange={(e) => setSearchValue(e.target.value)}
                className="pl-10"
              />
            </div>
          </div>

          {/* Current Location Button */}
          <Button
            variant="outline"
            onClick={handleGetCurrentLocation}
            disabled={isGettingLocation}
            className="w-full"
          >
            <Navigation className="w-4 h-4 mr-2" />
            {isGettingLocation ? 'Getting Location...' : 'Use Current Location'}
          </Button>

          {/* Map */}
          <div className="border rounded-lg overflow-hidden">
            <GoogleMaps
              center={mapCenter}
              zoom={13}
              markers={selectedLocation ? [
                {
                  position: selectedLocation.coordinates,
                  title: 'Selected Location',
                  info: selectedLocation.address
                }
              ] : []}
              className="w-full h-96"
              showControls={true}
              onMapClick={handleMapClick}
            />
          </div>

          {/* Selected Location Display */}
          {selectedLocation && (
            <Card className="p-4 bg-primary/5 border-primary/20">
              <div className="flex items-start gap-3">
                <MapPin className="w-5 h-5 text-primary mt-0.5" />
                <div className="flex-1">
                  <h4 className="font-medium">Selected Location</h4>
                  <p className="text-sm text-muted-foreground">{selectedLocation.address}</p>
                  <p className="text-sm text-muted-foreground">
                    {selectedLocation.city}, {selectedLocation.country}
                  </p>
                  <p className="text-xs text-muted-foreground mt-1">
                    Coordinates: {selectedLocation.coordinates.lat.toFixed(4)}, {selectedLocation.coordinates.lng.toFixed(4)}
                  </p>
                </div>
              </div>
            </Card>
          )}

          {/* Action Buttons */}
          <div className="flex gap-3 pt-4">
            <Button variant="outline" onClick={handleCancel} className="flex-1">
              Cancel
            </Button>
            <Button 
              onClick={handleConfirmLocation} 
              disabled={!selectedLocation}
              className="flex-1"
            >
              Confirm Location
            </Button>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}

// Simplified location display component
interface LocationDisplayProps {
  location: string;
  country: string;
  countryFlag?: string;
  flagImageUrl?: string;
  onClick?: () => void;
  showMapIcon?: boolean;
}

export function LocationDisplay({
  location,
  country,
  countryFlag,
  flagImageUrl,
  onClick,
  showMapIcon = true
}: LocationDisplayProps) {
  return (
    <div 
      className={`flex items-center gap-2 text-muted-foreground ${onClick ? 'cursor-pointer hover:text-primary transition-colors' : ''}`}
      onClick={onClick}
    >
      {showMapIcon && <MapPin className="w-4 h-4" />}
      <span>{location}, {country}</span>
      {flagImageUrl && (
        <img
          src={flagImageUrl}
          alt={`${country} flag`}
          className="w-6 h-4 object-cover rounded-sm border"
        />
      )}
      {countryFlag && <span>{countryFlag}</span>}
    </div>
  );
}

// Hook for managing location state
export function useLocationManager() {
  const [location, setLocation] = useState<{
    address: string;
    city: string;
    country: string;
    coordinates: { lat: number; lng: number };
  } | null>(null);

  const [showLocationPicker, setShowLocationPicker] = useState(false);

  const openLocationPicker = () => setShowLocationPicker(true);
  const closeLocationPicker = () => setShowLocationPicker(false);

  const handleLocationSelect = (selectedLocation: {
    address: string;
    city: string;
    country: string;
    coordinates: { lat: number; lng: number };
  }) => {
    setLocation(selectedLocation);
  };

  return {
    location,
    setLocation,
    showLocationPicker,
    openLocationPicker,
    closeLocationPicker,
    handleLocationSelect
  };
}