// Enhanced Matching Algorithm for SinglesZonne Dating Platform
// Calculates compatibility percentage based on comprehensive profile analysis

interface UserProfile {
  id: string;
  name: string;
  age: number;
  location: string;
  country: string;
  bio: string;
  occupation: string;
  education: string;
  height: string;
  isVerified: boolean;
  isPremium: boolean;
  interests: string[];
  distance: number;
  joinDate: string;
  lookingFor?: {
    relationshipType: string;
    lookingForText: string;
    ageRange: { min: number; max: number };
    maxDistance: number;
    dealBreakers: string[];
    importantQualities: string[];
    countriesPreference: "all" | "specific" | "multiple";
    selectedCountries: string[];
  };
}

interface MatchScore {
  overall: number;
  breakdown: {
    ageCompatibility: number;
    locationCompatibility: number;
    interestAlignment: number;
    relationshipGoals: number;
    qualityAlignment: number;
    dealBreakerConflicts: number;
    educationCompatibility: number;
    verificationBonus: number;
  };
  reasons: string[];
  concerns: string[];
}

export class MatchingAlgorithm {
  // Weight factors for different compatibility aspects
  private static readonly WEIGHTS = {
    age: 0.20,           // 20% - Age compatibility within desired range
    location: 0.15,      // 15% - Distance and country preferences
    interests: 0.25,     // 25% - Shared interests and hobbies
    relationshipGoals: 0.15, // 15% - Relationship type alignment
    qualities: 0.15,     // 15% - Important qualities alignment
    dealBreakers: -0.10, // -10% - Deal breaker conflicts (negative impact)
    education: 0.05,     // 5% - Educational compatibility
    verification: 0.05   // 5% - Verification status bonus
  };

  // Calculate comprehensive match percentage between two users
  static calculateMatch(user1: UserProfile, user2: UserProfile): MatchScore {
    const breakdown = {
      ageCompatibility: this.calculateAgeCompatibility(user1, user2),
      locationCompatibility: this.calculateLocationCompatibility(user1, user2),
      interestAlignment: this.calculateInterestAlignment(user1, user2),
      relationshipGoals: this.calculateRelationshipGoalsAlignment(user1, user2),
      qualityAlignment: this.calculateQualityAlignment(user1, user2),
      dealBreakerConflicts: this.calculateDealBreakerConflicts(user1, user2),
      educationCompatibility: this.calculateEducationCompatibility(user1, user2),
      verificationBonus: this.calculateVerificationBonus(user1, user2)
    };

    // Calculate weighted overall score
    const overall = Math.max(0, Math.min(100, 
      breakdown.ageCompatibility * this.WEIGHTS.age +
      breakdown.locationCompatibility * this.WEIGHTS.location +
      breakdown.interestAlignment * this.WEIGHTS.interests +
      breakdown.relationshipGoals * this.WEIGHTS.relationshipGoals +
      breakdown.qualityAlignment * this.WEIGHTS.qualities +
      breakdown.dealBreakerConflicts * this.WEIGHTS.dealBreakers +
      breakdown.educationCompatibility * this.WEIGHTS.education +
      breakdown.verificationBonus * this.WEIGHTS.verification
    ));

    // Generate reasons and concerns
    const reasons = this.generatePositiveReasons(breakdown, user1, user2);
    const concerns = this.generateConcerns(breakdown, user1, user2);

    return {
      overall: Math.round(overall),
      breakdown,
      reasons,
      concerns
    };
  }

  // Age compatibility based on looking for preferences
  private static calculateAgeCompatibility(user1: UserProfile, user2: UserProfile): number {
    let score = 0;
    let matches = 0;

    // Check if user1's age fits user2's preferences
    if (user2.lookingFor?.ageRange) {
      if (user1.age >= user2.lookingFor.ageRange.min && user1.age <= user2.lookingFor.ageRange.max) {
        score += 100;
      } else {
        // Partial score based on how close they are
        const distance = Math.min(
          Math.abs(user1.age - user2.lookingFor.ageRange.min),
          Math.abs(user1.age - user2.lookingFor.ageRange.max)
        );
        score += Math.max(0, 100 - (distance * 10)); // 10% penalty per year outside range
      }
      matches++;
    }

    // Check if user2's age fits user1's preferences
    if (user1.lookingFor?.ageRange) {
      if (user2.age >= user1.lookingFor.ageRange.min && user2.age <= user1.lookingFor.ageRange.max) {
        score += 100;
      } else {
        const distance = Math.min(
          Math.abs(user2.age - user1.lookingFor.ageRange.min),
          Math.abs(user2.age - user1.lookingFor.ageRange.max)
        );
        score += Math.max(0, 100 - (distance * 10));
      }
      matches++;
    }

    return matches > 0 ? score / matches : 75; // Default 75% if no preferences set
  }

  // Location and country compatibility
  private static calculateLocationCompatibility(user1: UserProfile, user2: UserProfile): number {
    let score = 0;
    let factors = 0;

    // Distance compatibility
    const user1MaxDistance = user1.lookingFor?.maxDistance || 100;
    const user2MaxDistance = user2.lookingFor?.maxDistance || 100;
    
    if (user1.distance <= user1MaxDistance && user1.distance <= user2MaxDistance) {
      score += 100;
    } else {
      // Partial score based on distance penalty
      const maxAllowedDistance = Math.min(user1MaxDistance, user2MaxDistance);
      if (user1.distance <= maxAllowedDistance * 1.5) {
        score += 100 - ((user1.distance - maxAllowedDistance) / maxAllowedDistance * 50);
      }
    }
    factors++;

    // Country preference compatibility
    const user1CountryPref = user1.lookingFor?.countriesPreference || "all";
    const user2CountryPref = user2.lookingFor?.countriesPreference || "all";

    let countryScore = 100;

    // Check if user2's country matches user1's preferences
    if (user1CountryPref !== "all") {
      const user1Countries = user1.lookingFor?.selectedCountries || [];
      if (!user1Countries.includes(user2.country)) {
        countryScore -= 30;
      }
    }

    // Check if user1's country matches user2's preferences
    if (user2CountryPref !== "all") {
      const user2Countries = user2.lookingFor?.selectedCountries || [];
      if (!user2Countries.includes(user1.country)) {
        countryScore -= 30;
      }
    }

    score += Math.max(0, countryScore);
    factors++;

    return score / factors;
  }

  // Interest overlap calculation
  private static calculateInterestAlignment(user1: UserProfile, user2: UserProfile): number {
    const interests1 = user1.interests || [];
    const interests2 = user2.interests || [];
    
    if (interests1.length === 0 && interests2.length === 0) return 50;
    if (interests1.length === 0 || interests2.length === 0) return 25;

    const commonInterests = interests1.filter(interest => 
      interests2.includes(interest)
    );

    const totalUniqueInterests = new Set([...interests1, ...interests2]).size;
    const overlapPercentage = (commonInterests.length * 2) / (interests1.length + interests2.length);
    
    // Bonus for having many shared interests
    const sharedBonus = commonInterests.length >= 3 ? 15 : commonInterests.length >= 2 ? 10 : 5;
    
    return Math.min(100, (overlapPercentage * 80) + sharedBonus);
  }

  // Relationship goals alignment
  private static calculateRelationshipGoalsAlignment(user1: UserProfile, user2: UserProfile): number {
    const type1 = user1.lookingFor?.relationshipType || "";
    const type2 = user2.lookingFor?.relationshipType || "";

    if (!type1 || !type2) return 50;

    // Exact match bonus
    if (type1 === type2) return 100;

    // Compatible relationship types
    const compatibilityMap: { [key: string]: string[] } = {
      "Serious relationship": ["Marriage", "Serious relationship", "Companionship"],
      "Marriage": ["Serious relationship", "Marriage"],
      "Casual dating": ["Casual dating", "Friendship", "Not sure yet"],
      "Friendship": ["Casual dating", "Friendship", "Companionship"],
      "Not sure yet": ["Casual dating", "Friendship", "Not sure yet", "Serious relationship"],
      "Companionship": ["Serious relationship", "Friendship", "Companionship"]
    };

    const compatible1 = compatibilityMap[type1] || [];
    const compatible2 = compatibilityMap[type2] || [];

    if (compatible1.includes(type2) || compatible2.includes(type1)) {
      return 75;
    }

    return 25; // Incompatible relationship goals
  }

  // Important qualities alignment
  private static calculateQualityAlignment(user1: UserProfile, user2: UserProfile): number {
    const qualities1 = user1.lookingFor?.importantQualities || [];
    const qualities2 = user2.lookingFor?.importantQualities || [];

    if (qualities1.length === 0 && qualities2.length === 0) return 75;

    // Simulate user personality traits based on bio, interests, and occupation
    const traits1 = this.extractPersonalityTraits(user1);
    const traits2 = this.extractPersonalityTraits(user2);

    let score = 0;
    let totalChecks = 0;

    // Check how well user1 matches user2's desired qualities
    qualities2.forEach(quality => {
      if (traits1.includes(quality.toLowerCase())) {
        score += 100;
      } else if (this.isRelatedTrait(quality, traits1)) {
        score += 60;
      } else {
        score += 20; // Benefit of doubt
      }
      totalChecks++;
    });

    // Check how well user2 matches user1's desired qualities
    qualities1.forEach(quality => {
      if (traits2.includes(quality.toLowerCase())) {
        score += 100;
      } else if (this.isRelatedTrait(quality, traits2)) {
        score += 60;
      } else {
        score += 20;
      }
      totalChecks++;
    });

    return totalChecks > 0 ? score / totalChecks : 75;
  }

  // Deal breaker conflicts detection
  private static calculateDealBreakerConflicts(user1: UserProfile, user2: UserProfile): number {
    const dealBreakers1 = user1.lookingFor?.dealBreakers || [];
    const dealBreakers2 = user2.lookingFor?.dealBreakers || [];

    if (dealBreakers1.length === 0 && dealBreakers2.length === 0) return 0;

    // Simulate user negative traits (for demo purposes)
    const negativeTraits1 = this.extractNegativeTraits(user1);
    const negativeTraits2 = this.extractNegativeTraits(user2);

    let conflicts = 0;
    let totalChecks = 0;

    // Check if user1 has traits that are deal breakers for user2
    dealBreakers2.forEach(dealBreaker => {
      if (negativeTraits1.includes(dealBreaker.toLowerCase()) || 
          this.isRelatedNegativeTrait(dealBreaker, negativeTraits1)) {
        conflicts++;
      }
      totalChecks++;
    });

    // Check if user2 has traits that are deal breakers for user1
    dealBreakers1.forEach(dealBreaker => {
      if (negativeTraits2.includes(dealBreaker.toLowerCase()) || 
          this.isRelatedNegativeTrait(dealBreaker, negativeTraits2)) {
        conflicts++;
      }
      totalChecks++;
    });

    // Return negative score proportional to conflicts
    return totalChecks > 0 ? -(conflicts / totalChecks * 100) : 0;
  }

  // Education compatibility
  private static calculateEducationCompatibility(user1: UserProfile, user2: UserProfile): number {
    const educationLevels = {
      "High School": 1,
      "Some College": 2,
      "Associate's Degree": 3,
      "Bachelor's Degree": 4,
      "Master's Degree": 5,
      "PhD": 6
    };

    const level1 = educationLevels[user1.education as keyof typeof educationLevels] || 3;
    const level2 = educationLevels[user2.education as keyof typeof educationLevels] || 3;

    const difference = Math.abs(level1 - level2);
    
    if (difference === 0) return 100;
    if (difference === 1) return 85;
    if (difference === 2) return 70;
    return 50;
  }

  // Verification status bonus
  private static calculateVerificationBonus(user1: UserProfile, user2: UserProfile): number {
    let bonus = 0;
    
    if (user1.isVerified) bonus += 50;
    if (user2.isVerified) bonus += 50;
    
    return bonus;
  }

  // Extract personality traits from profile (simplified AI simulation)
  private static extractPersonalityTraits(user: UserProfile): string[] {
    const traits: string[] = [];
    const bio = user.bio.toLowerCase();
    const interests = user.interests.map(i => i.toLowerCase());
    const occupation = user.occupation.toLowerCase();

    // Analyze bio for trait keywords
    if (bio.includes('funny') || bio.includes('humor') || bio.includes('laugh')) traits.push('humor');
    if (bio.includes('kind') || bio.includes('caring') || bio.includes('compassionate')) traits.push('kindness');
    if (bio.includes('honest') || bio.includes('genuine') || bio.includes('authentic')) traits.push('honesty', 'authenticity');
    if (bio.includes('ambitious') || bio.includes('driven') || bio.includes('goals')) traits.push('ambition');
    if (bio.includes('creative') || bio.includes('artistic') || bio.includes('imagination')) traits.push('creativity');
    if (bio.includes('intelligent') || bio.includes('smart') || bio.includes('educated')) traits.push('intelligence');
    if (bio.includes('adventure') || bio.includes('travel') || bio.includes('explore')) traits.push('adventurous');
    if (bio.includes('family') || bio.includes('children') || bio.includes('kids')) traits.push('family-oriented');

    // Analyze interests for traits
    if (interests.includes('art') || interests.includes('music') || interests.includes('photography')) traits.push('creativity');
    if (interests.includes('fitness') || interests.includes('yoga') || interests.includes('sports')) traits.push('health-conscious');
    if (interests.includes('travel') || interests.includes('adventure')) traits.push('adventurous');
    if (interests.includes('reading') || interests.includes('science')) traits.push('intelligence');
    if (interests.includes('volunteering')) traits.push('kindness', 'empathy');

    // Analyze occupation for traits
    if (occupation.includes('teacher') || occupation.includes('nurse') || occupation.includes('social')) traits.push('caring', 'empathy');
    if (occupation.includes('ceo') || occupation.includes('manager') || occupation.includes('entrepreneur')) traits.push('ambition', 'leadership');
    if (occupation.includes('artist') || occupation.includes('designer') || occupation.includes('writer')) traits.push('creativity');
    if (occupation.includes('doctor') || occupation.includes('engineer') || occupation.includes('scientist')) traits.push('intelligence');

    return [...new Set(traits)]; // Remove duplicates
  }

  // Extract potential negative traits (for demo - in reality this would be more sophisticated)
  private static extractNegativeTraits(user: UserProfile): string[] {
    const negativeTraits: string[] = [];
    const bio = user.bio.toLowerCase();

    // This is a simplified simulation - real implementation would need more sophisticated analysis
    if (bio.includes('party') && bio.includes('drink')) negativeTraits.push('heavy drinking');
    if (bio.includes('smoke') || bio.includes('cigarette')) negativeTraits.push('smoking');
    if (bio.includes('drama') || bio.includes('toxic')) negativeTraits.push('negativity');
    
    // For demo purposes, randomly assign some traits based on user ID
    const userId = parseInt(user.id.replace(/\D/g, '')) || 1;
    if (userId % 7 === 0) negativeTraits.push('smoking');
    if (userId % 11 === 0) negativeTraits.push('negativity');

    return negativeTraits;
  }

  // Check if traits are related
  private static isRelatedTrait(targetTrait: string, userTraits: string[]): boolean {
    const relatedTraits: { [key: string]: string[] } = {
      humor: ['funny', 'witty', 'playful'],
      kindness: ['caring', 'compassionate', 'empathy', 'gentle'],
      intelligence: ['smart', 'educated', 'wise', 'intellectual'],
      ambition: ['driven', 'motivated', 'goal-oriented'],
      creativity: ['artistic', 'imaginative', 'innovative'],
      authenticity: ['genuine', 'honest', 'real'],
      adventurous: ['spontaneous', 'explorer', 'outdoorsy']
    };

    const related = relatedTraits[targetTrait.toLowerCase()] || [];
    return related.some(trait => userTraits.includes(trait));
  }

  // Check if negative traits are related
  private static isRelatedNegativeTrait(targetTrait: string, userTraits: string[]): boolean {
    const relatedNegativeTraits: { [key: string]: string[] } = {
      smoking: ['cigarettes', 'tobacco'],
      'heavy drinking': ['alcoholic', 'drinking problem'],
      negativity: ['toxic', 'drama', 'pessimistic'],
      dishonesty: ['liar', 'fake', 'deceptive']
    };

    const related = relatedNegativeTraits[targetTrait.toLowerCase()] || [];
    return related.some(trait => userTraits.includes(trait));
  }

  // Generate positive match reasons
  private static generatePositiveReasons(breakdown: any, user1: UserProfile, user2: UserProfile): string[] {
    const reasons: string[] = [];

    if (breakdown.ageCompatibility >= 90) {
      reasons.push("Perfect age compatibility");
    } else if (breakdown.ageCompatibility >= 70) {
      reasons.push("Good age match");
    }

    if (breakdown.locationCompatibility >= 90) {
      reasons.push("Lives nearby and matches location preferences");
    } else if (breakdown.locationCompatibility >= 70) {
      reasons.push("Compatible location preferences");
    }

    if (breakdown.interestAlignment >= 80) {
      const commonInterests = user1.interests.filter(i => user2.interests.includes(i));
      reasons.push(`Shares ${commonInterests.length} interests: ${commonInterests.slice(0, 3).join(', ')}`);
    } else if (breakdown.interestAlignment >= 60) {
      reasons.push("Some shared interests");
    }

    if (breakdown.relationshipGoals >= 90) {
      reasons.push("Same relationship goals");
    } else if (breakdown.relationshipGoals >= 70) {
      reasons.push("Compatible relationship goals");
    }

    if (breakdown.qualityAlignment >= 80) {
      reasons.push("Strong personality compatibility");
    }

    if (breakdown.educationCompatibility >= 85) {
      reasons.push("Similar education level");
    }

    if (breakdown.verificationBonus > 75) {
      reasons.push("Both verified profiles");
    } else if (breakdown.verificationBonus > 25) {
      reasons.push("Verified profile");
    }

    return reasons;
  }

  // Generate concerns
  private static generateConcerns(breakdown: any, user1: UserProfile, user2: UserProfile): string[] {
    const concerns: string[] = [];

    if (breakdown.ageCompatibility < 50) {
      concerns.push("Age outside preferred range");
    }

    if (breakdown.locationCompatibility < 50) {
      concerns.push("Distance or country preferences don't align");
    }

    if (breakdown.interestAlignment < 40) {
      concerns.push("Few shared interests");
    }

    if (breakdown.relationshipGoals < 50) {
      concerns.push("Different relationship goals");
    }

    if (breakdown.dealBreakerConflicts < -50) {
      concerns.push("Potential deal breaker conflicts");
    }

    if (breakdown.verificationBonus === 0) {
      concerns.push("Neither profile is verified");
    }

    return concerns;
  }
}

// Export utility function for easy use in components
export function calculateMatchPercentage(currentUser: UserProfile, targetUser: UserProfile): MatchScore {
  return MatchingAlgorithm.calculateMatch(currentUser, targetUser);
}

// Quick match percentage for display purposes
export function getQuickMatchPercentage(currentUser: UserProfile, targetUser: UserProfile): number {
  return MatchingAlgorithm.calculateMatch(currentUser, targetUser).overall;
}