import { useState, useRef } from 'react';
import { Card } from './ui/card';
import { Button } from './ui/button';
import { Input } from './ui/input';
import { Label } from './ui/label';
import { Slider } from './ui/slider';
import { Switch } from './ui/switch';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from './ui/select';
import { Alert, AlertDescription } from './ui/alert';
import { Progress } from './ui/progress';
import { Badge } from './ui/badge';
import { Separator } from './ui/separator';
import { ImageWithFallback } from './figma/ImageWithFallback';
import { useMedia } from './MediaManager';
import { 
  Video, 
  Music, 
  Image, 
  Upload, 
  Play, 
  Pause, 
  Volume2, 
  VolumeX,
  X,
  Eye,
  Info,
  Sparkles,
  RotateCw,
  Download
} from 'lucide-react';

export function MediaSettingsTab() {
  const {
    introVideo,
    profileMusic,
    customBackground,
    isVideoPlaying,
    isMusicPlaying,
    isMusicPaused,
    musicVolume,
    setIntroVideo,
    setProfileMusic,
    setCustomBackground,
    playMusic,
    pauseMusic,
    setMusicVolume,
  } = useMedia();

  const [uploadingVideo, setUploadingVideo] = useState(false);
  const [uploadingMusic, setUploadingMusic] = useState(false);
  const [uploadingBackground, setUploadingBackground] = useState(false);
  const [videoProgress, setVideoProgress] = useState(0);
  const [musicProgress, setMusicProgress] = useState(0);
  const [backgroundProgress, setBackgroundProgress] = useState(0);
  const [autoPlayEnabled, setAutoPlayEnabled] = useState(true);
  const [musicLoop, setMusicLoop] = useState(true);
  const [showMusicPlayer, setShowMusicPlayer] = useState(true);

  const videoRef = useRef<HTMLInputElement>(null);
  const musicRef = useRef<HTMLInputElement>(null);
  const backgroundRef = useRef<HTMLInputElement>(null);

  const handleVideoUpload = (files: FileList | null) => {
    if (!files || files.length === 0) return;
    
    const file = files[0];
    if (!file.type.startsWith('video/')) {
      alert('Please select a video file');
      return;
    }

    if (file.size > 100 * 1024 * 1024) { // 100MB limit
      alert('Video file must be under 100MB');
      return;
    }

    // Create temporary URL to check video duration
    const tempUrl = URL.createObjectURL(file);
    const videoElement = document.createElement('video');
    videoElement.preload = 'metadata';
    
    videoElement.onloadedmetadata = () => {
      // Clean up
      window.URL.revokeObjectURL(tempUrl);
      
      // Check if video is longer than 30 seconds
      if (videoElement.duration > 30) {
        alert('Introduction video must be 30 seconds or less. Your video is ' + Math.round(videoElement.duration) + ' seconds long.');
        return;
      }
      
      // Video duration is valid, proceed with upload
      setUploadingVideo(true);
      setVideoProgress(0);

      // Simulate upload progress
      const interval = setInterval(() => {
        setVideoProgress(prev => {
          if (prev >= 100) {
            clearInterval(interval);
            setUploadingVideo(false);
            
            // Create temporary URL for preview
            const finalUrl = URL.createObjectURL(file);
            setIntroVideo({
              url: finalUrl,
              title: file.name,
              duration: `${Math.floor(videoElement.duration / 60)}:${Math.floor(videoElement.duration % 60).toString().padStart(2, '0')}`,
              size: `${(file.size / (1024 * 1024)).toFixed(1)}MB`
            });
            
            return 100;
          }
          return prev + 5;
        });
      }, 100);
    };
    
    videoElement.onerror = () => {
      window.URL.revokeObjectURL(tempUrl);
      alert('Error loading video file. Please try a different file.');
    };
    
    videoElement.src = tempUrl;
  };

  const handleMusicUpload = (files: FileList | null) => {
    if (!files || files.length === 0) return;
    
    const file = files[0];
    if (!file.type.startsWith('audio/')) {
      alert('Please select an audio file');
      return;
    }

    if (file.size > 50 * 1024 * 1024) { // 50MB limit
      alert('Audio file must be under 50MB');
      return;
    }

    setUploadingMusic(true);
    setMusicProgress(0);

    // Simulate upload progress
    const interval = setInterval(() => {
      setMusicProgress(prev => {
        if (prev >= 100) {
          clearInterval(interval);
          setUploadingMusic(false);
          
          // Create temporary URL for preview
          const tempUrl = URL.createObjectURL(file);
          setProfileMusic({
            url: tempUrl,
            title: file.name.replace(/\.[^/.]+$/, ""),
            artist: "Unknown Artist",
            duration: "3:45" // Would be calculated from actual audio
          });
          
          return 100;
        }
        return prev + 3;
      });
    }, 150);
  };

  const handleBackgroundUpload = (files: FileList | null) => {
    if (!files || files.length === 0) return;
    
    const file = files[0];
    if (!file.type.startsWith('image/')) {
      alert('Please select an image file');
      return;
    }

    if (file.size > 10 * 1024 * 1024) { // 10MB limit
      alert('Image file must be under 10MB');
      return;
    }

    setUploadingBackground(true);
    setBackgroundProgress(0);

    // Simulate upload progress
    const interval = setInterval(() => {
      setBackgroundProgress(prev => {
        if (prev >= 100) {
          clearInterval(interval);
          setUploadingBackground(false);
          
          // Create temporary URL for preview
          const tempUrl = URL.createObjectURL(file);
          setCustomBackground({
            url: tempUrl,
            type: 'image',
            name: file.name
          });
          
          return 100;
        }
        return prev + 8;
      });
    }, 100);
  };

  const removeVideo = () => {
    setIntroVideo(null);
  };

  const removeMusic = () => {
    setProfileMusic(null);
  };

  const removeBackground = () => {
    setCustomBackground(null);
  };

  const backgroundPresets = [
    { name: 'Sunset Gradient', type: 'gradient', value: 'linear-gradient(135deg, #ff6b6b, #feca57, #48cae4)' },
    { name: 'Ocean Blue', type: 'gradient', value: 'linear-gradient(135deg, #667eea, #764ba2)' },
    { name: 'Purple Dream', type: 'gradient', value: 'linear-gradient(135deg, #a8edea, #fed6e3)' },
    { name: 'Forest Green', type: 'gradient', value: 'linear-gradient(135deg, #d299c2, #fef9d7)' },
    { name: 'Warm Sunset', type: 'gradient', value: 'linear-gradient(135deg, #fa709a, #fee140)' },
    { name: 'Cool Mint', type: 'gradient', value: 'linear-gradient(135deg, #a8e6cf, #dcedc8)' }
  ];

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-3">
          <div className="p-2 rounded-full bg-gradient-to-r from-purple-500 to-pink-600">
            <Video className="w-6 h-6 text-white" />
          </div>
          <div>
            <h2 className="text-2xl font-bold">Media Settings</h2>
            <p className="text-muted-foreground">Manage your intro video, profile music, and backgrounds</p>
          </div>
        </div>
      </div>

      {/* Media Guidelines */}
      <Alert>
        <Info className="h-4 w-4" />
        <AlertDescription>
          <strong>Media Guidelines:</strong> Introduction videos must be 30 seconds or less. Video audio takes priority over background music. 
          Music should be royalty-free or owned by you. Custom backgrounds should be tasteful and not distracting.
        </AlertDescription>
      </Alert>

      {/* Intro Video Section */}
      <Card className="p-6">
        <div className="flex items-center justify-between mb-4">
          <h3 className="text-lg font-semibold flex items-center gap-2">
            <Video className="w-5 h-5" />
            Introduction Video
          </h3>
          <Badge variant="secondary">Optional</Badge>
        </div>

        {!introVideo && !uploadingVideo ? (
          <div className="border-2 border-dashed border-muted-foreground/25 rounded-lg p-8 text-center">
            <div className="mx-auto w-16 h-16 rounded-full bg-blue-100 dark:bg-blue-900/30 flex items-center justify-center mb-4">
              <Video className="w-8 h-8 text-blue-600" />
            </div>
            <h4 className="text-lg font-semibold mb-2">Add Introduction Video</h4>
            <p className="text-muted-foreground mb-4">
              Share a short video introducing yourself (max 60 seconds, 100MB)
            </p>
            <Button onClick={() => videoRef.current?.click()} className="flex items-center gap-2">
              <Upload className="w-4 h-4" />
              Upload Video
            </Button>
            <input
              ref={videoRef}
              type="file"
              accept="video/*"
              onChange={(e) => handleVideoUpload(e.target.files)}
              className="hidden"
            />
          </div>
        ) : uploadingVideo ? (
          <div className="space-y-4">
            <div className="flex items-center justify-between">
              <span className="text-sm font-medium">Uploading video...</span>
              <span className="text-sm text-muted-foreground">{videoProgress}%</span>
            </div>
            <Progress value={videoProgress} className="h-2" />
          </div>
        ) : (
          <div className="space-y-4">
            <div className="aspect-video bg-black rounded-lg overflow-hidden relative">
              <video 
                src={introVideo?.url} 
                controls 
                className="w-full h-full object-cover"
                poster="https://images.unsplash.com/photo-1492691527719-9d1e07e534b4?w=800&h=450&fit=crop"
              />
              <Button
                variant="destructive"
                size="sm"
                onClick={removeVideo}
                className="absolute top-2 right-2"
              >
                <X className="w-4 h-4" />
              </Button>
            </div>
            <div className="flex items-center justify-between">
              <div>
                <h4 className="font-medium">{introVideo?.title}</h4>
                <p className="text-sm text-muted-foreground">
                  Duration: {introVideo?.duration} • Size: {introVideo?.size}
                </p>
              </div>
              <Button variant="outline" size="sm">
                <RotateCw className="w-4 h-4 mr-2" />
                Replace
              </Button>
            </div>
          </div>
        )}
      </Card>

      {/* Profile Music Section */}
      <Card className="p-6">
        <div className="flex items-center justify-between mb-4">
          <h3 className="text-lg font-semibold flex items-center gap-2">
            <Music className="w-5 h-5" />
            Profile Music
          </h3>
          <Badge variant="secondary">Premium Feature</Badge>
        </div>

        {!profileMusic && !uploadingMusic ? (
          <div className="border-2 border-dashed border-muted-foreground/25 rounded-lg p-8 text-center">
            <div className="mx-auto w-16 h-16 rounded-full bg-purple-100 dark:bg-purple-900/30 flex items-center justify-center mb-4">
              <Music className="w-8 h-8 text-purple-600" />
            </div>
            <h4 className="text-lg font-semibold mb-2">Add Profile Music</h4>
            <p className="text-muted-foreground mb-4">
              Add background music to your profile (max 50MB, MP3/WAV)
            </p>
            <Button onClick={() => musicRef.current?.click()} className="flex items-center gap-2">
              <Upload className="w-4 h-4" />
              Upload Music
            </Button>
            <input
              ref={musicRef}
              type="file"
              accept="audio/*"
              onChange={(e) => handleMusicUpload(e.target.files)}
              className="hidden"
            />
          </div>
        ) : uploadingMusic ? (
          <div className="space-y-4">
            <div className="flex items-center justify-between">
              <span className="text-sm font-medium">Uploading music...</span>
              <span className="text-sm text-muted-foreground">{musicProgress}%</span>
            </div>
            <Progress value={musicProgress} className="h-2" />
          </div>
        ) : (
          <div className="space-y-4">
            <div className="flex items-center justify-between p-4 bg-muted rounded-lg">
              <div className="flex items-center gap-3">
                <Button
                  size="sm"
                  variant="outline"
                  onClick={isMusicPlaying ? pauseMusic : playMusic}
                >
                  {isMusicPlaying ? <Pause className="w-4 h-4" /> : <Play className="w-4 h-4" />}
                </Button>
                <div>
                  <h4 className="font-medium">{profileMusic?.title}</h4>
                  <p className="text-sm text-muted-foreground">
                    {profileMusic?.artist} • {profileMusic?.duration}
                  </p>
                </div>
              </div>
              <Button variant="destructive" size="sm" onClick={removeMusic}>
                <X className="w-4 h-4" />
              </Button>
            </div>

            {/* Music Controls */}
            <div className="space-y-4">
              <div className="space-y-2">
                <Label className="flex items-center gap-2">
                  <Volume2 className="w-4 h-4" />
                  Volume: {musicVolume}%
                </Label>
                <Slider
                  value={[musicVolume]}
                  onValueChange={(value) => setMusicVolume(value[0])}
                  max={100}
                  step={1}
                  className="w-full"
                />
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="flex items-center space-x-2">
                  <Switch 
                    id="autoplay" 
                    checked={autoPlayEnabled}
                    onCheckedChange={setAutoPlayEnabled}
                  />
                  <Label htmlFor="autoplay">Auto-play when profile is viewed</Label>
                </div>
                
                <div className="flex items-center space-x-2">
                  <Switch 
                    id="loop" 
                    checked={musicLoop}
                    onCheckedChange={setMusicLoop}
                  />
                  <Label htmlFor="loop">Loop music</Label>
                </div>
              </div>
            </div>
          </div>
        )}
      </Card>

      {/* Custom Background Section */}
      <Card className="p-6">
        <div className="flex items-center justify-between mb-4">
          <h3 className="text-lg font-semibold flex items-center gap-2">
            <Image className="w-5 h-5" />
            Profile Background
          </h3>
          <Badge variant="secondary">Customization</Badge>
        </div>

        <div className="space-y-6">
          {/* Background Presets */}
          <div>
            <h4 className="font-medium mb-3">Gradient Presets</h4>
            <div className="grid grid-cols-2 md:grid-cols-3 gap-3">
              {backgroundPresets.map((preset, index) => (
                <button
                  key={index}
                  onClick={() => setCustomBackground({
                    url: preset.value,
                    type: 'gradient',
                    name: preset.name
                  })}
                  className={`aspect-[3/2] rounded-lg border-2 transition-all hover:scale-105 ${
                    customBackground?.url === preset.value ? 'border-primary ring-2 ring-primary/20' : 'border-muted'
                  }`}
                  style={{ background: preset.value }}
                >
                  <div className="w-full h-full flex items-end p-2">
                    <span className="text-xs font-medium text-white bg-black/50 px-2 py-1 rounded">
                      {preset.name}
                    </span>
                  </div>
                </button>
              ))}
            </div>
          </div>

          <Separator />

          {/* Custom Image Upload */}
          <div>
            <h4 className="font-medium mb-3">Custom Image</h4>
            {!customBackground?.type === 'image' && !uploadingBackground ? (
              <div className="border-2 border-dashed border-muted-foreground/25 rounded-lg p-6 text-center">
                <div className="mx-auto w-12 h-12 rounded-full bg-green-100 dark:bg-green-900/30 flex items-center justify-center mb-3">
                  <Image className="w-6 h-6 text-green-600" />
                </div>
                <h5 className="font-medium mb-2">Upload Custom Background</h5>
                <p className="text-sm text-muted-foreground mb-3">
                  JPG, PNG, or GIF (max 10MB)
                </p>
                <Button 
                  variant="outline" 
                  size="sm" 
                  onClick={() => backgroundRef.current?.click()}
                  className="flex items-center gap-2"
                >
                  <Upload className="w-4 h-4" />
                  Choose Image
                </Button>
                <input
                  ref={backgroundRef}
                  type="file"
                  accept="image/*"
                  onChange={(e) => handleBackgroundUpload(e.target.files)}
                  className="hidden"
                />
              </div>
            ) : uploadingBackground ? (
              <div className="space-y-3">
                <div className="flex items-center justify-between">
                  <span className="text-sm font-medium">Uploading background...</span>
                  <span className="text-sm text-muted-foreground">{backgroundProgress}%</span>
                </div>
                <Progress value={backgroundProgress} className="h-2" />
              </div>
            ) : customBackground?.type === 'image' ? (
              <div className="space-y-3">
                <div className="aspect-[3/2] rounded-lg overflow-hidden relative">
                  <ImageWithFallback
                    src={customBackground.url}
                    alt="Custom background"
                    className="w-full h-full object-cover"
                  />
                  <Button
                    variant="destructive"
                    size="sm"
                    onClick={removeBackground}
                    className="absolute top-2 right-2"
                  >
                    <X className="w-4 h-4" />
                  </Button>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-sm font-medium">{customBackground.name}</span>
                  <Button variant="outline" size="sm">
                    <RotateCw className="w-4 h-4 mr-2" />
                    Replace
                  </Button>
                </div>
              </div>
            ) : null}
          </div>

          {/* Current Background Preview */}
          {customBackground && (
            <div>
              <h4 className="font-medium mb-3">Current Background</h4>
              <div 
                className="aspect-[3/2] rounded-lg border-2 border-primary flex items-center justify-center text-white font-medium"
                style={{ 
                  background: customBackground.type === 'gradient' ? customBackground.url : `url(${customBackground.url})`,
                  backgroundSize: 'cover',
                  backgroundPosition: 'center'
                }}
              >
                <div className="bg-black/50 px-3 py-1 rounded">
                  {customBackground.name}
                </div>
              </div>
            </div>
          )}
        </div>
      </Card>

      {/* Media Settings */}
      <Card className="p-6">
        <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
          <Sparkles className="w-5 h-5" />
          Media Preferences
        </h3>
        
        <div className="space-y-4">
          <div className="flex items-center justify-between">
            <div>
              <Label htmlFor="media-privacy">Show media to verified members only</Label>
              <p className="text-sm text-muted-foreground">Only verified members can see your media content</p>
            </div>
            <Switch id="media-privacy" />
          </div>
          
          <div className="flex items-center justify-between">
            <div>
              <Label htmlFor="media-autoplay">Allow media autoplay</Label>
              <p className="text-sm text-muted-foreground">Automatically play videos and music when profile is viewed</p>
            </div>
            <Switch id="media-autoplay" checked={autoPlayEnabled} onCheckedChange={setAutoPlayEnabled} />
          </div>
          
          <div className="flex items-center justify-between">
            <div>
              <Label htmlFor="download-protection">Prevent media downloads</Label>
              <p className="text-sm text-muted-foreground">Disable right-click and download options for your media</p>
            </div>
            <Switch id="download-protection" defaultChecked />
          </div>
        </div>
      </Card>

      {/* Media Statistics */}
      <Card className="p-6">
        <h4 className="font-semibold mb-4">Media Performance</h4>
        <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
          <div className="text-center">
            <div className="text-2xl font-bold text-blue-600">
              {Math.floor(Math.random() * 500) + 100}
            </div>
            <div className="text-sm text-muted-foreground">Video Views</div>
          </div>
          <div className="text-center">
            <div className="text-2xl font-bold text-purple-600">
              {Math.floor(Math.random() * 200) + 50}
            </div>
            <div className="text-sm text-muted-foreground">Music Plays</div>
          </div>
          <div className="text-center">
            <div className="text-2xl font-bold text-green-600">
              {Math.floor(Math.random() * 50) + 10}
            </div>
            <div className="text-sm text-muted-foreground">Shares</div>
          </div>
          <div className="text-center">
            <div className="text-2xl font-bold text-orange-600">
              {Math.floor(Math.random() * 20) + 5}
            </div>
            <div className="text-sm text-muted-foreground">Comments</div>
          </div>
        </div>
      </Card>
    </div>
  );
}