import { useState } from "react";
import { Header } from "./Header";
import { Footer } from "./Footer";
import { Card } from "./ui/card";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { ImageWithFallback } from "./figma/ImageWithFallback";
import { 
  Heart, 
  Share2, 
  Download, 
  ZoomIn,
  ChevronLeft,
  ChevronRight,
  Grid3X3,
  Users,
  Star
} from "lucide-react";
import { useTranslation } from "./TranslationProvider";

interface PhotoGalleryProps {
  onNavigate: (view: string) => void;
  onOpenCookieConsent?: () => void;
}

interface Photo {
  id: number;
  url: string;
  title: string;
  likes: number;
  isLiked: boolean;
}

export function PhotoGallery({ onNavigate, onOpenCookieConsent }: PhotoGalleryProps) {
  const { t } = useTranslation();
  
  // Photo data
  const [photos] = useState<Photo[]>([
    {
      id: 1,
      url: "https://images.unsplash.com/photo-1682459337036-e7bbc3986d53?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxyb21hbnRpYyUyMGNvdXBsZSUyMHBvcnRyYWl0fGVufDF8fHx8MTc1ODI3MzU1M3ww&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      title: "Romantic Moments",
      likes: 247,
      isLiked: false
    },
    {
      id: 2,
      url: "https://images.unsplash.com/photo-1734092916915-d16146c0726c?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHx5b3VuZyUyMHdvbWFuJTIwcG9ydHJhaXQlMjBkYXRpbmd8ZW58MXx8fHwxNzU4MzE1ODkxfDA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      title: "Elegant Portrait",
      likes: 189,
      isLiked: true
    },
    {
      id: 3,
      url: "https://images.unsplash.com/photo-1666980013846-7ba5605ac2c4?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxoYW5kc29tZSUyMG1hbiUyMHBvcnRyYWl0fGVufDF8fHx8MTc1ODMxNTg5NHww&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      title: "Confident Style",
      likes: 312,
      isLiked: false
    },
    {
      id: 4,
      url: "https://images.unsplash.com/photo-1695827679388-65607837df75?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxiZWF1dGlmdWwlMjB3b21hbiUyMHNtaWxlfGVufDF8fHx8MTc1ODMxNTg5N3ww&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      title: "Beautiful Smile",
      likes: 428,
      isLiked: true
    },
    {
      id: 5,
      url: "https://images.unsplash.com/photo-1690643712421-725f99ef7a85?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxhdHRyYWN0aXZlJTIweW91bmclMjBwZW9wbGV8ZW58MXx8fHwxNzU4MzE1OTAwfDA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      title: "Youthful Energy",
      likes: 203,
      isLiked: false
    },
    {
      id: 6,
      url: "https://images.unsplash.com/photo-1658249682516-c7789d418978?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxjb25maWRlbnQlMjBtYW4lMjBwcm9mZXNzaW9uYWx8ZW58MXx8fHwxNzU4MzE1OTA0fDA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      title: "Professional Look",
      likes: 156,
      isLiked: false
    },
    {
      id: 7,
      url: "https://images.unsplash.com/photo-1544070629-f6eb7b8c6c5b?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxlbGVnYW50JTIwd29tYW4lMjBwb3J0cmFpdHxlbnwxfHx8fDE3NTgyMTczMTh8MA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      title: "Grace & Elegance",
      likes: 367,
      isLiked: true
    },
    {
      id: 8,
      url: "https://images.unsplash.com/photo-1661181475147-bbd20ef65781?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxzdHlsaXNoJTIweW91bmclMjBtYW58ZW58MXx8fHwxNzU4MzE1OTA5fDA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      title: "Modern Style",
      likes: 291,
      isLiked: false
    },
    {
      id: 9,
      url: "https://images.unsplash.com/photo-1648621288703-88e2273ea454?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxoYXBweSUyMHdvbWFuJTIwb3V0ZG9vcnN8ZW58MXx8fHwxNzU4MzE1OTEzfDA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      title: "Joyful Moments",
      likes: 534,
      isLiked: true
    }
  ]);

  const [selectedPhoto, setSelectedPhoto] = useState<Photo>(photos[8]); // Featured photo (large one)
  const [photoLikes, setPhotoLikes] = useState<{ [key: number]: { likes: number; isLiked: boolean } }>(
    photos.reduce((acc, photo) => ({
      ...acc,
      [photo.id]: { likes: photo.likes, isLiked: photo.isLiked }
    }), {})
  );

  const smallPhotos = photos.slice(0, 8); // First 8 photos for the grid

  const handlePhotoSelect = (photo: Photo) => {
    setSelectedPhoto(photo);
  };

  const handleLike = (photoId: number, e: React.MouseEvent) => {
    e.stopPropagation();
    setPhotoLikes(prev => ({
      ...prev,
      [photoId]: {
        likes: prev[photoId].isLiked ? prev[photoId].likes - 1 : prev[photoId].likes + 1,
        isLiked: !prev[photoId].isLiked
      }
    }));
  };

  const handleShare = (e: React.MouseEvent) => {
    e.stopPropagation();
    // Share functionality would be implemented here
    console.log('Sharing photo:', selectedPhoto.title);
  };

  const handleDownload = (e: React.MouseEvent) => {
    e.stopPropagation();
    // Download functionality would be implemented here
    console.log('Downloading photo:', selectedPhoto.title);
  };

  return (
    <div className="min-h-screen bg-background">
      <Header currentView="gallery" onNavigate={onNavigate} />
      
      <div className="max-w-7xl mx-auto px-6 py-8">
        {/* Header */}
        <div className="mb-8">
          <div className="flex items-center justify-between mb-6">
            <div>
              <h1 className="text-3xl font-bold mb-2">Photo Gallery</h1>
              <p className="text-muted-foreground">
                Discover amazing photos from our community
              </p>
            </div>
            <div className="flex items-center gap-4">
              <Badge variant="secondary" className="flex items-center gap-2">
                <Grid3X3 className="w-4 h-4" />
                {photos.length} Photos
              </Badge>
              <Badge variant="secondary" className="flex items-center gap-2">
                <Users className="w-4 h-4" />
                Featured Gallery
              </Badge>
            </div>
          </div>
        </div>

        {/* Gallery Layout */}
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Small Photos Grid - Left Side */}
          <div className="lg:col-span-1">
            <div className="grid grid-cols-2 gap-4">
              {smallPhotos.map((photo) => (
                <div 
                  key={photo.id}
                  className="relative group cursor-pointer"
                  onClick={() => handlePhotoSelect(photo)}
                >
                  <Card className="overflow-hidden transition-all duration-300 hover:shadow-lg hover:scale-105">
                    <div className="relative aspect-square">
                      <ImageWithFallback
                        src={photo.url}
                        alt={photo.title}
                        className="w-full h-full object-cover"
                      />
                      
                      {/* Overlay */}
                      <div className="absolute inset-0 bg-black/0 group-hover:bg-black/20 transition-all duration-300" />
                      
                      {/* Zoom Icon */}
                      <div className="absolute top-2 right-2 opacity-0 group-hover:opacity-100 transition-opacity duration-300">
                        <div className="bg-white/90 backdrop-blur-sm rounded-full p-2">
                          <ZoomIn className="w-4 h-4 text-gray-800" />
                        </div>
                      </div>
                      
                      {/* Like Button */}
                      <button
                        onClick={(e) => handleLike(photo.id, e)}
                        className="absolute bottom-2 right-2 bg-white/90 backdrop-blur-sm rounded-full p-2 transition-all duration-300 hover:bg-white"
                      >
                        <Heart 
                          className={`w-4 h-4 transition-colors ${
                            photoLikes[photo.id]?.isLiked 
                              ? 'text-red-500 fill-red-500' 
                              : 'text-gray-600'
                          }`} 
                        />
                      </button>
                      
                      {/* Likes Count */}
                      <div className="absolute bottom-2 left-2 bg-black/70 text-white px-2 py-1 rounded text-xs">
                        {photoLikes[photo.id]?.likes || 0}
                      </div>
                    </div>
                  </Card>
                </div>
              ))}
            </div>
          </div>

          {/* Large Featured Photo - Right Side */}
          <div className="lg:col-span-2">
            <Card className="overflow-hidden">
              <div className="relative">
                <div className="aspect-[4/3] lg:aspect-[3/2]">
                  <ImageWithFallback
                    src={selectedPhoto.url}
                    alt={selectedPhoto.title}
                    className="w-full h-full object-cover"
                  />
                </div>
                
                {/* Photo Info Overlay */}
                <div className="absolute bottom-0 left-0 right-0 bg-gradient-to-t from-black/80 via-black/40 to-transparent p-6">
                  <div className="flex items-end justify-between">
                    <div>
                      <h3 className="text-2xl font-bold text-white mb-2">
                        {selectedPhoto.title}
                      </h3>
                      <div className="flex items-center gap-4 text-white/80">
                        <span className="flex items-center gap-1">
                          <Heart className="w-4 h-4" />
                          {photoLikes[selectedPhoto.id]?.likes || 0} likes
                        </span>
                        <span className="flex items-center gap-1">
                          <Star className="w-4 h-4" />
                          Featured
                        </span>
                      </div>
                    </div>
                    
                    {/* Action Buttons */}
                    <div className="flex items-center gap-3">
                      <Button
                        size="sm"
                        variant="secondary"
                        onClick={(e) => handleLike(selectedPhoto.id, e)}
                        className="bg-white/20 hover:bg-white/30 text-white border-white/30"
                      >
                        <Heart 
                          className={`w-4 h-4 mr-2 ${
                            photoLikes[selectedPhoto.id]?.isLiked 
                              ? 'text-red-400 fill-red-400' 
                              : ''
                          }`} 
                        />
                        {photoLikes[selectedPhoto.id]?.isLiked ? 'Liked' : 'Like'}
                      </Button>
                      <Button
                        size="sm"
                        variant="secondary"
                        onClick={handleShare}
                        className="bg-white/20 hover:bg-white/30 text-white border-white/30"
                      >
                        <Share2 className="w-4 h-4 mr-2" />
                        Share
                      </Button>
                      <Button
                        size="sm"
                        variant="secondary"
                        onClick={handleDownload}
                        className="bg-white/20 hover:bg-white/30 text-white border-white/30"
                      >
                        <Download className="w-4 h-4 mr-2" />
                        Save
                      </Button>
                    </div>
                  </div>
                </div>
              </div>
            </Card>
            
            {/* Navigation for Large Photo */}
            <div className="flex items-center justify-between mt-4">
              <Button
                variant="outline"
                size="sm"
                onClick={() => {
                  const currentIndex = photos.findIndex(p => p.id === selectedPhoto.id);
                  const prevIndex = currentIndex > 0 ? currentIndex - 1 : photos.length - 1;
                  setSelectedPhoto(photos[prevIndex]);
                }}
              >
                <ChevronLeft className="w-4 h-4 mr-2" />
                Previous
              </Button>
              
              <div className="text-sm text-muted-foreground">
                {photos.findIndex(p => p.id === selectedPhoto.id) + 1} of {photos.length}
              </div>
              
              <Button
                variant="outline"
                size="sm"
                onClick={() => {
                  const currentIndex = photos.findIndex(p => p.id === selectedPhoto.id);
                  const nextIndex = currentIndex < photos.length - 1 ? currentIndex + 1 : 0;
                  setSelectedPhoto(photos[nextIndex]);
                }}
              >
                Next
                <ChevronRight className="w-4 h-4 ml-2" />
              </Button>
            </div>
          </div>
        </div>

        {/* Stats Section */}
        <div className="mt-12 grid grid-cols-1 md:grid-cols-3 gap-6">
          <Card className="p-6 text-center">
            <div className="text-3xl font-bold text-primary mb-2">
              {photos.reduce((sum, photo) => sum + (photoLikes[photo.id]?.likes || 0), 0)}
            </div>
            <p className="text-muted-foreground">Total Likes</p>
          </Card>
          
          <Card className="p-6 text-center">
            <div className="text-3xl font-bold text-primary mb-2">
              {photos.length}
            </div>
            <p className="text-muted-foreground">Photos</p>
          </Card>
          
          <Card className="p-6 text-center">
            <div className="text-3xl font-bold text-primary mb-2">
              {Math.round(photos.reduce((sum, photo) => sum + (photoLikes[photo.id]?.likes || 0), 0) / photos.length)}
            </div>
            <p className="text-muted-foreground">Avg. Likes</p>
          </Card>
        </div>
      </div>

      <Footer onNavigate={onNavigate} onOpenCookieConsent={onOpenCookieConsent} />
    </div>
  );
}