import { useState, useRef } from 'react';
import { Card } from './ui/card';
import { Button } from './ui/button';
import { Badge } from './ui/badge';
import { Progress } from './ui/progress';
import { Alert, AlertDescription } from './ui/alert';
import { ImageWithFallback } from './figma/ImageWithFallback';
import { 
  Camera, 
  Upload, 
  X, 
  Star, 
  Eye, 
  RotateCw,
  Crop,
  Palette,
  AlertTriangle,
  CheckCircle,
  Info
} from 'lucide-react';
import { motion, Reorder } from 'motion/react';

interface Photo {
  id: string;
  url: string;
  isPrimary: boolean;
  isUploading?: boolean;
  uploadProgress?: number;
  caption?: string;
}

export function PhotoManagementTab() {
  const [photos, setPhotos] = useState<Photo[]>([
    {
      id: '1',
      url: 'https://images.unsplash.com/photo-1494790108755-2616b612b8c0?w=400&h=600&fit=crop&crop=face',
      isPrimary: true,
      caption: 'My favorite selfie from last summer'
    },
    {
      id: '2',
      url: 'https://images.unsplash.com/photo-1517841905240-472988babdf9?w=400&h=600&fit=crop&crop=face',
      isPrimary: false,
      caption: 'At my best friend\'s wedding'
    },
    {
      id: '3',
      url: 'https://images.unsplash.com/photo-1506277886164-e25aa3f4ef7f?w=400&h=600&fit=crop&crop=face',
      isPrimary: false,
      caption: 'Hiking in the mountains'
    }
  ]);

  const [draggedItem, setDraggedItem] = useState<Photo | null>(null);
  const fileInputRef = useRef<HTMLInputElement>(null);
  const maxPhotos = 9;

  const handleFileUpload = (files: FileList | null) => {
    if (!files) return;

    const filesToUpload = Array.from(files).slice(0, availableSlots);

    filesToUpload.forEach((file, index) => {
      if (file.type.startsWith('image/')) {
        const tempId = `temp-${Date.now()}-${index}`;
        const tempUrl = URL.createObjectURL(file);

        // Add photo with upload progress
        const newPhoto: Photo = {
          id: tempId,
          url: tempUrl,
          isPrimary: photos.length === 0, // First photo becomes primary
          isUploading: true,
          uploadProgress: 0
        };

        setPhotos(prev => [...prev, newPhoto]);

        // Simulate upload progress
        const interval = setInterval(() => {
          setPhotos(prev => prev.map(photo => {
            if (photo.id === tempId && photo.uploadProgress !== undefined) {
              const newProgress = Math.min(photo.uploadProgress + 10, 100);
              
              if (newProgress === 100) {
                clearInterval(interval);
                return {
                  ...photo,
                  isUploading: false,
                  uploadProgress: undefined,
                  id: `photo-${Date.now()}-${index}` // Assign permanent ID
                };
              }
              
              return { ...photo, uploadProgress: newProgress };
            }
            return photo;
          }));
        }, 200);
      }
    });
  };

  const handleDrop = (e: React.DragEvent) => {
    e.preventDefault();
    handleFileUpload(e.dataTransfer.files);
  };

  const handleDragOver = (e: React.DragEvent) => {
    e.preventDefault();
  };

  const removePhoto = (photoId: string) => {
    setPhotos(prev => {
      const updated = prev.filter(photo => photo.id !== photoId);
      
      // If we removed the primary photo, make the first remaining photo primary
      if (updated.length > 0 && !updated.some(photo => photo.isPrimary)) {
        updated[0].isPrimary = true;
      }
      
      return updated;
    });
  };

  const setPrimaryPhoto = (photoId: string) => {
    setPhotos(prev => prev.map(photo => ({
      ...photo,
      isPrimary: photo.id === photoId
    })));
  };

  const updateCaption = (photoId: string, caption: string) => {
    setPhotos(prev => prev.map(photo => 
      photo.id === photoId ? { ...photo, caption } : photo
    ));
  };

  const reorderPhotos = (newOrder: Photo[]) => {
    setPhotos(newOrder);
  };

  const canUploadMore = photos.length < maxPhotos;
  const uploadingCount = photos.filter(photo => photo.isUploading).length;
  const availableSlots = maxPhotos - photos.length;

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-3">
          <div className="p-2 rounded-full bg-gradient-to-r from-pink-500 to-purple-600">
            <Camera className="w-6 h-6 text-white" />
          </div>
          <div>
            <h2 className="text-2xl font-bold">Photo Management</h2>
            <p className="text-muted-foreground">Upload and manage your profile photos (max {maxPhotos})</p>
          </div>
        </div>
        
        <div className="text-right">
          <div className="text-lg font-semibold">{photos.length}/{maxPhotos}</div>
          <div className="text-sm text-muted-foreground">Photos uploaded</div>
        </div>
      </div>

      {/* Photo Guidelines */}
      <Alert>
        <Info className="h-4 w-4" />
        <AlertDescription>
          <strong>Photo Guidelines:</strong> Use clear, recent photos of yourself. 
          Avoid group photos, sunglasses, or heavily filtered images. 
          Your primary photo will be shown first to other members.
        </AlertDescription>
      </Alert>

      {/* Upload Area */}
      {canUploadMore && (
        <Card className="p-8 border-2 border-dashed border-muted-foreground/25 hover:border-primary/50 transition-colors">
          <div
            className="text-center space-y-4"
            onDrop={handleDrop}
            onDragOver={handleDragOver}
          >
            <div className="mx-auto w-16 h-16 rounded-full bg-primary/10 flex items-center justify-center">
              <Upload className="w-8 h-8 text-primary" />
            </div>
            
            <div>
              <h3 className="text-lg font-semibold mb-2">Upload Photos</h3>
              <p className="text-muted-foreground mb-4">
                Drag and drop photos here, or click to browse
              </p>
              
              <Button
                onClick={() => fileInputRef.current?.click()}
                className="flex items-center gap-2"
              >
                <Camera className="w-4 h-4" />
                Choose Photos
              </Button>
              
              <input
                ref={fileInputRef}
                type="file"
                multiple
                accept="image/*"
                onChange={(e) => handleFileUpload(e.target.files)}
                className="hidden"
              />
            </div>
            
            <div className="text-sm text-muted-foreground">
              {availableSlots} more photos can be uploaded
            </div>
          </div>
        </Card>
      )}

      {/* Upload Progress */}
      {uploadingCount > 0 && (
        <Card className="p-4">
          <div className="space-y-2">
            <div className="flex items-center justify-between">
              <span className="text-sm font-medium">Uploading {uploadingCount} photo(s)...</span>
              <CheckCircle className="w-4 h-4 text-green-500" />
            </div>
            
            {photos.filter(photo => photo.isUploading).map(photo => (
              <div key={photo.id} className="space-y-1">
                <div className="flex justify-between text-xs">
                  <span>Photo {photo.id}</span>
                  <span>{photo.uploadProgress}%</span>
                </div>
                <Progress value={photo.uploadProgress} className="h-2" />
              </div>
            ))}
          </div>
        </Card>
      )}

      {/* Photo Grid */}
      {photos.length > 0 && (
        <div className="space-y-4">
          <h3 className="text-lg font-semibold">Your Photos</h3>
          
          <Reorder.Group 
            values={photos} 
            onReorder={reorderPhotos}
            className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4"
          >
            {photos.map((photo, index) => (
              <Reorder.Item
                key={photo.id}
                value={photo}
                dragListener={!photo.isUploading}
                className="cursor-move"
              >
                <motion.div
                  layout
                  initial={{ opacity: 0, scale: 0.8 }}
                  animate={{ opacity: 1, scale: 1 }}
                  exit={{ opacity: 0, scale: 0.8 }}
                  whileHover={{ scale: 1.02 }}
                  className="group relative"
                >
                  <Card className={`overflow-hidden ${photo.isPrimary ? 'ring-2 ring-primary' : ''}`}>
                    <div className="aspect-[3/4] relative">
                      <ImageWithFallback
                        src={photo.url}
                        alt={`Photo ${index + 1}`}
                        className="w-full h-full object-cover"
                      />
                      
                      {/* Upload overlay */}
                      {photo.isUploading && (
                        <div className="absolute inset-0 bg-black/50 flex items-center justify-center">
                          <div className="text-center text-white">
                            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-white mx-auto mb-2"></div>
                            <div className="text-sm">Uploading...</div>
                          </div>
                        </div>
                      )}
                      
                      {/* Primary badge */}
                      {photo.isPrimary && !photo.isUploading && (
                        <Badge className="absolute top-2 left-2 bg-primary text-white">
                          <Star className="w-3 h-3 mr-1" />
                          Primary
                        </Badge>
                      )}
                      
                      {/* Action buttons */}
                      {!photo.isUploading && (
                        <div className="absolute inset-0 bg-black/0 group-hover:bg-black/50 transition-colors duration-200">
                          <div className="absolute top-2 right-2 opacity-0 group-hover:opacity-100 transition-opacity duration-200 flex gap-1">
                            <Button
                              size="sm"
                              variant="secondary"
                              onClick={() => setPrimaryPhoto(photo.id)}
                              disabled={photo.isPrimary}
                              className="h-8 w-8 p-0"
                            >
                              <Star className="w-4 h-4" />
                            </Button>
                            
                            <Button
                              size="sm"
                              variant="destructive"
                              onClick={() => removePhoto(photo.id)}
                              className="h-8 w-8 p-0"
                            >
                              <X className="w-4 h-4" />
                            </Button>
                          </div>
                          
                          {/* Center action buttons */}
                          <div className="absolute inset-0 flex items-center justify-center opacity-0 group-hover:opacity-100 transition-opacity duration-200">
                            <div className="flex gap-2">
                              <Button size="sm" variant="secondary" className="flex items-center gap-1">
                                <Crop className="w-4 h-4" />
                                Edit
                              </Button>
                              <Button size="sm" variant="secondary" className="flex items-center gap-1">
                                <RotateCw className="w-4 h-4" />
                                Rotate
                              </Button>
                            </div>
                          </div>
                        </div>
                      )}
                    </div>
                    
                    {/* Photo info */}
                    {!photo.isUploading && (
                      <div className="p-3 space-y-2">
                        <div className="flex items-center justify-between text-sm">
                          <span className="text-muted-foreground">Photo {index + 1}</span>
                          <div className="flex items-center gap-1 text-muted-foreground">
                            <Eye className="w-3 h-3" />
                            <span>{Math.floor(Math.random() * 100) + 10}</span>
                          </div>
                        </div>
                        
                        <input
                          type="text"
                          placeholder="Add a caption..."
                          value={photo.caption || ''}
                          onChange={(e) => updateCaption(photo.id, e.target.value)}
                          className="w-full text-sm bg-transparent border-0 border-b border-muted-foreground/20 focus:border-primary focus:outline-none pb-1"
                        />
                      </div>
                    )}
                  </Card>
                </motion.div>
              </Reorder.Item>
            ))}
          </Reorder.Group>
        </div>
      )}

      {/* Photo Tips */}
      <Card className="p-6 bg-gradient-to-br from-blue-50 to-indigo-50 dark:from-blue-950/20 dark:to-indigo-950/20">
        <h4 className="font-semibold mb-3 text-blue-900 dark:text-blue-100">
          📸 Photo Tips for Better Matches
        </h4>
        <div className="space-y-2 text-sm text-blue-800 dark:text-blue-200">
          <div className="flex items-start gap-2">
            <CheckCircle className="w-4 h-4 mt-0.5 text-green-600" />
            <span><strong>Smile naturally:</strong> Genuine smiles are more attractive than forced poses</span>
          </div>
          <div className="flex items-start gap-2">
            <CheckCircle className="w-4 h-4 mt-0.5 text-green-600" />
            <span><strong>Show variety:</strong> Include full body, close-up, and activity photos</span>
          </div>
          <div className="flex items-start gap-2">
            <CheckCircle className="w-4 h-4 mt-0.5 text-green-600" />
            <span><strong>Good lighting:</strong> Natural light works best for clear, attractive photos</span>
          </div>
          <div className="flex items-start gap-2">
            <AlertTriangle className="w-4 h-4 mt-0.5 text-amber-600" />
            <span><strong>Avoid:</strong> Sunglasses, group photos where you're hard to identify, or heavily filtered images</span>
          </div>
        </div>
      </Card>

      {/* Statistics */}
      {photos.length > 0 && (
        <Card className="p-6">
          <h4 className="font-semibold mb-4">Photo Performance</h4>
          <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
            <div className="text-center">
              <div className="text-2xl font-bold text-primary">
                {photos.reduce((acc, photo) => acc + (Math.floor(Math.random() * 100) + 10), 0)}
              </div>
              <div className="text-sm text-muted-foreground">Total Views</div>
            </div>
            <div className="text-center">
              <div className="text-2xl font-bold text-green-600">
                {Math.floor(Math.random() * 50) + 10}
              </div>
              <div className="text-sm text-muted-foreground">Likes This Week</div>
            </div>
            <div className="text-center">
              <div className="text-2xl font-bold text-blue-600">
                {Math.floor(Math.random() * 20) + 5}
              </div>
              <div className="text-sm text-muted-foreground">Profile Visits</div>
            </div>
            <div className="text-center">
              <div className="text-2xl font-bold text-purple-600">
                {Math.floor(Math.random() * 10) + 1}
              </div>
              <div className="text-sm text-muted-foreground">Matches</div>
            </div>
          </div>
        </Card>
      )}
    </div>
  );
}