import { useState, useEffect, useCallback } from "react";
import { Heart, MessageCircle, Star, Target } from "lucide-react";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { ImageWithFallback } from "./figma/ImageWithFallback";
import { getQuickMatchPercentage } from "./MatchingAlgorithm";
import { useColorTheme } from "./ColorThemeProvider";
import Masonry from "react-responsive-masonry";

interface Profile {
  id: string;
  name: string;
  age: number;
  country: string;
  imageUrl: string;
  isOnline?: boolean;
  gender: "male" | "female";
  aspectRatio: number;
  // Enhanced profile data for matching
  location: string;
  bio: string;
  occupation: string;
  education: string;
  height: string;
  isVerified: boolean;
  isPremium: boolean;
  interests: string[];
  distance: number;
  joinDate: string;
  colorTheme?: {
    primary: string;
    secondary: string;
    accent: string;
    gradient1: string;
    gradient2: string;
    gradient3: string;
  };
  lookingFor?: {
    relationshipType: string;
    lookingForText: string;
    ageRange: { min: number; max: number };
    maxDistance: number;
    dealBreakers: string[];
    importantQualities: string[];
    countriesPreference: "all" | "specific" | "multiple";
    selectedCountries: string[];
  };
}

interface ProfileGridProps {
  profiles: Profile[];
  lookingFor: string;
  onProfileClick?: (profileId: string) => void;
}

export function ProfileGrid({ profiles, lookingFor, onProfileClick }: ProfileGridProps) {
  const [allProfiles, setAllProfiles] = useState<Profile[]>([]);
  const [displayedProfiles, setDisplayedProfiles] = useState<Profile[]>([]);
  const [page, setPage] = useState(1);
  const [loading, setLoading] = useState(false);
  const [hasMore, setHasMore] = useState(true);

  const profilesPerPage = 20;

  // Mock current user profile for matching calculations
  const currentUser: Profile = {
    id: "current-user",
    name: "You",
    age: 32,
    country: "United States",
    location: "New York",
    imageUrl: "",
    gender: "female",
    aspectRatio: 1,
    bio: "Tech entrepreneur and dating app creator. Passionate about bringing people together and creating meaningful connections.",
    occupation: "CEO & Founder",
    education: "MBA in Business Administration",
    height: "5'6\"",
    isVerified: true,
    isPremium: true,
    interests: ["Technology", "Business", "Travel", "Innovation", "Networking", "Leadership", "Photography", "Coffee"],
    distance: 0,
    joinDate: "January 2020",
    lookingFor: {
      relationshipType: "Serious relationship",
      lookingForText: "Seeking someone who values growth, authenticity, and building something meaningful together.",
      ageRange: { min: 28, max: 40 },
      maxDistance: 25,
      dealBreakers: ["Dishonesty", "No ambition", "Disrespectful"],
      importantQualities: ["Honesty", "Ambition", "Humor", "Kindness", "Intelligence", "Authenticity"],
      countriesPreference: "multiple",
      selectedCountries: ["United States", "Canada", "United Kingdom"]
    }
  };

  // Helper function to generate default enhanced data for base profiles
  const getDefaultEnhancedData = (id: string, gender: "male" | "female"): Partial<Profile> => {
    const profileId = parseInt(id) || 1;
    const occupations = ["Software Engineer", "Marketing Manager", "Teacher", "Nurse", "Artist", "Consultant"];
    const educationLevels = ["Bachelor's Degree", "Master's Degree", "PhD"];
    const heights = gender === "male" ? ["5'9\"", "5'10\"", "6'0\"", "6'1\"", "6'2\""] : ["5'4\"", "5'5\"", "5'6\"", "5'7\"", "5'8\""];
    
    return {
      location: "City Center",
      bio: "Living life to the fullest and always open to new adventures. Looking for genuine connections and meaningful relationships.",
      occupation: occupations[profileId % occupations.length],
      education: educationLevels[profileId % educationLevels.length],
      height: heights[profileId % heights.length],
      isVerified: profileId % 3 === 0,
      isPremium: profileId % 4 === 0,
      interests: ["Travel", "Movies", "Music", "Food", "Fitness"].slice(0, 3 + (profileId % 3)),
      distance: (profileId * 7) % 45 + 5,
      joinDate: "2023",
      lookingFor: {
        relationshipType: "Serious relationship",
        lookingForText: "Looking for someone genuine and kind to share life's adventures with.",
        ageRange: { min: 25, max: 40 },
        maxDistance: 30,
        dealBreakers: ["Dishonesty", "Smoking"],
        importantQualities: ["Kindness", "Humor", "Honesty"],
        countriesPreference: "all",
        selectedCountries: []
      }
    };
  };

  const baseProfiles: Profile[] = [
    {
      id: "1",
      name: "Jessica",
      age: 28,
      country: "USA",
      location: "Los Angeles",
      imageUrl: "https://images.unsplash.com/photo-1635159387277-3721b0db850a?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHx3b21hbiUyMHBvcnRyYWl0JTIwbmF0dXJhbCUyMGxpZ2h0aW5nfGVufDF8fHx8MTc1ODExNzAzOXww&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      isOnline: true,
      gender: "female",
      aspectRatio: 1.2,
      bio: "Art director who loves hiking, yoga, and trying new restaurants. Looking for genuine connections and meaningful conversations.",
      occupation: "Art Director",
      education: "Bachelor's Degree",
      height: "5'5\"",
      isVerified: true,
      isPremium: false,
      interests: ["Art", "Hiking", "Yoga", "Food", "Travel", "Photography"],
      distance: 15,
      joinDate: "March 2023",
      colorTheme: { primary: '#ec4899', secondary: '#a855f7', accent: '#6366f1', gradient1: '#ec4899', gradient2: '#a855f7', gradient3: '#6366f1' },
      lookingFor: {
        relationshipType: "Serious relationship",
        lookingForText: "Looking for someone authentic, kind, and adventurous who wants to build something real together.",
        ageRange: { min: 26, max: 38 },
        maxDistance: 30,
        dealBreakers: ["Smoking", "Dishonesty"],
        importantQualities: ["Kindness", "Authenticity", "Humor", "Ambition"],
        countriesPreference: "specific",
        selectedCountries: ["United States"]
      }
    },
    {
      id: "2",
      name: "Sophia",
      age: 26,
      country: "France",
      location: "Paris",
      imageUrl: "https://images.unsplash.com/photo-1647218947427-d783309440d2?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHx3b21hbiUyMGZhc2hpb24lMjBzdHJlZXQlMjBzdHlsZXxlbnwxfHx8fDE3NTgxNTIwNTl8MA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      gender: "female",
      aspectRatio: 1.6,
      bio: "Fashion designer with a passion for sustainable fashion and creativity. Love exploring art galleries and cozy cafes.",
      occupation: "Fashion Designer",
      education: "Master's Degree",
      height: "5'7\"",
      isVerified: false,
      isPremium: true,
      interests: ["Fashion", "Art", "Sustainability", "Design", "Coffee", "Museums"],
      distance: 3400, // Paris to NYC
      joinDate: "August 2023",
      colorTheme: { primary: '#f59e0b', secondary: '#ef4444', accent: '#ec4899', gradient1: '#f59e0b', gradient2: '#ef4444', gradient3: '#ec4899' },
      lookingFor: {
        relationshipType: "Serious relationship",
        lookingForText: "Seeking a creative soul who appreciates art, values authenticity, and believes in building a beautiful life together.",
        ageRange: { min: 28, max: 40 },
        maxDistance: 5000,
        dealBreakers: ["No ambition", "Disrespectful"],
        importantQualities: ["Creativity", "Intelligence", "Respect", "Authenticity"],
        countriesPreference: "multiple",
        selectedCountries: ["France", "United States", "United Kingdom"]
      }
    },
    {
      id: "3",
      name: "Grace",
      age: 27,
      country: "Australia",
      location: "Sydney",
      imageUrl: "https://images.unsplash.com/photo-1622214678958-5a4b59f39091?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHx3b21hbiUyMG91dGRvb3IlMjBzdW5ueSUyMGRheXxlbnwxfHx8fDE3NTgxNTIwNjJ8MA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      gender: "female",
      aspectRatio: 0.8,
      bio: "Marine biologist passionate about ocean conservation. Love surfing, diving, and exploring the underwater world.",
      occupation: "Marine Biologist",
      education: "PhD",
      height: "5'8\"",
      isVerified: true,
      isPremium: false,
      interests: ["Marine Biology", "Surfing", "Diving", "Conservation", "Travel", "Photography"],
      distance: 9500, // Sydney to NYC
      joinDate: "June 2023",
      colorTheme: { primary: '#0ea5e9', secondary: '#06b6d4', accent: '#10b981', gradient1: '#0ea5e9', gradient2: '#06b6d4', gradient3: '#10b981' },
      lookingFor: {
        relationshipType: "Serious relationship",
        lookingForText: "Looking for an adventurous soul who cares about our planet and wants to explore the world together.",
        ageRange: { min: 25, max: 35 },
        maxDistance: 10000,
        dealBreakers: ["No ambition", "Bad hygiene"],
        importantQualities: ["Passion", "Intelligence", "Adventurous", "Caring"],
        countriesPreference: "all",
        selectedCountries: []
      }
    },
    {
      id: "4",
      name: "Amelia",
      age: 30,
      country: "Canada",
      imageUrl: "https://images.unsplash.com/photo-1599135577069-376c0afbe38f?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHx3b21hbiUyMGNhc3VhbCUyMHBvcnRyYWl0JTIwbGlmZXN0eWxlfGVufDF8fHx8MTc1ODE1MjA2NXww&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      gender: "female",
      aspectRatio: 1.4,
      ...getDefaultEnhancedData("4", "female"),
    },
    {
      id: "5",
      name: "Emma",
      age: 24,
      country: "Spain",
      imageUrl: "https://images.unsplash.com/photo-1601585612915-32d6be31dd24?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHx3b21hbiUyMGNvZmZlZSUyMHNob3AlMjBjYW5kaWR8ZW58MXx8fHwxNzU4MTUyMDY4fDA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      isOnline: true,
      gender: "female",
      aspectRatio: 1.1,
      ...getDefaultEnhancedData("5", "female"),
    },
    {
      id: "6",
      name: "Harper",
      age: 25,
      country: "USA",
      imageUrl: "https://images.unsplash.com/photo-1712673337451-dc20a21e9a45?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHx3b21hbiUyMHRyYXZlbGluZyUyMGJhY2twYWNrJTIwYWR2ZW50dXJlfGVufDF8fHx8MTc1ODE1MjA3MXww&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      gender: "female",
      aspectRatio: 0.9,
      ...getDefaultEnhancedData("6", "female"),
    },
    {
      id: "7",
      name: "Olivia",
      age: 29,
      country: "UK",
      imageUrl: "https://images.unsplash.com/photo-1568945721274-e0bf2daed2a4?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHx3b21hbiUyMHJlYWRpbmclMjBib29rJTIwbmF0dXJhbHxlbnwxfHx8fDE3NTgxNTIwNzR8MA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      gender: "female",
      aspectRatio: 1.3,
      ...getDefaultEnhancedData("7", "female"),
    },
    {
      id: "8",
      name: "Mia",
      age: 28,
      country: "Italy",
      imageUrl: "https://images.unsplash.com/photo-1650116385006-2a82a7b9941b?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHx3b21hbiUyMHdvcmtvdXQlMjBmaXRuZXNzJTIwaGVhbHRoeXxlbnwxfHx8fDE3NTgxNTIwNzd8MA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      isOnline: true,
      gender: "female",
      aspectRatio: 1.5,
      ...getDefaultEnhancedData("8", "female"),
    },
    {
      id: "9",
      name: "Ava",
      age: 31,
      country: "Australia",
      imageUrl: "https://images.unsplash.com/photo-1728484011399-380a83fbd68d?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHx3b21hbiUyMGJlYWNoJTIwc3VtbWVyJTIwdmFjYXRpb258ZW58MXx8fHwxNzU4MTUyMDgwfDA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      gender: "female",
      aspectRatio: 0.7,
      ...getDefaultEnhancedData("9", "female"),
    },
    {
      id: "10",
      name: "Luna",
      age: 26,
      country: "Sweden",
      imageUrl: "https://images.unsplash.com/photo-1676382703671-225138665da3?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHx3b21hbiUyMG11c2ljaWFuJTIwZ3VpdGFyJTIwY3JlYXRpdmV8ZW58MXx8fHx8MTc1ODE1MjA4M3ww&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      gender: "female",
      aspectRatio: 1.7,
      ...getDefaultEnhancedData("10", "female"),
    },
    // Male profiles
    {
      id: "11",
      name: "Alexander",
      age: 36,
      country: "Germany",
      imageUrl: "https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?w=400&h=600&fit=crop&crop=face",
      isOnline: true,
      gender: "male",
      aspectRatio: 1.5,
      ...getDefaultEnhancedData("11", "male"),
    },
    {
      id: "12",
      name: "Lucas",
      age: 28,
      country: "Brazil",
      imageUrl: "https://images.unsplash.com/photo-1472099645785-5658abf4ff4e?w=400&h=500&fit=crop&crop=face",
      isOnline: true,
      gender: "male",
      aspectRatio: 1.25,
      ...getDefaultEnhancedData("12", "male"),
    },
    {
      id: "13",
      name: "Daniel",
      age: 31,
      country: "UK",
      imageUrl: "https://images.unsplash.com/photo-1500648767791-00dcc994a43e?w=400&h=650&fit=crop&crop=face",
      gender: "male",
      aspectRatio: 1.6,
      ...getDefaultEnhancedData("13", "male"),
    },
    {
      id: "14",
      name: "James",
      age: 29,
      country: "Canada",
      imageUrl: "https://images.unsplash.com/photo-1608681299041-cc19878f79f1?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxtYW4lMjBiZWFyZCUyMGNhc3VhbCUyMHBvcnRyYWl0fGVufDF8fHx8MTc1ODE1MTc1OXww&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      gender: "male",
      aspectRatio: 1.3,
      ...getDefaultEnhancedData("14", "male"),
    },
    {
      id: "15",
      name: "Oliver",
      age: 27,
      country: "Netherlands",
      imageUrl: "https://images.unsplash.com/photo-1519085360753-af0119f7cbe7?w=400&h=550&fit=crop&crop=face",
      isOnline: true,
      gender: "male",
      aspectRatio: 1.4,
      ...getDefaultEnhancedData("15", "male"),
    },
  ];

  // Enhanced profile generation with realistic data
  const generateEnhancedProfileData = useCallback((profileIndex: number, baseProfile: Profile) => {
    const occupations = ["Software Engineer", "Marketing Manager", "Teacher", "Nurse", "Artist", "Consultant", "Designer", "Writer", "Photographer", "Chef"];
    const educationLevels = ["High School", "Bachelor's Degree", "Master's Degree", "PhD"];
    const heights = ["5'2\"", "5'3\"", "5'4\"", "5'5\"", "5'6\"", "5'7\"", "5'8\"", "5'9\"", "5'10\"", "6'0\"", "6'1\"", "6'2\""];
    const relationshipTypes = ["Serious relationship", "Casual dating", "Marriage", "Not sure yet", "Companionship"];
    const locations = ["New York", "Los Angeles", "Chicago", "Houston", "Phoenix", "Philadelphia", "San Antonio", "San Diego", "Dallas", "San Jose"];
    
    const interestGroups = [
      ["Technology", "Gaming", "Science", "Innovation"],
      ["Art", "Music", "Dancing", "Theatre"],
      ["Travel", "Adventure", "Hiking", "Photography"],
      ["Fitness", "Yoga", "Sports", "Health"],
      ["Cooking", "Wine", "Food", "Coffee"],
      ["Reading", "Writing", "Movies", "Books"],
      ["Volunteering", "Nature", "Animals", "Environment"]
    ];

    const qualityGroups = [
      ["Humor", "Kindness", "Intelligence"],
      ["Honesty", "Authenticity", "Loyalty"],
      ["Ambition", "Creativity", "Passion"],
      ["Empathy", "Respect", "Communication"]
    ];

    const dealBreakerGroups = [
      ["Smoking", "Heavy drinking"],
      ["Dishonesty", "Disrespectful"],
      ["No ambition", "Negativity"],
      ["Bad hygiene", "Unreliable"]
    ];

    // Random color themes for profiles
    const colorThemes = [
      { primary: '#6366f1', secondary: '#8b5cf6', accent: '#ec4899', gradient1: '#3b82f6', gradient2: '#8b5cf6', gradient3: '#ec4899' },
      { primary: '#f59e0b', secondary: '#ef4444', accent: '#ec4899', gradient1: '#f59e0b', gradient2: '#ef4444', gradient3: '#ec4899' },
      { primary: '#0ea5e9', secondary: '#06b6d4', accent: '#10b981', gradient1: '#0ea5e9', gradient2: '#06b6d4', gradient3: '#10b981' },
      { primary: '#10b981', secondary: '#22c55e', accent: '#84cc16', gradient1: '#10b981', gradient2: '#22c55e', gradient3: '#84cc16' },
      { primary: '#ec4899', secondary: '#a855f7', accent: '#6366f1', gradient1: '#ec4899', gradient2: '#a855f7', gradient3: '#6366f1' },
      { primary: '#374151', secondary: '#6b7280', accent: '#9ca3af', gradient1: '#374151', gradient2: '#6b7280', gradient3: '#9ca3af' }
    ];

    return {
      location: locations[profileIndex % locations.length],
      bio: `Passionate about life and always looking for new adventures. I believe in authentic connections and meaningful conversations.`,
      occupation: occupations[profileIndex % occupations.length],
      education: educationLevels[profileIndex % educationLevels.length],
      height: heights[profileIndex % heights.length],
      isVerified: Math.random() > 0.6,
      isPremium: Math.random() > 0.7,
      interests: interestGroups[profileIndex % interestGroups.length].concat(
        interestGroups[(profileIndex + 1) % interestGroups.length].slice(0, 2)
      ),
      distance: Math.floor(Math.random() * 50) + 1,
      joinDate: "2023",
      colorTheme: colorThemes[profileIndex % colorThemes.length],
      lookingFor: {
        relationshipType: relationshipTypes[profileIndex % relationshipTypes.length],
        lookingForText: "Looking for someone genuine who shares my values and wants to build something meaningful together.",
        ageRange: { min: 25, max: 40 },
        maxDistance: Math.floor(Math.random() * 30) + 20,
        dealBreakers: dealBreakerGroups[profileIndex % dealBreakerGroups.length],
        importantQualities: qualityGroups[profileIndex % qualityGroups.length],
        countriesPreference: Math.random() > 0.7 ? "all" : "specific",
        selectedCountries: Math.random() > 0.5 ? ["United States"] : ["United States", "Canada"]
      }
    };
  }, []);

  // Generate additional profiles by cycling through base profiles with different names and countries
  const generateProfiles = useCallback((count: number, startIndex: number = 0): Profile[] => {
    const names = {
      female: ["Emma", "Olivia", "Ava", "Isabella", "Sophia", "Charlotte", "Mia", "Amelia", "Harper", "Evelyn", "Abigail", "Emily", "Elizabeth", "Mila", "Ella", "Avery", "Sofia", "Camila", "Aria", "Scarlett", "Victoria", "Madison", "Luna", "Grace", "Chloe", "Penelope", "Layla", "Riley", "Zoey", "Nora"],
      male: ["Liam", "Noah", "Oliver", "Elijah", "William", "James", "Benjamin", "Lucas", "Henry", "Alexander", "Mason", "Michael", "Ethan", "Daniel", "Jacob", "Logan", "Jackson", "Levi", "Sebastian", "Mateo", "Jack", "Owen", "Theodore", "Aiden", "Samuel", "Joseph", "John", "David", "Wyatt", "Matthew"]
    };
    const countries = ["USA", "Canada", "UK", "Australia", "Germany", "France", "Spain", "Italy", "Japan", "Brazil", "Mexico", "Netherlands", "Sweden", "Norway", "Denmark"];
    
    return Array.from({ length: count }, (_, i) => {
      const baseProfile = baseProfiles[i % baseProfiles.length];
      const profileIndex = startIndex + i;
      const genderNames = names[baseProfile.gender];
      const enhancedData = generateEnhancedProfileData(profileIndex, baseProfile);
      
      return {
        ...baseProfile,
        ...enhancedData,
        id: `profile-${profileIndex}`,
        name: genderNames[profileIndex % genderNames.length],
        age: 18 + (profileIndex % 25),
        country: countries[profileIndex % countries.length],
        isOnline: Math.random() > 0.7,
      };
    });
  }, [generateEnhancedProfileData]);

  // Filter profiles based on search preference
  const filterProfilesByGender = useCallback((profiles: Profile[]): Profile[] => {
    if (lookingFor === "everyone") return profiles;
    return profiles.filter(profile => 
      (lookingFor === "women" && profile.gender === "female") ||
      (lookingFor === "men" && profile.gender === "male")
    );
  }, [lookingFor]);

  // Initialize profiles
  useEffect(() => {
    const initialProfiles = generateProfiles(100);
    const filteredProfiles = filterProfilesByGender(initialProfiles);
    setAllProfiles(filteredProfiles);
    setDisplayedProfiles(filteredProfiles.slice(0, profilesPerPage));
    setPage(1);
    setHasMore(filteredProfiles.length > profilesPerPage);
  }, [lookingFor, generateProfiles, filterProfilesByGender]);

  // Load more profiles
  const loadMoreProfiles = useCallback(() => {
    if (loading || !hasMore) return;
    
    setLoading(true);
    
    // Simulate API delay
    setTimeout(() => {
      const startIndex = page * profilesPerPage;
      const endIndex = startIndex + profilesPerPage;
      
      if (startIndex >= allProfiles.length) {
        // Generate more profiles if we've reached the end
        const newProfiles = generateProfiles(50, allProfiles.length);
        const filteredNewProfiles = filterProfilesByGender(newProfiles);
        const updatedAllProfiles = [...allProfiles, ...filteredNewProfiles];
        
        setAllProfiles(updatedAllProfiles);
        setDisplayedProfiles(prev => [...prev, ...updatedAllProfiles.slice(startIndex, endIndex)]);
        setHasMore(updatedAllProfiles.length > endIndex);
      } else {
        const newProfiles = allProfiles.slice(startIndex, endIndex);
        setDisplayedProfiles(prev => [...prev, ...newProfiles]);
        setHasMore(allProfiles.length > endIndex);
      }
      
      setPage(prev => prev + 1);
      setLoading(false);
    }, 800);
  }, [loading, hasMore, page, allProfiles, generateProfiles, filterProfilesByGender]);

  // Infinite scroll effect
  useEffect(() => {
    const handleScroll = () => {
      if (window.innerHeight + document.documentElement.scrollTop >= document.documentElement.offsetHeight - 1000) {
        loadMoreProfiles();
      }
    };

    window.addEventListener('scroll', handleScroll);
    return () => window.removeEventListener('scroll', handleScroll);
  }, [loadMoreProfiles]);

  return (
    <div className="py-8">
      <div className="max-w-7xl mx-auto px-6">
        {displayedProfiles.length === 0 && !loading ? (
          <div className="text-center py-12">
            <p className="text-muted-foreground text-lg">No profiles found matching your criteria.</p>
            <p className="text-muted-foreground mt-2">Try adjusting your search filters.</p>
          </div>
        ) : (
          <Masonry columnsCount={5} gutter="20px">
            {displayedProfiles.map((profile) => (
              <ProfileCard key={profile.id} profile={profile} onProfileClick={onProfileClick} currentUser={currentUser} />
            ))}
          </Masonry>
        )}

        {loading && (
          <div className="text-center py-8">
            <div className="w-8 h-8 border-4 border-purple-600 border-t-transparent rounded-full animate-spin mx-auto mb-4"></div>
            <p className="text-muted-foreground">Loading more profiles...</p>
          </div>
        )}
      </div>
    </div>
  );
}

function ProfileCard({ profile, onProfileClick, currentUser }: { 
  profile: Profile; 
  onProfileClick?: (profileId: string) => void;
  currentUser: Profile;
}) {
  const cardHeight = Math.floor(250 + (profile.aspectRatio * 100));
  const { theme: defaultTheme } = useColorTheme();
  
  // Use profile's custom theme or fall back to default
  const profileTheme = profile.colorTheme || defaultTheme;
  
  // Calculate match percentage
  const matchPercentage = getQuickMatchPercentage(currentUser, profile);
  
  // Get match quality color using profile's theme
  const getMatchColor = (percentage: number) => {
    if (percentage >= 80) return profileTheme.primary;
    if (percentage >= 60) return profileTheme.secondary;
    if (percentage >= 40) return profileTheme.accent;
    return profileTheme.gradient3;
  };
  
  return (
    <div 
      className="group relative bg-card rounded-xl overflow-hidden shadow-sm hover:shadow-lg transition-all duration-300 cursor-pointer mb-5 border border-border/50"
      style={{ height: `${cardHeight}px` }}
      onClick={() => onProfileClick?.(profile.id)}
    >
      {/* Profile Image */}
      <div className="relative h-full overflow-hidden">
        <ImageWithFallback
          src={profile.imageUrl}
          alt={`${profile.name}'s profile`}
          className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-300"
        />
        
        {/* Match Percentage Badge */}
        <div className="absolute top-3 left-3">
          <Badge 
            className="text-white flex items-center gap-1 px-2 py-1 border-0"
            style={{ backgroundColor: getMatchColor(matchPercentage) }}
          >
            <Target className="w-3 h-3" />
            {matchPercentage}%
          </Badge>
        </div>

        {/* Online Status */}
        {profile.isOnline && (
          <div 
            className="absolute top-3 right-3 w-3 h-3 rounded-full border-2 border-card shadow-sm"
            style={{ backgroundColor: profileTheme.primary }}
          />
        )}

        {/* Overlay with actions */}
        <div className="absolute inset-0 bg-gradient-to-t from-black/70 via-black/20 to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-300">
          <div className="absolute bottom-4 left-4 right-4">
            {/* Profile Info */}
            <div className="mb-3">
              <h3 className="font-semibold text-white mb-1">
                {profile.name}, {profile.age}
              </h3>
              <p className="text-white/80 text-sm">{profile.country}</p>
            </div>

            {/* Action Buttons */}
            <div className="flex items-center justify-between">
              <div className="flex gap-2">
                <Button 
                  size="sm" 
                  variant="secondary" 
                  className="bg-white/90 hover:bg-white border"
                  style={{ 
                    borderColor: profileTheme.primary,
                    color: profileTheme.primary
                  }}
                >
                  <Heart className="w-4 h-4" />
                </Button>
                <Button 
                  size="sm" 
                  variant="secondary" 
                  className="bg-white/90 hover:bg-white border"
                  style={{ 
                    borderColor: profileTheme.secondary,
                    color: profileTheme.secondary
                  }}
                >
                  <MessageCircle className="w-4 h-4" />
                </Button>
              </div>
              <Button 
                size="sm" 
                variant="secondary" 
                className="bg-white/90 hover:bg-white border"
                style={{ 
                  borderColor: profileTheme.accent,
                  color: profileTheme.accent
                }}
              >
                <Star className="w-4 h-4" />
              </Button>
            </div>
          </div>
        </div>

        {/* Static overlay for profile info when not hovering */}
        <div className="absolute bottom-0 left-0 right-0 bg-gradient-to-t from-black/50 to-transparent group-hover:opacity-0 transition-opacity duration-300">
          <div className="p-4">
            <h3 className="font-semibold text-white mb-1">
              {profile.name}, {profile.age}
            </h3>
            <p className="text-white/80 text-sm">{profile.country}</p>
          </div>
        </div>
      </div>
    </div>
  );
}