import { useState } from "react";
import { Header } from "./Header";
import { Footer } from "./Footer";
import { Button } from "./ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "./ui/card";
import { Input } from "./ui/input";
import { Label } from "./ui/label";
import { Textarea } from "./ui/textarea";
import { Switch } from "./ui/switch";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "./ui/select";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "./ui/tabs";
import { Badge } from "./ui/badge";
import { Avatar, AvatarFallback, AvatarImage } from "./ui/avatar";
import { Separator } from "./ui/separator";
import { Alert, AlertDescription } from "./ui/alert";
import { 
  User, 
  Camera, 
  Shield, 
  Bell, 
  Heart, 
  Crown,
  Trash2,
  Eye,
  EyeOff,
  Globe,
  MapPin,
  Calendar,
  Ruler,
  GraduationCap,
  Briefcase,
  Phone,
  Mail,
  Lock,
  AlertTriangle,
  Save,
  Upload,
  X,
  Plus,
  Video,
  Music,
  ImageIcon,
  Play,
  Pause,
  Volume2,
  VolumeX,
  Sparkles,
  Wand2,
  Bot,
  Search,
  Verified,
  CheckCircle,
  ChevronRight
} from "lucide-react";
import { ImageWithFallback } from "./figma/ImageWithFallback";
import { LocationSelector } from "./LocationSelector";
import { UsernameChangeRequest } from "./UsernameChangeRequest";
import { AIWritingAssistant } from "./AIWritingAssistant";
import { AccountTabWithVerification } from "./AccountTabWithVerification";
import { useTranslation } from "./TranslationProvider";

interface ProfileSettingsPageProps {
  onNavigate: (view: string) => void;
  onOpenCookieConsent?: () => void;
}

interface ProfilePhoto {
  id: string;
  url: string;
  isPrimary: boolean;
}

export function ProfileSettingsPage({ onNavigate, onOpenCookieConsent }: ProfileSettingsPageProps) {
  const { t } = useTranslation();
  
  // Profile State
  const [basicInfo, setBasicInfo] = useState({
    firstName: "Emma",
    lastName: "Johnson",
    profileName: "emma_photographer", // Username from signup
    email: "emma.johnson@email.com",
    phone: "+1 (555) 123-4567",
    dateOfBirth: "1998-05-15",
    height: "5'6\"",
    education: "Bachelor's Degree",
    occupation: "Photographer",
    bio: "Love traveling, photography, and trying new restaurants. Looking for someone to share adventures with!"
  });

  const [location, setLocation] = useState({
    country: "United States",
    city: "New York"
  });

  const [photos, setPhotos] = useState<ProfilePhoto[]>([
    {
      id: "1",
      url: "https://images.unsplash.com/photo-1544005313-94ddf0286df2?w=400&h=600&fit=crop&crop=face",
      isPrimary: true
    },
    {
      id: "2", 
      url: "https://images.unsplash.com/photo-1534528741775-53994a69daeb?w=400&h=600&fit=crop&crop=face",
      isPrimary: false
    },
    {
      id: "3",
      url: "https://images.unsplash.com/photo-1524504388940-b1c1722653e1?w=400&h=600&fit=crop&crop=face",
      isPrimary: false
    }
  ]);

  const [interests, setInterests] = useState([
    "Photography", "Travel", "Food", "Art", "Music", "Dancing"
  ]);

  // What I'm Looking For state
  const [lookingFor, setLookingFor] = useState({
    relationshipType: "Serious relationship",
    lookingForText: "Looking for someone who shares my passion for adventure and creativity. I value honesty, humor, and deep conversations. Someone who appreciates the little things in life and is ready for meaningful connection.",
    ageRange: { min: 25, max: 35 },
    maxDistance: 50,
    dealBreakers: ["Smoking", "No ambition"],
    importantQualities: ["Humor", "Kindness", "Ambition", "Authenticity"],
    countriesPreference: "all", // "all", "specific", "multiple"
    selectedCountries: [] as string[]
  });

  // New Media Settings
  const [mediaSettings, setMediaSettings] = useState({
    introVideo: "",
    introVideoTitle: "",
    profileMusic: "",
    musicTitle: "",
    musicArtist: "",
    customBackground: "",
    backgroundTitle: ""
  });

  // Privacy Settings
  const [privacySettings, setPrivacySettings] = useState({
    profileVisibility: "public",
    showAge: true,
    showLocation: true,
    showOnlineStatus: false,
    allowMessages: "verified",
    showInSearch: true
  });

  // Notification Settings
  const [notifications, setNotifications] = useState({
    newMatches: true,
    messages: true,
    likes: true,
    profileViews: false,
    promotions: false,
    email: true,
    push: true
  });

  // Account Settings
  const [accountSettings, setAccountSettings] = useState({
    twoFactor: false,
    emailVerified: true,
    phoneVerified: false,
    isVerified: false, // Identity verification status
    verificationStatus: 'not_started' // 'not_started', 'pending', 'approved', 'rejected'
  });

  const [activeTab, setActiveTab] = useState("profile");
  const [saveStatus, setSaveStatus] = useState<"idle" | "saving" | "saved" | "error">("idle");
  const [isAIAssistantOpen, setIsAIAssistantOpen] = useState(false);
  const [aiAssistantType, setAiAssistantType] = useState<"bio" | "interests" | "looking-for">("bio");

  const handleSave = async () => {
    setSaveStatus("saving");
    // Simulate API call
    setTimeout(() => {
      setSaveStatus("saved");
      setTimeout(() => setSaveStatus("idle"), 2000);
    }, 1000);
  };

  const handlePhotoUpload = () => {
    // In a real app, this would open a file picker
    console.log("Opening photo upload dialog");
  };

  const handlePhotoDelete = (photoId: string) => {
    setPhotos(photos.filter(photo => photo.id !== photoId));
  };

  const handleSetPrimaryPhoto = (photoId: string) => {
    setPhotos(photos.map(photo => ({
      ...photo,
      isPrimary: photo.id === photoId
    })));
  };

  const handleAddInterest = (interest: string) => {
    if (interest && !interests.includes(interest)) {
      setInterests([...interests, interest]);
    }
  };

  const handleRemoveInterest = (interest: string) => {
    setInterests(interests.filter(i => i !== interest));
  };

  const availableInterests = [
    "Art", "Music", "Travel", "Photography", "Cooking", "Fitness", "Reading", "Dancing",
    "Movies", "Sports", "Gaming", "Nature", "Fashion", "Technology", "Yoga", "Wine",
    "Coffee", "Adventure", "Theatre", "Science", "History", "Languages", "Volunteering"
  ];

  // Media handlers
  const handleVideoUpload = () => {
    // In a real app, this would open a file picker for video files
    console.log("Opening video upload dialog");
  };

  const handleMusicUpload = () => {
    // In a real app, this would open a file picker for audio files
    console.log("Opening music upload dialog");
  };

  const handleBackgroundUpload = () => {
    // In a real app, this would open a file picker for images
    console.log("Opening background upload dialog");
  };

  const handleRemoveVideo = () => {
    setMediaSettings({...mediaSettings, introVideo: "", introVideoTitle: ""});
  };

  const handleRemoveMusic = () => {
    setMediaSettings({...mediaSettings, profileMusic: "", musicTitle: "", musicArtist: ""});
  };

  const handleRemoveBackground = () => {
    setMediaSettings({...mediaSettings, customBackground: "", backgroundTitle: ""});
  };

  // AI Assistant handlers
  const openAIAssistant = (type: "bio" | "interests" | "looking-for") => {
    setAiAssistantType(type);
    setIsAIAssistantOpen(true);
  };

  const handleAITextApply = (text: string) => {
    if (aiAssistantType === "bio") {
      setBasicInfo({...basicInfo, bio: text});
    } else if (aiAssistantType === "interests") {
      // For interests, parse the comma-separated text into array
      const newInterests = text.split(",").map(interest => interest.trim()).filter(Boolean);
      setInterests(newInterests);
    } else if (aiAssistantType === "looking-for") {
      setLookingFor({...lookingFor, lookingForText: text});
    }
  };

  // Looking for handlers
  const handleAddDealBreaker = (dealBreaker: string) => {
    if (dealBreaker && !lookingFor.dealBreakers.includes(dealBreaker)) {
      setLookingFor({
        ...lookingFor,
        dealBreakers: [...lookingFor.dealBreakers, dealBreaker]
      });
    }
  };

  const handleRemoveDealBreaker = (dealBreaker: string) => {
    setLookingFor({
      ...lookingFor,
      dealBreakers: lookingFor.dealBreakers.filter(item => item !== dealBreaker)
    });
  };

  const handleAddImportantQuality = (quality: string) => {
    if (quality && !lookingFor.importantQualities.includes(quality)) {
      setLookingFor({
        ...lookingFor,
        importantQualities: [...lookingFor.importantQualities, quality]
      });
    }
  };

  const handleRemoveImportantQuality = (quality: string) => {
    setLookingFor({
      ...lookingFor,
      importantQualities: lookingFor.importantQualities.filter(item => item !== quality)
    });
  };

  const availableDealBreakers = [
    "Smoking", "Heavy drinking", "No ambition", "Dishonesty", "Bad hygiene",
    "Negativity", "No sense of humor", "Disrespectful", "Unreliable", "Closed-minded"
  ];

  const availableQualities = [
    "Humor", "Kindness", "Intelligence", "Ambition", "Honesty", "Loyalty",
    "Creativity", "Empathy", "Confidence", "Authenticity", "Passion", "Respect",
    "Communication", "Adventurous", "Family-oriented", "Spiritual", "Independent", "Caring"
  ];

  // Available countries for selection
  const availableCountries = [
    "United States", "Canada", "United Kingdom", "Australia", "Germany", "France", 
    "Italy", "Spain", "Netherlands", "Sweden", "Norway", "Denmark", "Finland",
    "Switzerland", "Austria", "Belgium", "Ireland", "Portugal", "Greece", "Poland",
    "Czech Republic", "Hungary", "Romania", "Bulgaria", "Croatia", "Slovenia",
    "Slovakia", "Estonia", "Latvia", "Lithuania", "Luxembourg", "Malta", "Cyprus",
    "Japan", "South Korea", "Singapore", "New Zealand", "Brazil", "Argentina",
    "Chile", "Mexico", "Colombia", "Peru", "Uruguay", "Costa Rica", "Panama"
  ];

  // Country preference handlers
  const handleCountryPreferenceChange = (preference: "all" | "specific" | "multiple") => {
    setLookingFor({
      ...lookingFor,
      countriesPreference: preference,
      selectedCountries: preference === "all" ? [] : lookingFor.selectedCountries
    });
  };

  const handleAddCountry = (country: string) => {
    if (country && !lookingFor.selectedCountries.includes(country)) {
      setLookingFor({
        ...lookingFor,
        selectedCountries: [...lookingFor.selectedCountries, country]
      });
    }
  };

  const handleRemoveCountry = (country: string) => {
    setLookingFor({
      ...lookingFor,
      selectedCountries: lookingFor.selectedCountries.filter(c => c !== country)
    });
  };

  return (
    <div className="min-h-screen bg-background">
      <Header currentView="profile-settings" onNavigate={onNavigate} />
      
      <div className="max-w-6xl mx-auto px-6 py-8">
        <div className="mb-8">
          <h1 className="text-3xl font-bold mb-2">Profile Settings</h1>
          <p className="text-muted-foreground">
            Manage your profile information, privacy settings, and account preferences
          </p>
        </div>

        <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-6">
          <TabsList className="grid w-full grid-cols-4 lg:grid-cols-7">
            <TabsTrigger value="profile" className="flex items-center gap-2">
              <User className="w-4 h-4" />
              <span className="hidden sm:inline">Profile</span>
            </TabsTrigger>
            <TabsTrigger value="username" className="flex items-center gap-2">
              <User className="w-4 h-4" />
              <span className="hidden sm:inline">Username</span>
            </TabsTrigger>
            <TabsTrigger value="photos" className="flex items-center gap-2">
              <Camera className="w-4 h-4" />
              <span className="hidden sm:inline">Photos</span>
            </TabsTrigger>
            <TabsTrigger value="media" className="flex items-center gap-2">
              <Video className="w-4 h-4" />
              <span className="hidden sm:inline">Media</span>
            </TabsTrigger>
            <TabsTrigger value="privacy" className="flex items-center gap-2">
              <Shield className="w-4 h-4" />
              <span className="hidden sm:inline">Privacy</span>
            </TabsTrigger>
            <TabsTrigger value="notifications" className="flex items-center gap-2">
              <Bell className="w-4 h-4" />
              <span className="hidden sm:inline">Notifications</span>
            </TabsTrigger>
            <TabsTrigger value="account" className="flex items-center gap-2">
              <Lock className="w-4 h-4" />
              <span className="hidden sm:inline">Account</span>
            </TabsTrigger>
          </TabsList>

          {/* Profile Tab */}
          <TabsContent value="profile" className="space-y-6">
            <div className="grid gap-6 lg:grid-cols-2">
              {/* Basic Information */}
              <Card>
                <CardHeader>
                  <CardTitle>Basic Information</CardTitle>
                  <CardDescription>
                    Update your personal details and profile information
                  </CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <Label htmlFor="firstName">First Name</Label>
                      <Input
                        id="firstName"
                        value={basicInfo.firstName}
                        onChange={(e) => setBasicInfo({...basicInfo, firstName: e.target.value})}
                      />
                    </div>
                    <div>
                      <Label htmlFor="lastName">Last Name</Label>
                      <Input
                        id="lastName"
                        value={basicInfo.lastName}
                        onChange={(e) => setBasicInfo({...basicInfo, lastName: e.target.value})}
                      />
                    </div>
                  </div>
                  
                  <div>
                    <Label htmlFor="email">Email</Label>
                    <Input
                      id="email"
                      type="email"
                      value={basicInfo.email}
                      onChange={(e) => setBasicInfo({...basicInfo, email: e.target.value})}
                    />
                  </div>
                  
                  <div>
                    <Label htmlFor="phone">Phone Number</Label>
                    <Input
                      id="phone"
                      value={basicInfo.phone}
                      onChange={(e) => setBasicInfo({...basicInfo, phone: e.target.value})}
                    />
                  </div>
                  
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <Label htmlFor="dateOfBirth">Date of Birth</Label>
                      <Input
                        id="dateOfBirth"
                        type="date"
                        value={basicInfo.dateOfBirth}
                        onChange={(e) => setBasicInfo({...basicInfo, dateOfBirth: e.target.value})}
                      />
                    </div>
                    <div>
                      <Label htmlFor="height">Height</Label>
                      <Input
                        id="height"
                        value={basicInfo.height}
                        onChange={(e) => setBasicInfo({...basicInfo, height: e.target.value})}
                      />
                    </div>
                  </div>
                </CardContent>
              </Card>

              {/* Professional & Education */}
              <Card>
                <CardHeader>
                  <CardTitle>Professional & Education</CardTitle>
                  <CardDescription>
                    Share your educational background and career
                  </CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <Label htmlFor="education">Education</Label>
                    <Select value={basicInfo.education} onValueChange={(value) => setBasicInfo({...basicInfo, education: value})}>
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="High School">High School</SelectItem>
                        <SelectItem value="Some College">Some College</SelectItem>
                        <SelectItem value="Associate's Degree">Associate's Degree</SelectItem>
                        <SelectItem value="Bachelor's Degree">Bachelor's Degree</SelectItem>
                        <SelectItem value="Master's Degree">Master's Degree</SelectItem>
                        <SelectItem value="PhD">PhD</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                  
                  <div>
                    <Label htmlFor="occupation">Occupation</Label>
                    <Input
                      id="occupation"
                      value={basicInfo.occupation}
                      onChange={(e) => setBasicInfo({...basicInfo, occupation: e.target.value})}
                    />
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* Location */}
            <Card>
              <CardHeader>
                <CardTitle>Location</CardTitle>
                <CardDescription>
                  Update your location to find matches nearby
                </CardDescription>
              </CardHeader>
              <CardContent>
                <LocationSelector
                  country={location.country}
                  city={location.city}
                  onCountryChange={(country) => setLocation(prev => ({...prev, country}))}
                  onCityChange={(city) => setLocation(prev => ({...prev, city}))}
                />
              </CardContent>
            </Card>

            {/* Bio & Interests */}
            <Card>
              <CardHeader>
                <CardTitle>About Me</CardTitle>
                <CardDescription>
                  Tell others about yourself and your interests
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <div className="flex items-center justify-between mb-2">
                    <Label htmlFor="bio">Bio</Label>
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => openAIAssistant("bio")}
                      className="text-purple-600 border-purple-200 hover:bg-purple-50 dark:border-purple-800 dark:hover:bg-purple-900/20"
                    >
                      <Sparkles className="w-4 h-4 mr-2" />
                      AI Help
                    </Button>
                  </div>
                  <Textarea
                    id="bio"
                    value={basicInfo.bio}
                    onChange={(e) => setBasicInfo({...basicInfo, bio: e.target.value})}
                    placeholder="Tell others about yourself..."
                    className="min-h-[100px]"
                  />
                  <div className="flex items-center justify-between mt-1">
                    <p className="text-sm text-muted-foreground">
                      {basicInfo.bio.length}/500 characters
                    </p>
                    {basicInfo.bio.length > 0 && (
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => openAIAssistant("bio")}
                        className="text-xs text-purple-600 hover:text-purple-700"
                      >
                        <Bot className="w-3 h-3 mr-1" />
                        Improve with AI
                      </Button>
                    )}
                  </div>
                </div>
                
                <div>
                  <div className="flex items-center justify-between mb-2">
                    <Label>Interests</Label>
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => openAIAssistant("interests")}
                      className="text-purple-600 border-purple-200 hover:bg-purple-50 dark:border-purple-800 dark:hover:bg-purple-900/20"
                    >
                      <Wand2 className="w-4 h-4 mr-2" />
                      AI Suggestions
                    </Button>
                  </div>
                  <div className="flex flex-wrap gap-2 mt-2 mb-4">
                    {interests.map((interest) => (
                      <Badge key={interest} variant="secondary" className="flex items-center gap-1">
                        {interest}
                        <button
                          onClick={() => handleRemoveInterest(interest)}
                          className="ml-1 hover:text-destructive"
                        >
                          <X className="w-3 h-3" />
                        </button>
                      </Badge>
                    ))}
                  </div>
                  
                  <div className="flex gap-2">
                    <Select onValueChange={handleAddInterest}>
                      <SelectTrigger>
                        <SelectValue placeholder="Add an interest" />
                      </SelectTrigger>
                      <SelectContent>
                        {availableInterests
                          .filter(interest => !interests.includes(interest))
                          .map((interest) => (
                            <SelectItem key={interest} value={interest}>
                              {interest}
                            </SelectItem>
                          ))}
                      </SelectContent>
                    </Select>
                    {interests.length > 0 && (
                      <Button
                        variant="outline"
                        onClick={() => openAIAssistant("interests")}
                        className="text-purple-600 border-purple-200 hover:bg-purple-50 dark:border-purple-800 dark:hover:bg-purple-900/20"
                      >
                        <Sparkles className="w-4 h-4" />
                      </Button>
                    )}
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* What I'm Looking For */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Search className="w-5 h-5" />
                  What I'm Looking For
                </CardTitle>
                <CardDescription>
                  Share what you're seeking in a relationship and partner
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-6">
                {/* Relationship Type */}
                <div>
                  <Label htmlFor="relationshipType">Relationship Type</Label>
                  <Select value={lookingFor.relationshipType} onValueChange={(value) => setLookingFor({...lookingFor, relationshipType: value})}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="Casual dating">Casual dating</SelectItem>
                      <SelectItem value="Serious relationship">Serious relationship</SelectItem>
                      <SelectItem value="Marriage">Marriage</SelectItem>
                      <SelectItem value="Friendship">Friendship</SelectItem>
                      <SelectItem value="Not sure yet">Not sure yet</SelectItem>
                      <SelectItem value="Companionship">Companionship</SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                {/* Looking For Text */}
                <div>
                  <div className="flex items-center justify-between mb-2">
                    <Label htmlFor="lookingForText">What I'm Looking For</Label>
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => openAIAssistant("looking-for")}
                      className="text-purple-600 border-purple-200 hover:bg-purple-50 dark:border-purple-800 dark:hover:bg-purple-900/20"
                    >
                      <Sparkles className="w-4 h-4 mr-2" />
                      AI Help
                    </Button>
                  </div>
                  <Textarea
                    id="lookingForText"
                    value={lookingFor.lookingForText}
                    onChange={(e) => setLookingFor({...lookingFor, lookingForText: e.target.value})}
                    placeholder="Describe what you're looking for in a partner and relationship..."
                    className="min-h-[100px]"
                  />
                  <div className="flex items-center justify-between mt-1">
                    <p className="text-sm text-muted-foreground">
                      {lookingFor.lookingForText.length}/400 characters
                    </p>
                    {lookingFor.lookingForText.length > 0 && (
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => openAIAssistant("looking-for")}
                        className="text-xs text-purple-600 hover:text-purple-700"
                      >
                        <Bot className="w-3 h-3 mr-1" />
                        Improve with AI
                      </Button>
                    )}
                  </div>
                </div>

                {/* Age Range and Distance */}
                <div className="grid grid-cols-2 gap-6">
                  <div>
                    <Label>Age Range</Label>
                    <div className="grid grid-cols-2 gap-2 mt-2">
                      <div>
                        <Label htmlFor="minAge" className="text-sm text-muted-foreground">Min</Label>
                        <Input
                          id="minAge"
                          type="number"
                          min="18"
                          max="99"
                          value={lookingFor.ageRange.min}
                          onChange={(e) => setLookingFor({
                            ...lookingFor,
                            ageRange: { ...lookingFor.ageRange, min: parseInt(e.target.value) || 18 }
                          })}
                        />
                      </div>
                      <div>
                        <Label htmlFor="maxAge" className="text-sm text-muted-foreground">Max</Label>
                        <Input
                          id="maxAge"
                          type="number"
                          min="18"
                          max="99"
                          value={lookingFor.ageRange.max}
                          onChange={(e) => setLookingFor({
                            ...lookingFor,
                            ageRange: { ...lookingFor.ageRange, max: parseInt(e.target.value) || 99 }
                          })}
                        />
                      </div>
                    </div>
                  </div>

                  <div>
                    <Label htmlFor="maxDistance">Maximum Distance (miles)</Label>
                    <Input
                      id="maxDistance"
                      type="number"
                      min="1"
                      max="500"
                      value={lookingFor.maxDistance}
                      onChange={(e) => setLookingFor({...lookingFor, maxDistance: parseInt(e.target.value) || 50})}
                      className="mt-2"
                    />
                  </div>
                </div>

                {/* Important Qualities */}
                <div>
                  <Label>Important Qualities</Label>
                  <div className="flex flex-wrap gap-2 mt-2 mb-4">
                    {lookingFor.importantQualities.map((quality) => (
                      <Badge key={quality} variant="outline" className="flex items-center gap-1 bg-green-50 border-green-200 text-green-700 dark:bg-green-900/20 dark:border-green-800 dark:text-green-400">
                        {quality}
                        <button
                          onClick={() => handleRemoveImportantQuality(quality)}
                          className="ml-1 hover:text-destructive"
                        >
                          <X className="w-3 h-3" />
                        </button>
                      </Badge>
                    ))}
                  </div>
                  <Select onValueChange={handleAddImportantQuality}>
                    <SelectTrigger>
                      <SelectValue placeholder="Add an important quality" />
                    </SelectTrigger>
                    <SelectContent>
                      {availableQualities
                        .filter(quality => !lookingFor.importantQualities.includes(quality))
                        .map((quality) => (
                          <SelectItem key={quality} value={quality}>
                            {quality}
                          </SelectItem>
                        ))}
                    </SelectContent>
                  </Select>
                </div>

                {/* Deal Breakers */}
                <div>
                  <Label>Deal Breakers</Label>
                  <div className="flex flex-wrap gap-2 mt-2 mb-4">
                    {lookingFor.dealBreakers.map((dealBreaker) => (
                      <Badge key={dealBreaker} variant="outline" className="flex items-center gap-1 bg-red-50 border-red-200 text-red-700 dark:bg-red-900/20 dark:border-red-800 dark:text-red-400">
                        {dealBreaker}
                        <button
                          onClick={() => handleRemoveDealBreaker(dealBreaker)}
                          className="ml-1 hover:text-destructive"
                        >
                          <X className="w-3 h-3" />
                        </button>
                      </Badge>
                    ))}
                  </div>
                  <Select onValueChange={handleAddDealBreaker}>
                    <SelectTrigger>
                      <SelectValue placeholder="Add a deal breaker" />
                    </SelectTrigger>
                    <SelectContent>
                      {availableDealBreakers
                        .filter(dealBreaker => !lookingFor.dealBreakers.includes(dealBreaker))
                        .map((dealBreaker) => (
                          <SelectItem key={dealBreaker} value={dealBreaker}>
                            {dealBreaker}
                          </SelectItem>
                        ))}
                    </SelectContent>
                  </Select>
                </div>

                {/* Country Preferences */}
                <div>
                  <Label>Country Preferences</Label>
                  <div className="space-y-4 mt-2">
                    <div>
                      <Label htmlFor="countryPreference" className="text-sm text-muted-foreground">Looking for matches in:</Label>
                      <Select value={lookingFor.countriesPreference} onValueChange={handleCountryPreferenceChange}>
                        <SelectTrigger>
                          <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="all">All Countries</SelectItem>
                          <SelectItem value="specific">Specific Country</SelectItem>
                          <SelectItem value="multiple">Multiple Countries</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>

                    {(lookingFor.countriesPreference === "specific" || lookingFor.countriesPreference === "multiple") && (
                      <div>
                        <div className="flex flex-wrap gap-2 mt-2 mb-4">
                          {lookingFor.selectedCountries.map((country) => (
                            <Badge key={country} variant="outline" className="flex items-center gap-1 bg-blue-50 border-blue-200 text-blue-700 dark:bg-blue-900/20 dark:border-blue-800 dark:text-blue-400">
                              {country}
                              <button
                                onClick={() => handleRemoveCountry(country)}
                                className="ml-1 hover:text-destructive"
                              >
                                <X className="w-3 h-3" />
                              </button>
                            </Badge>
                          ))}
                        </div>
                        
                        {(lookingFor.countriesPreference === "multiple" || 
                          (lookingFor.countriesPreference === "specific" && lookingFor.selectedCountries.length === 0)) && (
                          <Select onValueChange={handleAddCountry}>
                            <SelectTrigger>
                              <SelectValue placeholder={lookingFor.countriesPreference === "specific" ? "Select a country" : "Add another country"} />
                            </SelectTrigger>
                            <SelectContent>
                              {availableCountries
                                .filter(country => !lookingFor.selectedCountries.includes(country))
                                .map((country) => (
                                  <SelectItem key={country} value={country}>
                                    {country}
                                  </SelectItem>
                                ))}
                            </SelectContent>
                          </Select>
                        )}

                        {lookingFor.countriesPreference === "specific" && lookingFor.selectedCountries.length >= 1 && (
                          <p className="text-sm text-muted-foreground">
                            You can select one country for specific country preference.
                          </p>
                        )}
                      </div>
                    )}
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Save Button */}
            <Card>
              <CardContent className="pt-6">
                <Button 
                  onClick={handleSave}
                  disabled={saveStatus === "saving"}
                  className="w-full"
                  size="lg"
                >
                  {saveStatus === "saving" && <Save className="w-4 h-4 mr-2 animate-spin" />}
                  {saveStatus === "saved" && <Save className="w-4 h-4 mr-2" />}
                  {saveStatus === "idle" && <Save className="w-4 h-4 mr-2" />}
                  {saveStatus === "saving" ? "Saving Changes..." : 
                   saveStatus === "saved" ? "Changes Saved!" : 
                   "Save Profile Changes"}
                </Button>
                {saveStatus === "saved" && (
                  <p className="text-sm text-green-600 text-center mt-2">
                    Your profile has been updated successfully!
                  </p>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          {/* Username Tab */}
          <TabsContent value="username" className="space-y-6">
            <UsernameChangeRequest 
              currentUsername={basicInfo.profileName}
              onRequestSubmitted={(newUsername, reason) => {
                console.log(`Username change requested: ${newUsername} - Reason: ${reason}`);
                // In a real app, this would make an API call to submit the request
              }}
            />
          </TabsContent>

          {/* Photos Tab */}
          <TabsContent value="photos" className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle>Profile Photos</CardTitle>
                <CardDescription>
                  Upload and manage your profile photos. Your primary photo will be shown first.
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                  {photos.map((photo) => (
                    <div key={photo.id} className="relative group">
                      <div className="aspect-square overflow-hidden rounded-lg border">
                        <ImageWithFallback
                          src={photo.url}
                          alt="Profile photo"
                          className="w-full h-full object-cover"
                        />
                      </div>
                      
                      {photo.isPrimary && (
                        <Badge className="absolute top-2 left-2 bg-green-500">
                          Primary
                        </Badge>
                      )}
                      
                      <div className="absolute inset-0 bg-black/50 opacity-0 group-hover:opacity-100 transition-opacity rounded-lg flex items-center justify-center gap-2">
                        {!photo.isPrimary && (
                          <Button
                            size="sm"
                            variant="secondary"
                            onClick={() => handleSetPrimaryPhoto(photo.id)}
                          >
                            Set Primary
                          </Button>
                        )}
                        <Button
                          size="sm"
                          variant="destructive"
                          onClick={() => handlePhotoDelete(photo.id)}
                        >
                          <Trash2 className="w-4 h-4" />
                        </Button>
                      </div>
                    </div>
                  ))}
                  
                  {photos.length < 9 && (
                    <button
                      onClick={handlePhotoUpload}
                      className="aspect-square border-2 border-dashed border-muted-foreground/25 rounded-lg flex flex-col items-center justify-center gap-2 hover:border-muted-foreground/50 transition-colors"
                    >
                      <Upload className="w-8 h-8 text-muted-foreground" />
                      <span className="text-sm text-muted-foreground">Add Photo</span>
                    </button>
                  )}
                </div>
                
                <Alert className="mt-4">
                  <Camera className="w-4 h-4" />
                  <AlertDescription>
                    You can upload up to 9 photos. High-quality photos with good lighting work best!
                  </AlertDescription>
                </Alert>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Media Tab */}
          <TabsContent value="media" className="space-y-6">
            {/* Introduction Video */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Video className="w-5 h-5" />
                  Introduction Video
                </CardTitle>
                <CardDescription>
                  Upload a 30-second video to introduce yourself to potential matches. This helps others get to know your personality!
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                {mediaSettings.introVideo ? (
                  <div className="space-y-4">
                    <div className="relative aspect-video bg-black rounded-lg overflow-hidden">
                      <video
                        src={mediaSettings.introVideo}
                        className="w-full h-full object-cover"
                        controls
                        preload="metadata"
                      />
                    </div>
                    <div className="flex items-center justify-between">
                      <div>
                        <Label htmlFor="videoTitle">Video Title</Label>
                        <Input
                          id="videoTitle"
                          value={mediaSettings.introVideoTitle}
                          onChange={(e) => setMediaSettings({...mediaSettings, introVideoTitle: e.target.value})}
                          placeholder="e.g., My Introduction"
                          className="mt-1"
                        />
                      </div>
                      <Button variant="destructive" onClick={handleRemoveVideo}>
                        <Trash2 className="w-4 h-4 mr-2" />
                        Remove Video
                      </Button>
                    </div>
                  </div>
                ) : (
                  <div className="border-2 border-dashed border-muted-foreground/25 rounded-lg p-8 text-center">
                    <Video className="w-12 h-12 text-muted-foreground mx-auto mb-4" />
                    <h3 className="text-lg font-medium mb-2">No introduction video uploaded</h3>
                    <p className="text-muted-foreground mb-4">
                      Share your personality with a 30-second introduction video
                    </p>
                    <Button onClick={handleVideoUpload}>
                      <Upload className="w-4 h-4 mr-2" />
                      Upload Video
                    </Button>
                  </div>
                )}
                
                <Alert>
                  <Video className="w-4 h-4" />
                  <AlertDescription>
                    Video must be under 30 seconds and in MP4 format. Keep it engaging and authentic!
                  </AlertDescription>
                </Alert>
              </CardContent>
            </Card>

            {/* Profile Music */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Music className="w-5 h-5" />
                  Profile Music
                </CardTitle>
                <CardDescription>
                  Add background music that plays when someone visits your profile. Choose a song that represents you!
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                {mediaSettings.profileMusic ? (
                  <div className="space-y-4">
                    <div className="bg-muted/50 rounded-lg p-4">
                      <div className="flex items-center justify-between mb-3">
                        <div className="flex items-center gap-3">
                          <div className="w-12 h-12 bg-purple-100 dark:bg-purple-900/20 rounded-lg flex items-center justify-center">
                            <Music className="w-6 h-6 text-purple-600" />
                          </div>
                          <div>
                            <h4 className="font-medium">{mediaSettings.musicTitle || "Untitled"}</h4>
                            <p className="text-sm text-muted-foreground">{mediaSettings.musicArtist || "Unknown Artist"}</p>
                          </div>
                        </div>
                        <div className="flex items-center gap-2">
                          <Button size="sm" variant="outline">
                            <Play className="w-4 h-4" />
                          </Button>
                          <Button size="sm" variant="destructive" onClick={handleRemoveMusic}>
                            <Trash2 className="w-4 h-4" />
                          </Button>
                        </div>
                      </div>
                      <audio src={mediaSettings.profileMusic} controls className="w-full" />
                    </div>
                    
                    <div className="grid grid-cols-2 gap-4">
                      <div>
                        <Label htmlFor="musicTitle">Song Title</Label>
                        <Input
                          id="musicTitle"
                          value={mediaSettings.musicTitle}
                          onChange={(e) => setMediaSettings({...mediaSettings, musicTitle: e.target.value})}
                          placeholder="Enter song title"
                        />
                      </div>
                      <div>
                        <Label htmlFor="musicArtist">Artist</Label>
                        <Input
                          id="musicArtist"
                          value={mediaSettings.musicArtist}
                          onChange={(e) => setMediaSettings({...mediaSettings, musicArtist: e.target.value})}
                          placeholder="Enter artist name"
                        />
                      </div>
                    </div>
                  </div>
                ) : (
                  <div className="border-2 border-dashed border-muted-foreground/25 rounded-lg p-8 text-center">
                    <Music className="w-12 h-12 text-muted-foreground mx-auto mb-4" />
                    <h3 className="text-lg font-medium mb-2">No profile music uploaded</h3>
                    <p className="text-muted-foreground mb-4">
                      Add background music to make your profile more engaging
                    </p>
                    <Button onClick={handleMusicUpload}>
                      <Upload className="w-4 h-4 mr-2" />
                      Upload Music
                    </Button>
                  </div>
                )}
                
                <Alert>
                  <Music className="w-4 h-4" />
                  <AlertDescription>
                    Music should be in MP3 or WAV format. Choose appropriate content that represents your personality.
                  </AlertDescription>
                </Alert>
              </CardContent>
            </Card>

            {/* Custom Background */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <ImageIcon className="w-5 h-5" />
                  Custom Background
                </CardTitle>
                <CardDescription>
                  Set a custom background image for your profile page. This appears subtly behind your profile content.
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                {mediaSettings.customBackground ? (
                  <div className="space-y-4">
                    <div className="relative aspect-video rounded-lg overflow-hidden border">
                      <ImageWithFallback
                        src={mediaSettings.customBackground}
                        alt="Custom background preview"
                        className="w-full h-full object-cover"
                      />
                      <div className="absolute inset-0 bg-gradient-to-t from-black/50 to-transparent" />
                      <div className="absolute bottom-4 left-4 text-white">
                        <p className="text-sm opacity-90">Background Preview</p>
                      </div>
                    </div>
                    
                    <div className="flex items-center justify-between">
                      <div className="flex-1 mr-4">
                        <Label htmlFor="backgroundTitle">Background Title</Label>
                        <Input
                          id="backgroundTitle"
                          value={mediaSettings.backgroundTitle}
                          onChange={(e) => setMediaSettings({...mediaSettings, backgroundTitle: e.target.value})}
                          placeholder="e.g., My Favorite Sunset"
                          className="mt-1"
                        />
                      </div>
                      <Button variant="destructive" onClick={handleRemoveBackground}>
                        <Trash2 className="w-4 h-4 mr-2" />
                        Remove Background
                      </Button>
                    </div>
                  </div>
                ) : (
                  <div className="border-2 border-dashed border-muted-foreground/25 rounded-lg p-8 text-center">
                    <ImageIcon className="w-12 h-12 text-muted-foreground mx-auto mb-4" />
                    <h3 className="text-lg font-medium mb-2">No custom background set</h3>
                    <p className="text-muted-foreground mb-4">
                      Add a personal touch with a custom background image
                    </p>
                    <Button onClick={handleBackgroundUpload}>
                      <Upload className="w-4 h-4 mr-2" />
                      Upload Background
                    </Button>
                  </div>
                )}
                
                <Alert>
                  <ImageIcon className="w-4 h-4" />
                  <AlertDescription>
                    Background images should be high resolution (1920x1080 or higher) and appropriate content. The image will appear subtly behind your profile.
                  </AlertDescription>
                </Alert>
              </CardContent>
            </Card>

            {/* Media Settings Summary */}
            <Card>
              <CardHeader>
                <CardTitle>Media Features Summary</CardTitle>
                <CardDescription>
                  Overview of your active media features
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                  <div className="flex items-center gap-3 p-3 rounded-lg bg-muted/50">
                    <Video className={`w-8 h-8 ${mediaSettings.introVideo ? 'text-green-600' : 'text-muted-foreground'}`} />
                    <div>
                      <p className="font-medium">Introduction Video</p>
                      <p className="text-sm text-muted-foreground">
                        {mediaSettings.introVideo ? 'Active' : 'Not set'}
                      </p>
                    </div>
                  </div>
                  
                  <div className="flex items-center gap-3 p-3 rounded-lg bg-muted/50">
                    <Music className={`w-8 h-8 ${mediaSettings.profileMusic ? 'text-purple-600' : 'text-muted-foreground'}`} />
                    <div>
                      <p className="font-medium">Profile Music</p>
                      <p className="text-sm text-muted-foreground">
                        {mediaSettings.profileMusic ? 'Active' : 'Not set'}
                      </p>
                    </div>
                  </div>
                  
                  <div className="flex items-center gap-3 p-3 rounded-lg bg-muted/50">
                    <ImageIcon className={`w-8 h-8 ${mediaSettings.customBackground ? 'text-blue-600' : 'text-muted-foreground'}`} />
                    <div>
                      <p className="font-medium">Custom Background</p>
                      <p className="text-sm text-muted-foreground">
                        {mediaSettings.customBackground ? 'Active' : 'Not set'}
                      </p>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Privacy Tab */}
          <TabsContent value="privacy" className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle>Profile Visibility</CardTitle>
                <CardDescription>
                  Control who can see your profile and personal information
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-6">
                <div className="flex items-center justify-between">
                  <div>
                    <Label>Profile Visibility</Label>
                    <p className="text-sm text-muted-foreground">Who can see your profile</p>
                  </div>
                  <Select
                    value={privacySettings.profileVisibility}
                    onValueChange={(value) => setPrivacySettings({...privacySettings, profileVisibility: value})}
                  >
                    <SelectTrigger className="w-40">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="public">Everyone</SelectItem>
                      <SelectItem value="members">Members Only</SelectItem>
                      <SelectItem value="verified">Verified Only</SelectItem>
                      <SelectItem value="private">Private</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                
                <Separator />
                
                <div className="space-y-4">
                  <div className="flex items-center justify-between">
                    <div>
                      <Label>Show Age</Label>
                      <p className="text-sm text-muted-foreground">Display your age on your profile</p>
                    </div>
                    <Switch
                      checked={privacySettings.showAge}
                      onCheckedChange={(checked) => setPrivacySettings({...privacySettings, showAge: checked})}
                    />
                  </div>
                  
                  <div className="flex items-center justify-between">
                    <div>
                      <Label>Show Location</Label>
                      <p className="text-sm text-muted-foreground">Display your city and country</p>
                    </div>
                    <Switch
                      checked={privacySettings.showLocation}
                      onCheckedChange={(checked) => setPrivacySettings({...privacySettings, showLocation: checked})}
                    />
                  </div>
                  
                  <div className="flex items-center justify-between">
                    <div>
                      <Label>Show Online Status</Label>
                      <p className="text-sm text-muted-foreground">Let others see when you're online</p>
                    </div>
                    <Switch
                      checked={privacySettings.showOnlineStatus}
                      onCheckedChange={(checked) => setPrivacySettings({...privacySettings, showOnlineStatus: checked})}
                    />
                  </div>
                  
                  <div className="flex items-center justify-between">
                    <div>
                      <Label>Show in Search Results</Label>
                      <p className="text-sm text-muted-foreground">Appear in other users' searches</p>
                    </div>
                    <Switch
                      checked={privacySettings.showInSearch}
                      onCheckedChange={(checked) => setPrivacySettings({...privacySettings, showInSearch: checked})}
                    />
                  </div>
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Message Settings</CardTitle>
                <CardDescription>
                  Control who can send you messages
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="flex items-center justify-between">
                  <div>
                    <Label>Allow Messages From</Label>
                    <p className="text-sm text-muted-foreground">Who can send you messages</p>
                  </div>
                  <Select
                    value={privacySettings.allowMessages}
                    onValueChange={(value) => setPrivacySettings({...privacySettings, allowMessages: value})}
                  >
                    <SelectTrigger className="w-40">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="everyone">Everyone</SelectItem>
                      <SelectItem value="verified">Verified Only</SelectItem>
                      <SelectItem value="premium">Premium Only</SelectItem>
                      <SelectItem value="matches">Matches Only</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Notifications Tab */}
          <TabsContent value="notifications" className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle>Push Notifications</CardTitle>
                <CardDescription>
                  Choose what notifications you'd like to receive
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex items-center justify-between">
                  <div>
                    <Label>New Matches</Label>
                    <p className="text-sm text-muted-foreground">When someone likes you back</p>
                  </div>
                  <Switch
                    checked={notifications.newMatches}
                    onCheckedChange={(checked) => setNotifications({...notifications, newMatches: checked})}
                  />
                </div>
                
                <div className="flex items-center justify-between">
                  <div>
                    <Label>Messages</Label>
                    <p className="text-sm text-muted-foreground">When you receive new messages</p>
                  </div>
                  <Switch
                    checked={notifications.messages}
                    onCheckedChange={(checked) => setNotifications({...notifications, messages: checked})}
                  />
                </div>
                
                <div className="flex items-center justify-between">
                  <div>
                    <Label>Likes</Label>
                    <p className="text-sm text-muted-foreground">When someone likes your profile</p>
                  </div>
                  <Switch
                    checked={notifications.likes}
                    onCheckedChange={(checked) => setNotifications({...notifications, likes: checked})}
                  />
                </div>
                
                <div className="flex items-center justify-between">
                  <div>
                    <Label>Profile Views</Label>
                    <p className="text-sm text-muted-foreground">When someone views your profile</p>
                  </div>
                  <Switch
                    checked={notifications.profileViews}
                    onCheckedChange={(checked) => setNotifications({...notifications, profileViews: checked})}
                  />
                </div>
                
                <div className="flex items-center justify-between">
                  <div>
                    <Label>Promotions</Label>
                    <p className="text-sm text-muted-foreground">Special offers and updates</p>
                  </div>
                  <Switch
                    checked={notifications.promotions}
                    onCheckedChange={(checked) => setNotifications({...notifications, promotions: checked})}
                  />
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Email & SMS</CardTitle>
                <CardDescription>
                  Control email and SMS notification preferences
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex items-center justify-between">
                  <div>
                    <Label>Email Notifications</Label>
                    <p className="text-sm text-muted-foreground">Receive notifications via email</p>
                  </div>
                  <Switch
                    checked={notifications.email}
                    onCheckedChange={(checked) => setNotifications({...notifications, email: checked})}
                  />
                </div>
                
                <div className="flex items-center justify-between">
                  <div>
                    <Label>Push Notifications</Label>
                    <p className="text-sm text-muted-foreground">Receive push notifications on your device</p>
                  </div>
                  <Switch
                    checked={notifications.push}
                    onCheckedChange={(checked) => setNotifications({...notifications, push: checked})}
                  />
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Account Tab */}
          <TabsContent value="account" className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle>Account Security</CardTitle>
                <CardDescription>
                  Manage your account security and verification settings
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex items-center justify-between">
                  <div>
                    <Label>Two-Factor Authentication</Label>
                    <p className="text-sm text-muted-foreground">Add an extra layer of security</p>
                  </div>
                  <Switch
                    checked={accountSettings.twoFactor}
                    onCheckedChange={(checked) => setAccountSettings({...accountSettings, twoFactor: checked})}
                  />
                </div>
                
                <Separator />
                
                <div className="space-y-4">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-2">
                      <Mail className="w-4 h-4" />
                      <div>
                        <Label>Email Verification</Label>
                        <p className="text-sm text-muted-foreground">{basicInfo.email}</p>
                      </div>
                    </div>
                    <Badge variant={accountSettings.emailVerified ? "default" : "secondary"}>
                      {accountSettings.emailVerified ? "Verified" : "Unverified"}
                    </Badge>
                  </div>
                  
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-2">
                      <Phone className="w-4 h-4" />
                      <div>
                        <Label>Phone Verification</Label>
                        <p className="text-sm text-muted-foreground">{basicInfo.phone}</p>
                      </div>
                    </div>
                    <Badge variant={accountSettings.phoneVerified ? "default" : "secondary"}>
                      {accountSettings.phoneVerified ? "Verified" : "Unverified"}
                    </Badge>
                  </div>
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Subscription</CardTitle>
                <CardDescription>
                  Manage your premium subscription
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="flex items-center justify-between">
                  <div className="flex items-center gap-2">
                    <Crown className="w-5 h-5 text-yellow-500" />
                    <div>
                      <Label>Premium Plan</Label>
                      <p className="text-sm text-muted-foreground">Active until March 15, 2024</p>
                    </div>
                  </div>
                  <Button 
                    variant="outline"
                    onClick={() => onNavigate("subscription")}
                  >
                    Manage Plan
                  </Button>
                </div>
              </CardContent>
            </Card>

            <Card className="border-destructive">
              <CardHeader>
                <CardTitle className="text-destructive">Danger Zone</CardTitle>
                <CardDescription>
                  Irreversible actions that will affect your account
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex items-center justify-between">
                  <div>
                    <Label>Deactivate Account</Label>
                    <p className="text-sm text-muted-foreground">Temporarily hide your profile</p>
                  </div>
                  <Button variant="outline">
                    Deactivate
                  </Button>
                </div>
                
                <div className="flex items-center justify-between">
                  <div>
                    <Label className="text-destructive">Delete Account</Label>
                    <p className="text-sm text-muted-foreground">Permanently delete your account and data</p>
                  </div>
                  <Button variant="destructive">
                    <Trash2 className="w-4 h-4 mr-2" />
                    Delete Account
                  </Button>
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>

        {/* Save Button */}
        <div className="flex justify-end pt-6 border-t">
          <Button
            onClick={handleSave}
            disabled={saveStatus === "saving"}
            className="min-w-[120px]"
          >
            {saveStatus === "saving" ? (
              "Saving..."
            ) : saveStatus === "saved" ? (
              "Saved!"
            ) : (
              <>
                <Save className="w-4 h-4 mr-2" />
                Save Changes
              </>
            )}
          </Button>
        </div>
      </div>
      
      <Footer onNavigate={onNavigate} onOpenCookieConsent={onOpenCookieConsent} />
    </div>
  );
}