import { useState } from "react";
import { Button } from "./ui/button";
import { Input } from "./ui/input";
import { Label } from "./ui/label";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "./ui/card";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "./ui/select";
import { Textarea } from "./ui/textarea";
import { Checkbox } from "./ui/checkbox";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "./ui/tabs";
import { Progress } from "./ui/progress";
import { Alert, AlertDescription } from "./ui/alert";
import { Heart, User, MapPin, Camera, Check, Mail, AlertTriangle } from "lucide-react";
import { ReCaptcha } from "./ReCaptcha";
import { PhotoUpload } from "./PhotoUpload";
import { LocationSelector } from "./LocationSelector";
import { LanguageSelector } from "./LanguageSelector";
import { UsernameHelper, UsernameAI } from "./UsernameHelper";
import { useTranslation } from "./TranslationProvider";

interface RegistrationFormProps {
  onSwitchToLogin?: () => void;
  onRegistrationSuccess?: () => void;
}

interface FormData {
  // Basic Info
  firstName: string;
  lastName: string;
  profileName: string;
  email: string;
  password: string;
  confirmPassword: string;
  
  // Verification
  emailVerified: boolean;
  captchaVerified: boolean;
  verificationCode: string;
  
  // Personal Details
  dateOfBirth: string;
  gender: string;
  lookingFor: string;
  country: string;
  city: string;
  
  // Profile Info
  bio: string;
  interests: string[];
  relationshipGoals: string;
  photos: string[];
  
  // Agreements
  termsAccepted: boolean;
  privacyAccepted: boolean;
  ageConfirmed: boolean;
}

const initialFormData: FormData = {
  firstName: "",
  lastName: "",
  profileName: "",
  email: "",
  password: "",
  confirmPassword: "",
  emailVerified: false,
  captchaVerified: false,
  verificationCode: "",
  dateOfBirth: "",
  gender: "",
  lookingFor: "",
  country: "",
  city: "",
  bio: "",
  interests: [],
  relationshipGoals: "",
  photos: [],
  termsAccepted: false,
  privacyAccepted: false,
  ageConfirmed: false,
};



const interests = [
  "Travel", "Music", "Movies", "Sports", "Cooking", "Reading", "Photography",
  "Dancing", "Hiking", "Art", "Fitness", "Gaming", "Nature", "Fashion",
  "Technology", "Food", "Animals", "Adventure"
];

export function RegistrationForm({ onSwitchToLogin, onRegistrationSuccess }: RegistrationFormProps = {}) {
  const { t } = useTranslation();
  const [currentStep, setCurrentStep] = useState("basic");
  const [formData, setFormData] = useState<FormData>(initialFormData);
  const [ageError, setAgeError] = useState("");
  const [emailSent, setEmailSent] = useState(false);
  const [isUsernameValid, setIsUsernameValid] = useState(false);

  const updateFormData = (field: keyof FormData, value: any) => {
    setFormData(prev => ({ ...prev, [field]: value }));
    
    // Age validation when date of birth changes
    if (field === "dateOfBirth") {
      const age = calculateAge(value);
      if (age < 18) {
        setAgeError("You must be at least 18 years old to register.");
      } else {
        setAgeError("");
      }
    }
    
    // Username validation is now handled by UsernameHelper component
    if (field === "profileName") {
      // Check if username is valid and available
      const profileName = value.trim();
      const isValid = profileName.length >= 2 && 
                     profileName.length <= 20 && 
                     /^[a-zA-Z0-9_\s-]+$/.test(profileName) &&
                     UsernameAI.checkAvailability(profileName);
      setIsUsernameValid(isValid);
    }
  };

  const calculateAge = (dateString: string) => {
    if (!dateString) return 0;
    const today = new Date();
    const birthDate = new Date(dateString);
    let age = today.getFullYear() - birthDate.getFullYear();
    const monthDiff = today.getMonth() - birthDate.getMonth();
    if (monthDiff < 0 || (monthDiff === 0 && today.getDate() < birthDate.getDate())) {
      age--;
    }
    return age;
  };

  const sendVerificationEmail = async () => {
    // Simulate sending verification email
    setEmailSent(true);
    // In real implementation, call backend API
    console.log("Verification email sent to:", formData.email);
  };

  const verifyEmailCode = () => {
    // Simulate email verification
    if (formData.verificationCode === "123456") {
      updateFormData("emailVerified", true);
      setCurrentStep("personal");
    } else {
      alert("Invalid verification code. Try 123456 for demo.");
    }
  };

  const toggleInterest = (interest: string) => {
    setFormData(prev => ({
      ...prev,
      interests: prev.interests.includes(interest)
        ? prev.interests.filter(i => i !== interest)
        : [...prev.interests, interest]
    }));
  };

  const getStepProgress = () => {
    switch (currentStep) {
      case "basic": return 20;
      case "verify": return 40;
      case "personal": return 60;
      case "profile": return 80;
      case "complete": return 100;
      default: return 0;
    }
  };

  const handleSubmit = () => {
    // Here we would normally submit to backend
    console.log("Registration data:", formData);
    
    // Add the username to our mock database to prevent future duplicates
    UsernameAI.addToDatabase(formData.profileName);
    
    setCurrentStep("complete");
  };

  const isFormValid = () => {
    const { firstName, lastName, profileName, email, password, confirmPassword, dateOfBirth, gender, lookingFor, country, city, termsAccepted, privacyAccepted, ageConfirmed, emailVerified, captchaVerified, photos } = formData;
    const age = calculateAge(dateOfBirth);
    
    return firstName && lastName && profileName && email && password && confirmPassword && 
           password === confirmPassword && dateOfBirth && age >= 18 && gender && lookingFor && 
           country && city && termsAccepted && privacyAccepted && ageConfirmed && 
           emailVerified && captchaVerified && photos.length > 0 && isUsernameValid;
  };

  return (
    <div className="w-full max-w-2xl mx-auto">
      <div className="mb-8 text-center">
        <div className="flex items-center justify-center gap-2 mb-4">
          <Heart className="w-8 h-8 text-red-500" />
          <h1 className="text-3xl">SinglesZonne</h1>
        </div>
        <h2 className="text-2xl font-semibold mb-2">{t('auth.register.title')}</h2>
        <p className="text-muted-foreground">{t('auth.register.subtitle')}</p>
        {onSwitchToLogin && (
          <p className="text-sm text-muted-foreground mt-2">
            {t('auth.hasAccount')}{" "}
            <button 
              onClick={onSwitchToLogin}
              className="text-primary hover:underline"
            >
              {t('auth.signIn')}
            </button>
          </p>
        )}
      </div>

      <Progress value={getStepProgress()} className="mb-8" />

      <Card>
        <Tabs value={currentStep} onValueChange={setCurrentStep}>
          <TabsList className="grid w-full grid-cols-5">
            <TabsTrigger value="basic" className="flex items-center gap-2">
              <User className="w-4 h-4" />
              Basic
            </TabsTrigger>
            <TabsTrigger value="verify" className="flex items-center gap-2">
              <Mail className="w-4 h-4" />
              Verify
            </TabsTrigger>
            <TabsTrigger value="personal" className="flex items-center gap-2">
              <MapPin className="w-4 h-4" />
              Personal
            </TabsTrigger>
            <TabsTrigger value="profile" className="flex items-center gap-2">
              <Camera className="w-4 h-4" />
              Profile
            </TabsTrigger>
            <TabsTrigger value="complete" className="flex items-center gap-2">
              <Check className="w-4 h-4" />
              Complete
            </TabsTrigger>
          </TabsList>

          <TabsContent value="basic">
            <CardHeader>
              <CardTitle>Basic Information</CardTitle>
              <CardDescription>Let's start with the basics</CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="firstName">{t('auth.firstName')}</Label>
                  <Input
                    id="firstName"
                    value={formData.firstName}
                    onChange={(e) => updateFormData("firstName", e.target.value)}
                    placeholder={t('auth.firstName')}
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="lastName">{t('auth.lastName')}</Label>
                  <Input
                    id="lastName"
                    value={formData.lastName}
                    onChange={(e) => updateFormData("lastName", e.target.value)}
                    placeholder={t('auth.lastName')}
                  />
                </div>
              </div>
              
              <UsernameHelper
                value={formData.profileName}
                onChange={(value) => updateFormData("profileName", value)}
                firstName={formData.firstName}
                interests={formData.interests}
                onValidationChange={setIsUsernameValid}
              />
              
              <div className="space-y-2">
                <Label htmlFor="email">{t('auth.email')}</Label>
                <Input
                  id="email"
                  type="email"
                  value={formData.email}
                  onChange={(e) => updateFormData("email", e.target.value)}
                  placeholder={t('auth.email')}
                />
              </div>
              
              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="password">{t('auth.password')}</Label>
                  <Input
                    id="password"
                    type="password"
                    value={formData.password}
                    onChange={(e) => updateFormData("password", e.target.value)}
                    placeholder={t('auth.password')}
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="confirmPassword">{t('auth.confirmPassword')}</Label>
                  <Input
                    id="confirmPassword"
                    type="password"
                    value={formData.confirmPassword}
                    onChange={(e) => updateFormData("confirmPassword", e.target.value)}
                    placeholder={t('auth.confirmPassword')}
                  />
                </div>
              </div>

              <LanguageSelector />

              <ReCaptcha 
                verified={formData.captchaVerified}
                onVerify={(verified) => updateFormData("captchaVerified", verified)}
              />

              <Button 
                onClick={() => setCurrentStep("verify")} 
                className="w-full"
                disabled={!formData.firstName || !formData.lastName || !formData.profileName || !formData.email || !formData.password || formData.password !== formData.confirmPassword || !formData.captchaVerified || !isUsernameValid}
              >
                Next: Verify Email
              </Button>
            </CardContent>
          </TabsContent>

          <TabsContent value="verify">
            <CardHeader>
              <CardTitle>Verify Your Email</CardTitle>
              <CardDescription>We've sent a verification code to your email</CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              {!emailSent ? (
                <div className="text-center space-y-4">
                  <p>Click the button below to send a verification code to:</p>
                  <p className="font-medium">{formData.email}</p>
                  <Button onClick={sendVerificationEmail} className="w-full">
                    Send Verification Code
                  </Button>
                </div>
              ) : (
                <div className="space-y-4">
                  <Alert>
                    <Mail className="h-4 w-4" />
                    <AlertDescription>
                      A verification code has been sent to {formData.email}. 
                      Please check your inbox and enter the code below.
                    </AlertDescription>
                  </Alert>
                  
                  <div className="space-y-2">
                    <Label htmlFor="verificationCode">Verification Code</Label>
                    <Input
                      id="verificationCode"
                      value={formData.verificationCode}
                      onChange={(e) => updateFormData("verificationCode", e.target.value)}
                      placeholder="Enter 6-digit code (use 123456 for demo)"
                      maxLength={6}
                    />
                  </div>
                  
                  {formData.emailVerified && (
                    <Alert className="border-green-200 bg-green-50">
                      <Check className="h-4 w-4 text-green-600" />
                      <AlertDescription className="text-green-700">
                        Email verified successfully!
                      </AlertDescription>
                    </Alert>
                  )}
                  
                  <div className="flex gap-2">
                    <Button variant="outline" onClick={() => setCurrentStep("basic")} className="flex-1">
                      Back
                    </Button>
                    <Button 
                      onClick={verifyEmailCode}
                      className="flex-1"
                      disabled={!formData.verificationCode || formData.emailVerified}
                    >
                      {formData.emailVerified ? "Verified" : "Verify Code"}
                    </Button>
                  </div>
                </div>
              )}

              {formData.emailVerified && (
                <Button 
                  onClick={() => setCurrentStep("personal")} 
                  className="w-full"
                >
                  Continue to Personal Details
                </Button>
              )}
            </CardContent>
          </TabsContent>

          <TabsContent value="personal">
            <CardHeader>
              <CardTitle>Personal Details</CardTitle>
              <CardDescription>Tell us more about yourself</CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-4">
                <div className="space-y-2">
                  <Label htmlFor="dateOfBirth">Date of Birth</Label>
                  <Input
                    id="dateOfBirth"
                    type="date"
                    value={formData.dateOfBirth}
                    onChange={(e) => updateFormData("dateOfBirth", e.target.value)}
                    max={new Date(new Date().setFullYear(new Date().getFullYear() - 18)).toISOString().split('T')[0]}
                  />
                  {ageError && (
                    <Alert variant="destructive">
                      <AlertTriangle className="h-4 w-4" />
                      <AlertDescription>{ageError}</AlertDescription>
                    </Alert>
                  )}
                </div>

                <LocationSelector
                  country={formData.country}
                  city={formData.city}
                  onCountryChange={(country) => updateFormData("country", country)}
                  onCityChange={(city) => updateFormData("city", city)}
                />
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label>I am</Label>
                  <Select value={formData.gender} onValueChange={(value) => updateFormData("gender", value)}>
                    <SelectTrigger>
                      <SelectValue placeholder="Select gender" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="man">Man</SelectItem>
                      <SelectItem value="woman">Woman</SelectItem>
                      <SelectItem value="non-binary">Non-binary</SelectItem>
                      <SelectItem value="prefer-not-to-say">Prefer not to say</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div className="space-y-2">
                  <Label>Looking for</Label>
                  <Select value={formData.lookingFor} onValueChange={(value) => updateFormData("lookingFor", value)}>
                    <SelectTrigger>
                      <SelectValue placeholder="Looking for" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="men">Men</SelectItem>
                      <SelectItem value="women">Women</SelectItem>
                      <SelectItem value="everyone">Everyone</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>

              <div className="space-y-2">
                <Label>Relationship Goals</Label>
                <Select value={formData.relationshipGoals} onValueChange={(value) => updateFormData("relationshipGoals", value)}>
                  <SelectTrigger>
                    <SelectValue placeholder="What are you looking for?" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="serious-relationship">Serious Relationship</SelectItem>
                    <SelectItem value="casual-dating">Casual Dating</SelectItem>
                    <SelectItem value="friendship">Friendship</SelectItem>
                    <SelectItem value="marriage">Marriage</SelectItem>
                    <SelectItem value="open-to-explore">Open to Explore</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-4">
                <div className="flex items-center space-x-2">
                  <Checkbox
                    id="ageConfirmed"
                    checked={formData.ageConfirmed}
                    onCheckedChange={(checked) => updateFormData("ageConfirmed", checked)}
                  />
                  <Label htmlFor="ageConfirmed" className="text-sm">
                    I confirm that I am at least 18 years old
                  </Label>
                </div>
              </div>

              <div className="flex gap-2">
                <Button variant="outline" onClick={() => setCurrentStep("verify")} className="flex-1">
                  Back
                </Button>
                <Button 
                  onClick={() => setCurrentStep("profile")} 
                  className="flex-1"
                  disabled={!formData.dateOfBirth || !formData.gender || !formData.lookingFor || !formData.country || !formData.city || !formData.ageConfirmed || calculateAge(formData.dateOfBirth) < 18}
                >
                  Next: Create Profile
                </Button>
              </div>
            </CardContent>
          </TabsContent>

          <TabsContent value="profile">
            <CardHeader>
              <CardTitle>Create Your Profile</CardTitle>
              <CardDescription>Make yourself stand out</CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="bio">About Me</Label>
                <Textarea
                  id="bio"
                  value={formData.bio}
                  onChange={(e) => updateFormData("bio", e.target.value)}
                  placeholder="Tell others about yourself, your hobbies, and what you're passionate about..."
                  className="min-h-[100px]"
                />
              </div>

              <div className="space-y-2">
                <Label>Interests (Select all that apply)</Label>
                <div className="grid grid-cols-3 gap-2">
                  {interests.map((interest) => (
                    <Button
                      key={interest}
                      variant={formData.interests.includes(interest) ? "default" : "outline"}
                      size="sm"
                      onClick={() => toggleInterest(interest)}
                      className="justify-start"
                    >
                      {interest}
                    </Button>
                  ))}
                </div>
              </div>

              <div className="space-y-4 border rounded-lg p-4 bg-muted/20">
                <PhotoUpload
                  photos={formData.photos}
                  onPhotosChange={(photos) => updateFormData("photos", photos)}
                  required={true}
                />
              </div>

              <div className="space-y-4">
                <div className="flex items-center space-x-2">
                  <Checkbox
                    id="terms"
                    checked={formData.termsAccepted}
                    onCheckedChange={(checked) => updateFormData("termsAccepted", checked)}
                  />
                  <Label htmlFor="terms" className="text-sm">
                    I agree to the <a href="#" className="text-primary hover:underline">Terms of Service</a>
                  </Label>
                </div>
                <div className="flex items-center space-x-2">
                  <Checkbox
                    id="privacy"
                    checked={formData.privacyAccepted}
                    onCheckedChange={(checked) => updateFormData("privacyAccepted", checked)}
                  />
                  <Label htmlFor="privacy" className="text-sm">
                    I agree to the <a href="#" className="text-primary hover:underline">Privacy Policy</a>
                  </Label>
                </div>
              </div>

              <div className="flex gap-2">
                <Button variant="outline" onClick={() => setCurrentStep("personal")} className="flex-1">
                  Back
                </Button>
                <Button 
                  onClick={handleSubmit}
                  className="flex-1"
                  disabled={!isFormValid()}
                >
                  Create My Profile
                </Button>
              </div>
              
              {!isFormValid() && (
                <Alert>
                  <AlertTriangle className="h-4 w-4" />
                  <AlertDescription>
                    Please complete all required fields including uploading at least one photo.
                  </AlertDescription>
                </Alert>
              )}
            </CardContent>
          </TabsContent>

          <TabsContent value="complete">
            <CardHeader className="text-center">
              <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                <Check className="w-8 h-8 text-green-600" />
              </div>
              <CardTitle>Welcome to SinglesZonne!</CardTitle>
              <CardDescription>Your profile has been created successfully</CardDescription>
            </CardHeader>
            <CardContent className="text-center space-y-4">
              <p>You're all set to start connecting with amazing people around the world.</p>
              <div className="space-y-2">
                <Button 
                  className="w-full"
                  onClick={() => onRegistrationSuccess && onRegistrationSuccess()}
                >
                  Start Browsing Profiles
                </Button>
                <Button 
                  variant="outline" 
                  className="w-full"
                  onClick={() => onRegistrationSuccess && onRegistrationSuccess()}
                >
                  Complete Profile Later
                </Button>
              </div>
            </CardContent>
          </TabsContent>
        </Tabs>
      </Card>
    </div>
  );
}