import { useState } from "react";
import { Header } from "./Header";
import { SearchBanner } from "./SearchBanner";
import { SearchFilters } from "./SearchFilters";
import { ProfileGrid } from "./ProfileGrid";
import { Footer } from "./Footer";

interface SearchPageProps {
  onNavigate: (view: string) => void;
  onOpenCookieConsent?: () => void;
}

interface SearchFilters {
  lookingFor: string;
  ageMin: number;
  ageMax: number;
  country: string;
  distance: string;
}

export function SearchPage({ onNavigate, onOpenCookieConsent }: SearchPageProps) {
  const [profiles, setProfiles] = useState([]);
  const [currentFilters, setCurrentFilters] = useState<SearchFilters>({
    lookingFor: "women",
    ageMin: 18,
    ageMax: 40,
    country: "any",
    distance: "",
  });
  const [isSearching, setIsSearching] = useState(false);

  const handleSearch = async (filters: SearchFilters) => {
    setIsSearching(true);
    setCurrentFilters(filters);
    
    // Simulate API search delay
    await new Promise(resolve => setTimeout(resolve, 1000));
    
    // In a real app, this would make an API call
    console.log("Searching with filters:", filters);
    
    setIsSearching(false);
    // The ProfileGrid component will handle filtering based on lookingFor
  };

  const handleProfileClick = (profileId: string) => {
    // In a real app, you would pass the profileId to the user profile page
    onNavigate("user-profile");
  };

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <Header currentView="search" onNavigate={onNavigate} />
      
      {/* Search Banner */}
      <SearchBanner />
      
      {/* Search Filters */}
      <SearchFilters onSearch={handleSearch} />
      
      {/* Results */}
      <div className="relative">
        {isSearching && (
          <div className="absolute inset-0 bg-background/80 backdrop-blur-sm z-10 flex items-center justify-center">
            <div className="text-center">
              <div className="w-8 h-8 border-4 border-purple-600 border-t-transparent rounded-full animate-spin mx-auto mb-4"></div>
              <p className="text-muted-foreground">Searching for your perfect matches...</p>
            </div>
          </div>
        )}
        <ProfileGrid profiles={profiles} lookingFor={currentFilters.lookingFor} onProfileClick={handleProfileClick} />
      </div>
      
      {/* Footer */}
      <Footer onNavigate={onNavigate} onOpenCookieConsent={onOpenCookieConsent} />
    </div>
  );
}