import { useState } from "react";
import { Header } from "./Header";
import { Footer } from "./Footer";
import { CancellationModal } from "./CancellationModal";
import { Button } from "./ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "./ui/card";
import { Badge } from "./ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "./ui/tabs";
import { Alert, AlertDescription } from "./ui/alert";
import { Progress } from "./ui/progress";
import { 
  Crown, 
  Check, 
  X, 
  Heart, 
  Zap, 
  Eye, 
  MessageCircle, 
  Users, 
  Star,
  Shield,
  Sparkles,
  Gift,
  CreditCard,
  Calendar,
  AlertTriangle,
  CheckCircle
} from "lucide-react";
import { motion } from "motion/react";
import { useTranslation } from "./TranslationProvider";

interface SubscriptionPageProps {
  onNavigate: (view: string) => void;
  onOpenCookieConsent?: () => void;
}

interface SubscriptionPlan {
  id: string;
  name: string;
  price: number;
  duration: string;
  originalPrice?: number;
  popular?: boolean;
  features: string[];
  limitations?: string[];
  color: string;
  icon: React.ComponentType<any>;
}

interface UserSubscription {
  plan: string;
  status: "active" | "cancelled" | "expired";
  nextBilling: string;
  autoRenew: boolean;
  daysRemaining: number;
}

export function SubscriptionPage({ onNavigate, onOpenCookieConsent }: SubscriptionPageProps) {
  const { t } = useTranslation();
  const [activeTab, setActiveTab] = useState("plans");
  const [selectedBilling, setSelectedBilling] = useState<"monthly" | "yearly">("monthly");
  const [showCancellationModal, setShowCancellationModal] = useState(false);
  
  // Mock current user subscription
  const [userSubscription] = useState<UserSubscription>({
    plan: "premium",
    status: "active",
    nextBilling: "2024-01-15",
    autoRenew: true,
    daysRemaining: 23
  });

  const subscriptionPlans: SubscriptionPlan[] = [
    {
      id: "free",
      name: "Free",
      price: 0,
      duration: selectedBilling === "monthly" ? "/month" : "/year",
      features: [
        "5 likes per day",
        "Basic profile",
        "Limited matches",
        "Standard support"
      ],
      limitations: [
        "No premium features",
        "Ads included",
        "Limited messaging"
      ],
      color: "gray",
      icon: Heart
    },
    {
      id: "premium",
      name: "Premium",
      price: selectedBilling === "monthly" ? 19.99 : 199.99,
      originalPrice: selectedBilling === "monthly" ? 29.99 : 299.99,
      duration: selectedBilling === "monthly" ? "/month" : "/year",
      popular: true,
      features: [
        "Unlimited likes",
        "See who liked you",
        "Premium profile boost",
        "Advanced filters",
        "Read receipts",
        "Priority support",
        "Ad-free experience"
      ],
      color: "blue",
      icon: Crown
    },
    {
      id: "platinum",
      name: "Platinum",
      price: selectedBilling === "monthly" ? 39.99 : 399.99,
      originalPrice: selectedBilling === "monthly" ? 49.99 : 499.99,
      duration: selectedBilling === "monthly" ? "/month" : "/year",
      features: [
        "All Premium features",
        "Weekly profile boost",
        "VIP badge",
        "Message before matching",
        "Advanced analytics",
        "Priority customer support",
        "Exclusive events access",
        "Travel mode"
      ],
      color: "purple",
      icon: Sparkles
    }
  ];

  const handleUpgrade = (planId: string) => {
    console.log(`Upgrading to ${planId}`);
    // In real app, this would integrate with payment processor
  };

  const handleDowngrade = (planId: string) => {
    console.log(`Downgrading to ${planId}`);
    // In real app, this would process downgrade
  };

  const handleCancelSubscription = () => {
    setShowCancellationModal(true);
  };

  const handleConfirmCancellation = () => {
    console.log("Subscription cancelled");
    // In real app, this would process the cancellation
    setShowCancellationModal(false);
    // You could update the userSubscription state here to reflect cancellation
  };

  const handleRetentionOffer = (offerId: string) => {
    console.log(`Accepted retention offer: ${offerId}`);
    // In real app, this would process the retention offer
    // Could redirect to payment processing or update subscription
  };

  const getDiscountPercentage = (plan: SubscriptionPlan) => {
    if (!plan.originalPrice) return 0;
    return Math.round(((plan.originalPrice - plan.price) / plan.originalPrice) * 100);
  };

  const currentPlan = subscriptionPlans.find(p => p.id === userSubscription.plan);

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <Header currentView="subscription" onNavigate={onNavigate} />
      
      {/* Subscription Banner */}
      <div className="bg-gradient-to-r from-yellow-500 via-yellow-600 to-yellow-700 dark:from-yellow-600 dark:via-yellow-700 dark:to-yellow-800 text-white py-11 relative overflow-hidden">
        {/* Moving gradient overlay */}
        <motion.div 
          className="absolute inset-0 bg-gradient-to-r from-transparent via-yellow-300/30 to-transparent opacity-40"
          animate={{
            x: ['-100%', '100%']
          }}
          transition={{
            duration: 8,
            repeat: Infinity,
            ease: "linear",
            repeatDelay: 2
          }}
          style={{
            width: '200%',
            background: 'linear-gradient(90deg, transparent 0%, rgba(255,255,255,0.15) 25%, rgba(255,255,255,0.3) 50%, rgba(255,255,255,0.15) 75%, transparent 100%)'
          }}
        />
        
        <div className="max-w-7xl mx-auto px-6 text-center relative z-10">
          <motion.h1 
            className="text-4xl font-bold mb-4"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ 
              duration: 0.8,
              ease: "easeOut"
            }}
          >
            <motion.span
              animate={{ 
                scale: [1, 1.05, 1],
              }}
              transition={{
                duration: 3,
                repeat: Infinity,
                ease: "easeInOut",
                repeatDelay: 0.5
              }}
            >
              👑 Premium Membership
            </motion.span>
          </motion.h1>
          <motion.p 
            className="text-lg text-yellow-100 dark:text-yellow-200"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ 
              duration: 0.8,
              delay: 0.2,
              ease: "easeOut"
            }}
          >
            Unlock exclusive features and find your perfect match faster
          </motion.p>
        </div>
      </div>

      {/* Main Content */}
      <div className="max-w-7xl mx-auto px-6 py-12">
        <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
          {/* Tab Navigation */}
          <TabsList className="grid w-full grid-cols-3 mb-8 h-auto p-1">
            <TabsTrigger 
              value="plans" 
              className="flex items-center gap-2 px-6 py-3 data-[state=active]:bg-yellow-600 data-[state=active]:text-white"
            >
              <Crown className="w-4 h-4" />
              <span>Subscription Plans</span>
            </TabsTrigger>
            <TabsTrigger 
              value="current" 
              className="flex items-center gap-2 px-6 py-3 data-[state=active]:bg-yellow-600 data-[state=active]:text-white"
            >
              <CreditCard className="w-4 h-4" />
              <span>Current Plan</span>
            </TabsTrigger>
            <TabsTrigger 
              value="billing" 
              className="flex items-center gap-2 px-6 py-3 data-[state=active]:bg-yellow-600 data-[state=active]:text-white"
            >
              <Calendar className="w-4 h-4" />
              <span>Billing History</span>
            </TabsTrigger>
          </TabsList>

          {/* Subscription Plans Tab */}
          <TabsContent value="plans" className="space-y-8">
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.5 }}
            >
              {/* Billing Toggle */}
              <div className="flex justify-center mb-8">
                <div className="bg-muted rounded-lg p-1 flex">
                  <Button
                    variant={selectedBilling === "monthly" ? "default" : "ghost"}
                    size="sm"
                    onClick={() => setSelectedBilling("monthly")}
                    className={selectedBilling === "monthly" ? "bg-yellow-600 text-white" : ""}
                  >
                    Monthly
                  </Button>
                  <Button
                    variant={selectedBilling === "yearly" ? "default" : "ghost"}
                    size="sm"
                    onClick={() => setSelectedBilling("yearly")}
                    className={selectedBilling === "yearly" ? "bg-yellow-600 text-white" : ""}
                  >
                    <div className="flex items-center gap-2">
                      Yearly
                      <Badge variant="secondary" className="bg-green-100 text-green-800 text-xs">
                        Save 33%
                      </Badge>
                    </div>
                  </Button>
                </div>
              </div>

              {/* Plans Grid */}
              <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                {subscriptionPlans.map((plan, index) => {
                  const Icon = plan.icon;
                  const isCurrentPlan = plan.id === userSubscription.plan;
                  const discount = getDiscountPercentage(plan);
                  
                  return (
                    <motion.div
                      key={plan.id}
                      initial={{ opacity: 0, y: 20 }}
                      animate={{ opacity: 1, y: 0 }}
                      transition={{ duration: 0.3, delay: index * 0.1 }}
                      className="relative"
                    >
                      <Card className={`relative ${plan.popular ? 'ring-2 ring-yellow-500 ring-offset-4' : ''} ${isCurrentPlan ? 'bg-yellow-50 dark:bg-yellow-950' : ''}`}>
                        {plan.popular && (
                          <div className="absolute -top-4 left-1/2 transform -translate-x-1/2">
                            <Badge className="bg-yellow-600 text-white px-4 py-1">
                              Most Popular
                            </Badge>
                          </div>
                        )}
                        
                        {isCurrentPlan && (
                          <div className="absolute -top-4 right-4">
                            <Badge className="bg-green-600 text-white px-3 py-1">
                              <CheckCircle className="w-3 h-3 mr-1" />
                              Current
                            </Badge>
                          </div>
                        )}

                        <CardHeader className="text-center pb-4">
                          <div className="flex justify-center mb-4">
                            <div className={`w-16 h-16 rounded-full flex items-center justify-center ${
                              plan.color === 'gray' ? 'bg-gray-100 dark:bg-gray-800' :
                              plan.color === 'blue' ? 'bg-blue-100 dark:bg-blue-900' :
                              'bg-purple-100 dark:bg-purple-900'
                            }`}>
                              <Icon className={`w-8 h-8 ${
                                plan.color === 'gray' ? 'text-gray-600' :
                                plan.color === 'blue' ? 'text-blue-600' :
                                'text-purple-600'
                              }`} />
                            </div>
                          </div>
                          
                          <CardTitle className="text-2xl">{plan.name}</CardTitle>
                          
                          <div className="space-y-2">
                            {plan.originalPrice && discount > 0 && (
                              <div className="flex items-center justify-center gap-2">
                                <span className="text-sm text-muted-foreground line-through">
                                  ${plan.originalPrice}{plan.duration}
                                </span>
                                <Badge variant="destructive" className="text-xs">
                                  {discount}% OFF
                                </Badge>
                              </div>
                            )}
                            
                            <div className="flex items-baseline justify-center gap-1">
                              <span className="text-4xl font-bold">${plan.price}</span>
                              <span className="text-muted-foreground">{plan.duration}</span>
                            </div>
                          </div>
                        </CardHeader>

                        <CardContent className="space-y-6">
                          {/* Features */}
                          <div className="space-y-3">
                            {plan.features.map((feature) => (
                              <div key={feature} className="flex items-center gap-3">
                                <Check className="w-5 h-5 text-green-600 flex-shrink-0" />
                                <span className="text-sm">{feature}</span>
                              </div>
                            ))}
                            
                            {plan.limitations && plan.limitations.map((limitation) => (
                              <div key={limitation} className="flex items-center gap-3 opacity-60">
                                <X className="w-5 h-5 text-red-500 flex-shrink-0" />
                                <span className="text-sm">{limitation}</span>
                              </div>
                            ))}
                          </div>

                          {/* Action Button */}
                          <div className="pt-4">
                            {isCurrentPlan ? (
                              <Button variant="outline" className="w-full" disabled>
                                <CheckCircle className="w-4 h-4 mr-2" />
                                Current Plan
                              </Button>
                            ) : plan.id === "free" ? (
                              <Button 
                                variant="outline" 
                                className="w-full"
                                onClick={() => handleDowngrade(plan.id)}
                              >
                                Downgrade to Free
                              </Button>
                            ) : (
                              <Button 
                                className={`w-full ${
                                  plan.popular 
                                    ? 'bg-yellow-600 hover:bg-yellow-700' 
                                    : 'bg-primary hover:bg-primary/90'
                                }`}
                                onClick={() => handleUpgrade(plan.id)}
                              >
                                <Crown className="w-4 h-4 mr-2" />
                                Upgrade Now
                              </Button>
                            )}
                          </div>
                        </CardContent>
                      </Card>
                    </motion.div>
                  );
                })}
              </div>

              {/* Money Back Guarantee */}
              <div className="mt-8 text-center">
                <div className="flex items-center justify-center gap-2 text-muted-foreground">
                  <Shield className="w-5 h-5" />
                  <span>30-day money-back guarantee</span>
                </div>
              </div>
            </motion.div>
          </TabsContent>

          {/* Current Plan Tab */}
          <TabsContent value="current" className="space-y-6">
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.5 }}
            >
              {/* Current Subscription Card */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Crown className="w-5 h-5 text-yellow-600" />
                    Your Current Subscription
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-6">
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div className="space-y-4">
                      <div>
                        <h3 className="font-semibold mb-2">{currentPlan?.name} Plan</h3>
                        <p className="text-2xl font-bold text-yellow-600">
                          ${currentPlan?.price}{currentPlan?.duration}
                        </p>
                      </div>
                      
                      <div className="space-y-2">
                        <div className="flex justify-between">
                          <span className="text-sm text-muted-foreground">Status</span>
                          <Badge className={
                            userSubscription.status === 'active' ? 'bg-green-100 text-green-800' :
                            userSubscription.status === 'cancelled' ? 'bg-red-100 text-red-800' :
                            'bg-gray-100 text-gray-800'
                          }>
                            {userSubscription.status.charAt(0).toUpperCase() + userSubscription.status.slice(1)}
                          </Badge>
                        </div>
                        
                        <div className="flex justify-between">
                          <span className="text-sm text-muted-foreground">Next billing</span>
                          <span className="text-sm font-medium">{userSubscription.nextBilling}</span>
                        </div>
                        
                        <div className="flex justify-between">
                          <span className="text-sm text-muted-foreground">Auto-renew</span>
                          <span className="text-sm font-medium">
                            {userSubscription.autoRenew ? 'Enabled' : 'Disabled'}
                          </span>
                        </div>
                      </div>
                    </div>
                    
                    <div className="space-y-4">
                      <div>
                        <h4 className="font-medium mb-2">Subscription Progress</h4>
                        <div className="space-y-2">
                          <div className="flex justify-between text-sm">
                            <span>Days remaining</span>
                            <span>{userSubscription.daysRemaining} days</span>
                          </div>
                          <Progress value={(userSubscription.daysRemaining / 30) * 100} className="h-2" />
                        </div>
                      </div>
                      
                      <div>
                        <h4 className="font-medium mb-2">Usage This Month</h4>
                        <div className="space-y-2 text-sm">
                          <div className="flex justify-between">
                            <span>Likes sent</span>
                            <span>∞ (Unlimited)</span>
                          </div>
                          <div className="flex justify-between">
                            <span>Super likes</span>
                            <span>5/5 remaining</span>
                          </div>
                          <div className="flex justify-between">
                            <span>Profile boosts</span>
                            <span>1/1 used</span>
                          </div>
                        </div>
                      </div>
                    </div>
                  </div>

                  <div className="border-t pt-6">
                    <div className="flex gap-4">
                      <Button 
                        variant="outline" 
                        onClick={handleCancelSubscription}
                        className="flex-1"
                      >
                        <X className="w-4 h-4 mr-2" />
                        Cancel Subscription
                      </Button>
                      <Button 
                        onClick={() => setActiveTab("plans")}
                        className="flex-1 bg-yellow-600 hover:bg-yellow-700"
                      >
                        <Crown className="w-4 h-4 mr-2" />
                        Upgrade Plan
                      </Button>
                    </div>
                  </div>
                </CardContent>
              </Card>

              {/* Subscription Benefits */}
              <Card>
                <CardHeader>
                  <CardTitle>Your Premium Benefits</CardTitle>
                  <CardDescription>
                    Here's what you get with your {currentPlan?.name} subscription
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                    {currentPlan?.features.map((feature, index) => (
                      <div key={feature} className="flex items-center gap-3 p-3 bg-muted/50 rounded-lg">
                        <Check className="w-5 h-5 text-green-600" />
                        <span className="text-sm">{feature}</span>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            </motion.div>
          </TabsContent>

          {/* Billing History Tab */}
          <TabsContent value="billing" className="space-y-6">
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.5 }}
            >
              <Card>
                <CardHeader>
                  <CardTitle>Billing History</CardTitle>
                  <CardDescription>
                    View your past payments and download receipts
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    {[
                      { date: '2024-01-15', amount: '$19.99', status: 'Paid', invoice: 'INV-001' },
                      { date: '2023-12-15', amount: '$19.99', status: 'Paid', invoice: 'INV-002' },
                      { date: '2023-11-15', amount: '$19.99', status: 'Paid', invoice: 'INV-003' },
                    ].map((payment, index) => (
                      <div key={index} className="flex items-center justify-between p-4 border rounded-lg">
                        <div className="space-y-1">
                          <p className="font-medium">Premium Subscription</p>
                          <p className="text-sm text-muted-foreground">{payment.date}</p>
                        </div>
                        <div className="text-right space-y-1">
                          <p className="font-medium">{payment.amount}</p>
                          <Badge className="bg-green-100 text-green-800">
                            {payment.status}
                          </Badge>
                        </div>
                        <Button variant="outline" size="sm">
                          Download
                        </Button>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            </motion.div>
          </TabsContent>
        </Tabs>
      </div>
      
      {/* Footer */}
      <Footer onNavigate={onNavigate} onOpenCookieConsent={onOpenCookieConsent} />

      {/* Cancellation Modal */}
      <CancellationModal
        isOpen={showCancellationModal}
        onClose={() => setShowCancellationModal(false)}
        currentPlan={{
          name: currentPlan?.name || "Premium",
          price: currentPlan?.price || 19.99,
          duration: currentPlan?.duration || "/month"
        }}
        onConfirmCancellation={handleConfirmCancellation}
        onRetentionOffer={handleRetentionOffer}
      />
    </div>
  );
}