import { useState } from 'react';

// Unit conversion utilities
export const convertKmToMiles = (km: number): number => {
  return Math.round(km * 0.621371 * 10) / 10; // Round to 1 decimal place
};

export const convertMilesToKm = (miles: number): number => {
  return Math.round(miles * 1.60934 * 10) / 10; // Round to 1 decimal place
};

export const convertKgToLbs = (kg: number): number => {
  return Math.round(kg * 2.20462 * 10) / 10; // Round to 1 decimal place
};

export const convertLbsToKg = (lbs: number): number => {
  return Math.round(lbs * 0.453592 * 10) / 10; // Round to 1 decimal place
};

export const convertCmToFeet = (cm: number): string => {
  const totalInches = cm / 2.54;
  const feet = Math.floor(totalInches / 12);
  const inches = Math.round(totalInches % 12);
  return `${feet}'${inches}"`;
};

export const convertFeetToCm = (feet: number, inches: number): number => {
  return Math.round((feet * 12 + inches) * 2.54);
};

// Format distance with both units
export const formatDistance = (km: number): string => {
  const miles = convertKmToMiles(km);
  return `${km} km (${miles} mi)`;
};

// Format weight with both units
export const formatWeight = (kg: number): string => {
  const lbs = convertKgToLbs(kg);
  return `${kg} kg (${lbs} lbs)`;
};

// Format height with both units
export const formatHeight = (cm: number): string => {
  const feetInches = convertCmToFeet(cm);
  return `${cm} cm (${feetInches})`;
};

// Height options with both metric and imperial
export const getHeightOptions = () => {
  const heights = [];
  
  // Generate heights from 140cm (4'7") to 220cm (7'2")
  for (let cm = 140; cm <= 220; cm++) {
    const feetInches = convertCmToFeet(cm);
    heights.push({
      value: cm,
      label: `${cm} cm (${feetInches})`,
      cm: cm,
      display: feetInches
    });
  }
  
  return heights;
};

// Weight options with both metric and imperial
export const getWeightOptions = () => {
  const weights = [];
  
  // Generate weights from 40kg (88lbs) to 200kg (441lbs)
  for (let kg = 40; kg <= 200; kg += 2) {
    const lbs = convertKgToLbs(kg);
    weights.push({
      value: kg,
      label: `${kg} kg (${lbs} lbs)`,
      kg: kg,
      lbs: Math.round(lbs)
    });
  }
  
  return weights;
};

// Distance options for preferences
export const getDistanceOptions = () => {
  const distances = [5, 10, 15, 20, 25, 30, 40, 50, 75, 100, 150, 200, 300, 500, 1000];
  
  return distances.map(km => ({
    value: km,
    label: formatDistance(km),
    km: km,
    miles: convertKmToMiles(km)
  }));
};

// Hook for managing units preference
export const useUnitsPreference = () => {
  const [preferredUnits, setPreferredUnits] = useState<'metric' | 'imperial'>(() => {
    // Default to metric, but could be determined by user location
    const saved = localStorage.getItem('preferred-units');
    return (saved as 'metric' | 'imperial') || 'metric';
  });

  const setUnits = (units: 'metric' | 'imperial') => {
    setPreferredUnits(units);
    localStorage.setItem('preferred-units', units);
  };

  return { preferredUnits, setUnits };
};

// Component for displaying dual units
interface DualUnitsDisplayProps {
  type: 'distance' | 'weight' | 'height';
  value: number;
  className?: string;
}

export function DualUnitsDisplay({ type, value, className = '' }: DualUnitsDisplayProps) {
  const { preferredUnits } = useUnitsPreference();
  
  const formatValue = () => {
    switch (type) {
      case 'distance':
        if (preferredUnits === 'metric') {
          return `${value} km (${convertKmToMiles(value)} mi)`;
        } else {
          return `${convertKmToMiles(value)} mi (${value} km)`;
        }
      case 'weight':
        if (preferredUnits === 'metric') {
          return `${value} kg (${convertKgToLbs(value)} lbs)`;
        } else {
          return `${convertKgToLbs(value)} lbs (${value} kg)`;
        }
      case 'height':
        if (preferredUnits === 'metric') {
          return `${value} cm (${convertCmToFeet(value)})`;
        } else {
          return `${convertCmToFeet(value)} (${value} cm)`;
        }
      default:
        return value.toString();
    }
  };

  return <span className={className}>{formatValue()}</span>;
}

// Units selector component
interface UnitsSelectorProps {
  value: 'metric' | 'imperial';
  onChange: (units: 'metric' | 'imperial') => void;
  className?: string;
}

export function UnitsSelector({ value, onChange, className = '' }: UnitsSelectorProps) {
  return (
    <div className={`flex items-center space-x-2 ${className}`}>
      <span className="text-sm font-medium">Units:</span>
      <button
        onClick={() => onChange('metric')}
        className={`px-3 py-1 text-xs rounded-full transition-colors ${
          value === 'metric' 
            ? 'bg-primary text-primary-foreground' 
            : 'bg-muted text-muted-foreground hover:bg-muted/80'
        }`}
      >
        Metric
      </button>
      <button
        onClick={() => onChange('imperial')}
        className={`px-3 py-1 text-xs rounded-full transition-colors ${
          value === 'imperial' 
            ? 'bg-primary text-primary-foreground' 
            : 'bg-muted text-muted-foreground hover:bg-muted/80'
        }`}
      >
        Imperial
      </button>
    </div>
  );
}