import { useState, useEffect, useRef } from "react";
import { Header } from "./Header";
import { Footer } from "./Footer";
import { Button } from "./ui/button";
import { Card } from "./ui/card";
import { Badge } from "./ui/badge";
import { Separator } from "./ui/separator";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "./ui/dialog";
import { Alert, AlertDescription } from "./ui/alert";
import { 
  Heart, 
  MessageCircle, 
  Share2, 
  MoreHorizontal,
  MapPin,
  Calendar,
  Ruler,
  GraduationCap,
  Briefcase,
  Camera,
  Flag,
  Shield,
  Eye,
  Verified,
  Crown,
  Star,
  Users,
  Play,
  Pause,
  Volume2,
  VolumeX,
  Video,
  Music,
  ImageIcon,
  Search,
  Target,
  Globe,
  UserCheck
} from "lucide-react";
import { ImageWithFallback } from "./figma/ImageWithFallback";
import { useTranslation } from "./TranslationProvider";
import { PhotoCollage } from "./PhotoCollage";
import { 
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from "./ui/dropdown-menu";
import { Slider } from "./ui/slider";
import { MediaProvider, useMedia, useVideoManager, useMusicManager } from "./MediaManager";
import { GoogleMaps } from "./GoogleMaps";
import { useColorTheme } from "./ColorThemeProvider";

interface UserProfilePageProps {
  onNavigate: (view: string) => void;
  onOpenCookieConsent?: () => void;
  userId?: string;
}

interface UserProfile {
  id: string;
  name: string;
  age: number;
  location: string;
  country: string;
  countryFlag: string;
  coordinates: { lat: number; lng: number };
  bio: string;
  occupation: string;
  education: string;
  height: string;
  lastSeen: string;
  isOnline: boolean;
  isVerified: boolean;
  isPremium: boolean;
  photos: string[];
  interests: string[];
  distance: number;
  mutualFriends: number;
  joinDate: string;
  introVideo?: string;
  profileMusic?: string;
  customBackground?: string;
  musicTitle?: string;
  musicArtist?: string;
  lookingFor?: {
    relationshipType: string;
    lookingForText: string;
    ageRange: { min: number; max: number };
    maxDistance: number;
    dealBreakers: string[];
    importantQualities: string[];
    countriesPreference: "all" | "specific" | "multiple";
    selectedCountries: string[];
  };
}

function UserProfileContent({ onNavigate, onOpenCookieConsent, userId = "1" }: UserProfilePageProps) {
  const { t } = useTranslation();
  const { theme } = useColorTheme();
  const {
    introVideo,
    profileMusic,
    customBackground,
    isVideoPlaying,
    isMusicPlaying,
    isMusicPaused,
    musicVolume,
    setIntroVideo,
    setProfileMusic,
    setCustomBackground,
    playMusic,
    pauseMusic,
    setMusicVolume,
  } = useMedia();
  
  const [userProfile] = useState<UserProfile>({
    id: "admin",
    name: "Administrator",
    age: 32,
    location: "New York",
    country: "United States",
    countryFlag: "🇺🇸",
    coordinates: { lat: 40.7128, lng: -74.0060 }, // New York City coordinates
    bio: "Tech entrepreneur and dating app creator. Passionate about bringing people together and creating meaningful connections. Always looking for ways to improve the dating experience! 💫",
    occupation: "CEO & Founder",
    education: "MBA in Business Administration",
    height: "6 feet 0 inches",
    lastSeen: "Online now",
    isOnline: true,
    isVerified: true,
    isPremium: true,
    photos: [
      "https://images.unsplash.com/photo-1572994315280-5964e2cd990e?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxiZWF1dGlmdWwlMjBhc2lhbiUyMHdvbWFuJTIwcHJvZmVzc2lvbmFsJTIwcG9ydHJhaXR8ZW58MXx8fHwxNzU4NDU0NTQzfDA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      "https://images.unsplash.com/photo-1618804160900-585d0f5a9c9f?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxiZWF1dGlmdWwlMjBsYXRpbmElMjB3b21hbiUyMG91dGRvb3IlMjBuYXR1cmFsfGVufDF8fHx8MTc1ODQ1NDU0Nnww&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      "https://images.unsplash.com/photo-1646526822542-7c1d60db52cf?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxiZWF1dGlmdWwlMjBibGFjayUyMHdvbWFuJTIwZWxlZ2FudCUyMGZhc2hpb258ZW58MXx8fHwxNzU4NDU0NTQ5fDA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      "https://images.unsplash.com/photo-1753874383857-67db5aa3ff60?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxiZWF1dGlmdWwlMjBibG9uZGUlMjB3b21hbiUyMGJvaGVtaWFuJTIwc3R5bGV8ZW58MXx8fHwxNzU4NDU0NTUyfDA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      "https://images.unsplash.com/photo-1658699599150-87d313cf76a5?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxiZWF1dGlmdWwlMjBpbmRpYW4lMjB3b21hbiUyMHRyYWRpdGlvbmFsJTIwbW9kZXJufGVufDF8fHx8MTc1ODQ1NDU1NXww&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      "https://images.unsplash.com/photo-1621095058332-3c71ce927ee5?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxiZWF1dGlmdWwlMjByZWRoZWFkJTIwd29tYW4lMjBhcnRpc3RpYyUyMGNyZWF0aXZlfGVufDF8fHx8MTc1ODQ1NDU1OHww&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      "https://images.unsplash.com/photo-1659097618038-1a56dacdae99?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxiZWF1dGlmdWwlMjBtaWRkbGUlMjBlYXN0ZXJuJTIwd29tYW4lMjBidXNpbmVzcyUyMHByb2Zlc3Npb25hbHxlbnwxfHx8fDE3NTg0NTQ1NjB8MA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      "https://images.unsplash.com/photo-1635632195180-d637eda77d82?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxiZWF1dGlmdWwlMjBicnVuZXR0ZSUyMHdvbWFuJTIwY2FzdWFsJTIwbGlmZXN0eWxlfGVufDF8fHx8MTc1ODQ1NDU2NHww&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
      "https://images.unsplash.com/photo-1756699495345-6877309eb20b?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxiZWF1dGlmdWwlMjB3b21hbiUyMGZpdG5lc3MlMjBhdGhsZXRpYyUyMHNwb3J0eXxlbnwxfHx8fDE3NTg0NTQ1Njd8MA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral"
    ],
    interests: ["Technology", "Business", "Travel", "Innovation", "Networking", "Leadership", "Photography", "Coffee"],
    distance: 0,
    mutualFriends: 127,
    joinDate: "January 2020",
    introVideo: "https://commondatastorage.googleapis.com/gtv-videos-bucket/sample/BigBuckBunny.mp4",
    profileMusic: "https://www.soundjay.com/misc/sounds/magic-chime-02.wav",
    customBackground: "https://images.unsplash.com/photo-1519681393784-d120267933ba?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxyb21hbnRpYyUyMGJhY2tncm91bmR8ZW58MXx8fHwxNzU4MzE1OTAwfDA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
    musicTitle: "My Romantic Playlist",
    musicArtist: "Various Artists",
    lookingFor: {
      relationshipType: "Serious relationship",
      lookingForText: "Seeking a genuine connection with someone who values growth, authenticity, and building something meaningful together. I appreciate intelligence, humor, and emotional maturity. Looking for a true partnership where we can support each other's dreams while creating our own adventure.",
      ageRange: { min: 28, max: 40 },
      maxDistance: 25,
      dealBreakers: ["Dishonesty", "No ambition", "Disrespectful"],
      importantQualities: ["Honesty", "Ambition", "Humor", "Kindness", "Intelligence", "Authenticity"],
      countriesPreference: "multiple",
      selectedCountries: ["United States", "Canada", "United Kingdom"]
    }
  });

  const [selectedPhoto, setSelectedPhoto] = useState(userProfile.photos[0]);
  const [isLiked, setIsLiked] = useState(false);
  const [showReportDialog, setShowReportDialog] = useState(false);
  const [photoLikes, setPhotoLikes] = useState<{ [key: string]: { likes: number; isLiked: boolean } }>(
    userProfile.photos.reduce((acc, photo) => ({
      ...acc,
      [photo]: { likes: Math.floor(Math.random() * 500) + 50, isLiked: Math.random() > 0.7 }
    }), {})
  );

  const [isMusicMuted, setIsMusicMuted] = useState(false);
  const [showVideoModal, setShowVideoModal] = useState(false);
  const [showViewedMeModal, setShowViewedMeModal] = useState(false);
  const [showMapModal, setShowMapModal] = useState(false);
  
  const videoRef = useRef<HTMLVideoElement>(null);
  const audioRef = useRef<HTMLAudioElement>(null);
  const videoCloseTimeoutRef = useRef<NodeJS.Timeout | null>(null);
  
  useVideoManager(videoRef);
  useMusicManager(audioRef);

  useEffect(() => {
    if (userProfile.introVideo) {
      setIntroVideo(userProfile.introVideo);
    }
    if (userProfile.profileMusic) {
      setProfileMusic(userProfile.profileMusic);
    }
    if (userProfile.customBackground) {
      setCustomBackground(userProfile.customBackground);
    }
  }, [userProfile.introVideo, userProfile.profileMusic, userProfile.customBackground, setIntroVideo, setProfileMusic, setCustomBackground]);

  useEffect(() => {
    if (profileMusic && audioRef.current && !isVideoPlaying) {
      audioRef.current.muted = false;
      audioRef.current.volume = 0.3; // Set a moderate volume for auto-play
      
      const tryAutoPlay = () => {
        if (audioRef.current && profileMusic) {
          audioRef.current.play().then(() => {
            console.log('Profile music auto-playing for visiting member');
            playMusic(); // Ensure music state is updated
          }).catch((error) => {
            console.log('Auto-play prevented, will start on user interaction:', error.message);
          });
        }
      };

      // More aggressive auto-play attempts for visiting members
      tryAutoPlay();
      setTimeout(tryAutoPlay, 50);
      setTimeout(tryAutoPlay, 100);
      setTimeout(tryAutoPlay, 250);
      setTimeout(tryAutoPlay, 500);
      setTimeout(tryAutoPlay, 1000);
      setTimeout(tryAutoPlay, 2000);
    }
  }, [profileMusic, isVideoPlaying, playMusic]);

  const handleLike = () => {
    setIsLiked(!isLiked);
  };

  const handleMessage = () => {
    onNavigate("messages");
  };

  const handleShare = () => {
    console.log("Share profile");
  };

  const handleReport = () => {
    setShowReportDialog(true);
  };

  const handleViewedMe = () => {
    setShowViewedMeModal(true);
  };

  const handleMapClick = () => {
    setShowMapModal(true);
  };

  const handlePhotoSelect = (photoUrl: string) => {
    setSelectedPhoto(photoUrl);
  };

  const handlePhotoLike = (photoUrl: string, e: React.MouseEvent) => {
    e.stopPropagation();
    setPhotoLikes(prev => ({
      ...prev,
      [photoUrl]: {
        likes: prev[photoUrl].isLiked ? prev[photoUrl].likes - 1 : prev[photoUrl].likes + 1,
        isLiked: !prev[photoUrl].isLiked
      }
    }));
  };

  const toggleMusic = () => {
    if (isMusicPlaying || isMusicPaused) {
      pauseMusic();
    } else {
      playMusic();
    }
  };

  const toggleMusicMute = () => {
    if (audioRef.current) {
      audioRef.current.muted = !isMusicMuted;
      setIsMusicMuted(!isMusicMuted);
    }
  };

  const handleUserInteraction = () => {
    if (profileMusic && audioRef.current && !isMusicPlaying && !isVideoPlaying) {
      audioRef.current.volume = 0.3;
      audioRef.current.play().then(() => {
        console.log('Music started on user interaction for visiting member');
        playMusic();
      }).catch(console.error);
    }
  };

  const toggleVideo = () => {
    if (videoRef.current) {
      if (isVideoPlaying) {
        videoRef.current.pause();
      } else {
        videoRef.current.play().catch(console.error);
      }
    }
  };

  // Handle video ended event - auto-close 1 second after video finishes
  useEffect(() => {
    const video = videoRef.current;
    
    const handleVideoEnded = () => {
      // Clear any existing timeout
      if (videoCloseTimeoutRef.current) {
        clearTimeout(videoCloseTimeoutRef.current);
      }
      
      // Close video modal 1 second after it ends
      videoCloseTimeoutRef.current = setTimeout(() => {
        setShowVideoModal(false);
      }, 1000);
    };
    
    if (video) {
      video.addEventListener('ended', handleVideoEnded);
    }
    
    return () => {
      if (video) {
        video.removeEventListener('ended', handleVideoEnded);
      }
      if (videoCloseTimeoutRef.current) {
        clearTimeout(videoCloseTimeoutRef.current);
      }
    };
  }, [videoRef]);

  // Clean up timeout when modal closes manually
  useEffect(() => {
    if (!showVideoModal && videoCloseTimeoutRef.current) {
      clearTimeout(videoCloseTimeoutRef.current);
      videoCloseTimeoutRef.current = null;
    }
  }, [showVideoModal]);

  return (
    <div className="min-h-screen bg-background">
      {/* Background music auto-plays for visiting members */}
      {profileMusic && (
        <audio
          ref={audioRef}
          src={profileMusic}
          loop
          preload="auto"
          playsInline
          autoPlay
          muted={false}
        />
      )}

      <Dialog open={showVideoModal} onOpenChange={setShowVideoModal}>
        <DialogContent className="max-w-4xl p-0">
          <DialogHeader className="sr-only">
            <DialogTitle>Introduction Video</DialogTitle>
            <DialogDescription>
              Watch introduction video
            </DialogDescription>
          </DialogHeader>
          {introVideo && (
            <div className="relative">
              <video
                ref={videoRef}
                src={introVideo}
                className="w-full aspect-video rounded-lg"
                controls
                autoPlay
                muted={false}
                onPlay={() => {
                  // Video audio takes priority - ensure it's not muted
                  if (videoRef.current) {
                    videoRef.current.muted = false;
                  }
                  // Music will automatically pause via MediaManager
                }}
              />
              <div className="absolute top-4 right-4">
                <Button
                  size="sm"
                  variant="secondary"
                  onClick={() => setShowVideoModal(false)}
                  className="bg-black text-white hover:bg-gray-800"
                >
                  Close
                </Button>
              </div>
            </div>
          )}
        </DialogContent>
      </Dialog>

      <Dialog open={showViewedMeModal} onOpenChange={setShowViewedMeModal}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>Members Who Viewed Your Profile</DialogTitle>
            <DialogDescription>
              Recent profile visitors from the last 30 days
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4 max-h-96 overflow-y-auto">
            {/* Mock data for viewed me */}
            {[
              { name: "Sarah Johnson", age: 28, location: "Boston", time: "2 hours ago", photo: "https://images.unsplash.com/photo-1494790108755-2616b612b47c?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHx5b3VuZyUyMHdvbWFuJTIwcG9ydHJhaXR8ZW58MXx8fHwxNzU4MzE1ODkxfDA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral" },
              { name: "Emily Davis", age: 25, location: "Chicago", time: "5 hours ago", photo: "https://images.unsplash.com/photo-1438761681033-6461ffad8d80?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxhdHRyYWN0aXZlJTIwd29tYW4lMjBwb3J0cmFpdHxlbnwxfHx8fDE3NTgyMTczMTh8MA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral" },
              { name: "Jessica Wilson", age: 30, location: "Miami", time: "1 day ago", photo: "https://images.unsplash.com/photo-1607746882042-944635dfe10e?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxiZWF1dGlmdWwlMjB3b21hbiUyMHNtaWxlfGVufDF8fHx8MTc1ODMxNTg5N3ww&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral" },
              { name: "Amanda Brown", age: 27, location: "Seattle", time: "2 days ago", photo: "https://images.unsplash.com/photo-1534528741775-53994a69daeb?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxwcmV0dHklMjB3b21hbiUyMHBvcnRyYWl0fGVufDF8fHx8MTc1ODMxNTg5N3ww&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral" },
              { name: "Rachel Martinez", age: 29, location: "Denver", time: "3 days ago", photo: "https://images.unsplash.com/photo-1529626455594-4ff0802cfb7e?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxlbGVnYW50JTIwd29tYW4lMjBwb3J0cmFpdHxlbnwxfHx8fDE3NTgyMTczMTh8MA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral" }
            ].map((viewer, index) => (
              <div key={index} className="flex items-center gap-4 p-3 hover:bg-muted rounded-lg transition-colors cursor-pointer">
                <div className="relative">
                  <ImageWithFallback
                    src={viewer.photo}
                    alt={viewer.name}
                    className="w-12 h-12 rounded-full object-cover"
                  />
                  <div className="absolute -bottom-1 -right-1 w-4 h-4 bg-green-500 rounded-full border-2 border-white"></div>
                </div>
                
                <div className="flex-1">
                  <div className="flex items-center gap-2">
                    <h4 className="font-medium">{viewer.name}</h4>
                    <span className="text-sm text-muted-foreground">{viewer.age}</span>
                  </div>
                  <p className="text-sm text-muted-foreground">{viewer.location}</p>
                </div>
                
                <div className="text-right">
                  <p className="text-sm text-muted-foreground">{viewer.time}</p>
                  <Button size="sm" variant="outline" className="mt-1">
                    View Profile
                  </Button>
                </div>
              </div>
            ))}
          </div>
          
          <div className="flex justify-between items-center pt-4 border-t">
            <p className="text-sm text-muted-foreground">
              5 members viewed your profile in the last 30 days
            </p>
            <Button variant="outline" onClick={() => setShowViewedMeModal(false)}>
              Close
            </Button>
          </div>
        </DialogContent>
      </Dialog>

      <Dialog open={showMapModal} onOpenChange={setShowMapModal}>
        <DialogContent className="max-w-4xl p-0">
          <DialogHeader className="sr-only">
            <DialogTitle>Location Map</DialogTitle>
            <DialogDescription>
              View {userProfile.name}'s location on the map
            </DialogDescription>
          </DialogHeader>
          
          <div className="relative">
            {/* Google Maps Integration */}
            <GoogleMaps
              center={userProfile.coordinates}
              zoom={13}
              markers={[
                {
                  position: userProfile.coordinates,
                  title: `${userProfile.name}'s Location`,
                  info: `${userProfile.location}, ${userProfile.country}`
                }
              ]}
              className="w-full h-96"
              showControls={true}
            />
            
            {/* Modal Header */}
            <div className="absolute top-4 left-4 bg-white/95 dark:bg-gray-800/95 backdrop-blur-sm rounded-lg p-3 shadow-lg">
              <div className="flex items-center gap-3">
                <ImageWithFallback
                  src="https://images.unsplash.com/photo-1558721700-227001209525?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHx1bml0ZWQlMjBzdGF0ZXMlMjBmbGFnJTIwYW1lcmljYXxlbnwxfHx8fDE3NTg0ODIwMTl8MA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral"
                  alt={`${userProfile.country} flag`}
                  className="w-8 h-5 object-cover rounded-sm border"
                />
                <div>
                  <h3 className="font-semibold">{userProfile.location}</h3>
                  <p className="text-sm text-muted-foreground">{userProfile.country} {userProfile.countryFlag}</p>
                </div>
              </div>
            </div>
            
            {/* Close Button */}
            <div className="absolute top-4 right-4">
              <Button
                size="sm"
                variant="secondary"
                onClick={() => setShowMapModal(false)}
                className="bg-white/95 dark:bg-gray-800/95 backdrop-blur-sm hover:bg-white dark:hover:bg-gray-800"
              >
                Close
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>

      <div className="relative z-10" onClick={handleUserInteraction} onTouchStart={handleUserInteraction}>
        <Header currentView="user-profile" onNavigate={onNavigate} />
        
        {/* Navigation Bar */}
        <div className="w-full bg-background/95 backdrop-blur-sm border-b border-border/20 sticky top-0 z-20">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-4">
            <div className="flex items-center justify-between">
              <Button
                variant="ghost"
                onClick={() => onNavigate("search")}
                className="flex items-center gap-2"
              >
                <span>←</span>
                Back to Search
              </Button>
              
              <div className="bg-blue-50 dark:bg-blue-900 px-4 py-2 rounded-lg border border-blue-200 dark:border-blue-800">
                <p className="text-sm text-blue-700 dark:text-blue-400 flex items-center gap-2">
                  <Eye className="w-4 h-4" />
                  This is how other members see your profile
                </p>
              </div>
            </div>
          </div>
        </div>

        {/* Main Profile Content */}
        <div className="w-full">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
            
            {/* User Information Above Photos */}
            <div className="mb-6">
              <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between mb-2 gap-4">
                <div className="flex flex-wrap items-center gap-3">
                  <h1 className="text-2xl sm:text-3xl font-bold">{userProfile.name}</h1>
                  <span className="text-xl sm:text-2xl text-muted-foreground">{userProfile.age}</span>
                  {userProfile.isVerified && (
                    <Badge variant="secondary" className="flex items-center gap-1">
                      <Verified className="w-3 h-3 text-blue-500" />
                      Verified
                    </Badge>
                  )}
                  {userProfile.isOnline && (
                    <Badge className="bg-green-500 text-white">
                      Online Now
                    </Badge>
                  )}
                  {userProfile.isPremium && (
                    <Badge className="bg-gradient-to-r from-yellow-500 to-yellow-600 text-white">
                      <Crown className="w-4 h-4 mr-1" />
                      Premium
                    </Badge>
                  )}
                </div>
                
                {introVideo && (
                  <Button
                    onClick={() => setShowVideoModal(true)}
                    className="flex items-center gap-2 bg-red-600 hover:bg-red-700 text-white w-full sm:w-auto"
                  >
                    <Video className="w-4 h-4" />
                    Watch Introduction
                  </Button>
                )}
              </div>
              
              {/* Location Information */}
              <div className="flex flex-wrap items-center gap-2 text-muted-foreground mb-2">
                <button
                  onClick={handleMapClick}
                  className="flex items-center gap-2 hover:text-primary transition-colors cursor-pointer"
                >
                  <MapPin className="w-4 h-4" />
                  <span>{userProfile.location}, {userProfile.country}</span>
                </button>
                <ImageWithFallback
                  src="https://images.unsplash.com/photo-1558721700-227001209525?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHx1bml0ZWQlMjBzdGF0ZXMlMjBmbGFnJTIwYW1lcmljYXxlbnwxfHx8fDE3NTg0ODIwMTl8MA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral"
                  alt={`${userProfile.country} flag`}
                  className="w-6 h-4 object-cover rounded-sm border"
                />
                <span>{userProfile.countryFlag}</span>
              </div>
            </div>

            {/* Photo Collage - Full Width */}
            <div className="mb-6">
              <PhotoCollage
                photos={userProfile.photos}
                onPhotoSelect={handlePhotoSelect}
                selectedPhoto={selectedPhoto}
                photoLikes={photoLikes}
                onPhotoLike={handlePhotoLike}
              />
            </div>
            
            {/* Action Buttons */}
            <div className="flex flex-wrap items-center justify-start gap-3 mb-8">
              <Button onClick={handleLike} variant={isLiked ? "default" : "outline"} size="lg" className="flex-1 sm:flex-none sm:min-w-36">
                <Heart className="w-5 h-5 mr-2" />
                {isLiked ? 'Liked' : 'Like'}
              </Button>
              
              <Button onClick={handleMessage} size="lg" className="flex-1 sm:flex-none sm:min-w-36">
                <MessageCircle className="w-5 h-5 mr-2" />
                Message
              </Button>
              
              <Button onClick={handleViewedMe} variant="outline" size="lg" className="flex-1 sm:flex-none sm:min-w-36">
                <UserCheck className="w-5 h-5 mr-2" />
                Viewed Me
              </Button>
              
              <DropdownMenu>
                <DropdownMenuTrigger asChild>
                  <Button variant="outline" size="lg">
                    <MoreHorizontal className="w-5 h-5" />
                  </Button>
                </DropdownMenuTrigger>
                <DropdownMenuContent align="end">
                  <DropdownMenuItem onClick={handleShare}>
                    <Share2 className="w-4 h-4 mr-2" />
                    Share Profile
                  </DropdownMenuItem>
                  <DropdownMenuItem onClick={handleReport}>
                    <Flag className="w-4 h-4 mr-2" />
                    Report User
                  </DropdownMenuItem>
                </DropdownMenuContent>
              </DropdownMenu>
            </div>

            <div className="space-y-6">
              {/* About Me Section - Enhanced */}
              <Card 
                className="overflow-hidden border-0 shadow-lg"
                style={{
                  background: `linear-gradient(135deg, ${theme.gradient1}15, ${theme.gradient2}15, ${theme.gradient3}15)`
                }}
              >
                <div className="p-6">
                  <div className="flex items-center gap-3 mb-6">
                    <div 
                      className="p-2 rounded-full"
                      style={{ background: `linear-gradient(135deg, ${theme.primary}, ${theme.secondary})` }}
                    >
                      <Users className="w-6 h-6 text-white" />
                    </div>
                    <div>
                      <h3 
                        className="text-2xl font-bold bg-clip-text text-transparent"
                        style={{
                          background: `linear-gradient(135deg, ${theme.primary}, ${theme.secondary})`,
                          WebkitBackgroundClip: 'text',
                          WebkitTextFillColor: 'transparent'
                        }}
                      >
                        About Me
                      </h3>
                      <p className="text-sm text-muted-foreground">Get to know who I really am</p>
                    </div>
                  </div>
                  
                  <div className="relative">
                    <div 
                      className="absolute left-0 top-0 w-1 h-full rounded-full"
                      style={{ background: `linear-gradient(135deg, ${theme.primary}, ${theme.secondary})` }}
                    />
                    <div className="pl-6">
                      <p className="text-foreground leading-relaxed text-base sm:text-lg font-medium mb-4">
                        {userProfile.bio}
                      </p>
                      
                      {/* Personality Indicators */}
                      <div className="flex flex-wrap gap-2 mt-4">
                        <Badge 
                          variant="secondary" 
                          className="border"
                          style={{ 
                            backgroundColor: `${theme.primary}20`,
                            color: theme.primary,
                            borderColor: `${theme.primary}40`
                          }}
                        >
                          <Star className="w-3 h-3 mr-1" />
                          Authentic
                        </Badge>
                        <Badge 
                          variant="secondary" 
                          className="border"
                          style={{ 
                            backgroundColor: `${theme.secondary}20`,
                            color: theme.secondary,
                            borderColor: `${theme.secondary}40`
                          }}
                        >
                          <Heart className="w-3 h-3 mr-1" />
                          Passionate
                        </Badge>
                        <Badge 
                          variant="secondary" 
                          className="border"
                          style={{ 
                            backgroundColor: `${theme.accent}20`,
                            color: theme.accent,
                            borderColor: `${theme.accent}40`
                          }}
                        >
                          <Target className="w-3 h-3 mr-1" />
                          Goal-oriented
                        </Badge>
                      </div>
                    </div>
                  </div>
                </div>
                
                {/* Bottom accent */}
                <div 
                  className="h-2"
                  style={{
                    background: `linear-gradient(135deg, ${theme.gradient1}, ${theme.gradient2}, ${theme.gradient3})`
                  }}
                />
              </Card>

              {/* Quick Info Grid - Enhanced */}
              <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4">
                <Card 
                  className="group relative overflow-hidden border-0 hover:shadow-xl hover:scale-105 transition-all duration-300"
                  style={{
                    background: `linear-gradient(135deg, ${theme.primary}15, ${theme.primary}25)`
                  }}
                >
                  <div 
                    className="absolute inset-0"
                    style={{ background: `linear-gradient(135deg, ${theme.primary}10, transparent)` }}
                  />
                  <div className="relative p-6 text-center">
                    <div 
                      className="mb-3 mx-auto w-12 h-12 rounded-full flex items-center justify-center group-hover:scale-110 transition-transform duration-300"
                      style={{ backgroundColor: theme.primary }}
                    >
                      <Briefcase className="w-6 h-6 text-white" />
                    </div>
                    <span 
                      className="block text-xs font-medium uppercase tracking-wide mb-1"
                      style={{ color: theme.primary }}
                    >
                      Occupation
                    </span>
                    <p className="font-bold text-gray-900 dark:text-gray-100">{userProfile.occupation}</p>
                  </div>
                </Card>
                
                <Card 
                  className="group relative overflow-hidden border-0 hover:shadow-xl hover:scale-105 transition-all duration-300"
                  style={{
                    background: `linear-gradient(135deg, ${theme.secondary}15, ${theme.secondary}25)`
                  }}
                >
                  <div 
                    className="absolute inset-0"
                    style={{ background: `linear-gradient(135deg, ${theme.secondary}10, transparent)` }}
                  />
                  <div className="relative p-6 text-center">
                    <div 
                      className="mb-3 mx-auto w-12 h-12 rounded-full flex items-center justify-center group-hover:scale-110 transition-transform duration-300"
                      style={{ backgroundColor: theme.secondary }}
                    >
                      <GraduationCap className="w-6 h-6 text-white" />
                    </div>
                    <span 
                      className="block text-xs font-medium uppercase tracking-wide mb-1"
                      style={{ color: theme.secondary }}
                    >
                      Education
                    </span>
                    <p className="font-bold text-gray-900 dark:text-gray-100">{userProfile.education}</p>
                  </div>
                </Card>
                
                <Card 
                  className="group relative overflow-hidden border-0 hover:shadow-xl hover:scale-105 transition-all duration-300"
                  style={{
                    background: `linear-gradient(135deg, ${theme.accent}15, ${theme.accent}25)`
                  }}
                >
                  <div 
                    className="absolute inset-0"
                    style={{ background: `linear-gradient(135deg, ${theme.accent}10, transparent)` }}
                  />
                  <div className="relative p-6 text-center">
                    <div 
                      className="mb-3 mx-auto w-12 h-12 rounded-full flex items-center justify-center group-hover:scale-110 transition-transform duration-300"
                      style={{ backgroundColor: theme.accent }}
                    >
                      <Ruler className="w-6 h-6 text-white" />
                    </div>
                    <span 
                      className="block text-xs font-medium uppercase tracking-wide mb-1"
                      style={{ color: theme.accent }}
                    >
                      Height
                    </span>
                    <p className="font-bold text-gray-900 dark:text-gray-100">{userProfile.height}</p>
                  </div>
                </Card>
                
                <Card 
                  className="group relative overflow-hidden border-0 hover:shadow-xl hover:scale-105 transition-all duration-300"
                  style={{
                    background: `linear-gradient(135deg, ${theme.gradient3}15, ${theme.gradient3}25)`
                  }}
                >
                  <div 
                    className="absolute inset-0"
                    style={{ background: `linear-gradient(135deg, ${theme.gradient3}10, transparent)` }}
                  />
                  <div className="relative p-6 text-center">
                    <div 
                      className="mb-3 mx-auto w-12 h-12 rounded-full flex items-center justify-center group-hover:scale-110 transition-transform duration-300"
                      style={{ backgroundColor: theme.gradient3 }}
                    >
                      <Calendar className="w-6 h-6 text-white" />
                    </div>
                    <span 
                      className="block text-xs font-medium uppercase tracking-wide mb-1"
                      style={{ color: theme.gradient3 }}
                    >
                      Joined
                    </span>
                    <p className="font-bold text-gray-900 dark:text-gray-100">{userProfile.joinDate}</p>
                  </div>
                </Card>
              </div>

              {/* Interests Section - Enhanced */}
              <Card 
                className="overflow-hidden border-0 shadow-lg"
                style={{
                  background: `linear-gradient(135deg, ${theme.gradient2}15, ${theme.gradient3}15, ${theme.accent}15)`
                }}
              >
                <div className="p-6">
                  <div className="flex items-center gap-3 mb-6">
                    <div 
                      className="p-2 rounded-full"
                      style={{ background: `linear-gradient(135deg, ${theme.secondary}, ${theme.accent})` }}
                    >
                      <Star className="w-6 h-6 text-white" />
                    </div>
                    <div>
                      <h3 
                        className="text-2xl font-bold bg-clip-text text-transparent"
                        style={{
                          background: `linear-gradient(135deg, ${theme.secondary}, ${theme.accent})`,
                          WebkitBackgroundClip: 'text',
                          WebkitTextFillColor: 'transparent'
                        }}
                      >
                        My Interests
                      </h3>
                      <p className="text-sm text-muted-foreground">Things that make me passionate</p>
                    </div>
                  </div>
                  
                  <div className="flex flex-wrap gap-3">
                    {userProfile.interests.map((interest, index) => {
                      const colorIndex = index % 3;
                      const colors = [theme.primary, theme.secondary, theme.accent];
                      const selectedColor = colors[colorIndex];
                      
                      return (
                        <Badge 
                          key={index} 
                          variant="secondary" 
                          className="px-4 py-2 bg-white/80 dark:bg-gray-800/80 border-2 hover:scale-105 transition-all duration-200 font-medium shadow-sm"
                          style={{
                            color: selectedColor,
                            borderColor: `${selectedColor}40`,
                            backgroundColor: `${selectedColor}10`
                          }}
                        >
                          <span className="mr-1">✨</span>
                          {interest}
                        </Badge>
                      );
                    })}
                  </div>
                </div>
                
                {/* Bottom accent */}
                <div 
                  className="h-2"
                  style={{
                    background: `linear-gradient(135deg, ${theme.secondary}, ${theme.accent}, ${theme.gradient3})`
                  }}
                />
              </Card>

              {/* What I'm Looking For Section - Enhanced */}
              {userProfile.lookingFor && (
                <Card 
                  className="overflow-hidden border-0 shadow-lg"
                  style={{
                    background: `linear-gradient(135deg, ${theme.accent}15, ${theme.gradient1}15, ${theme.gradient2}15)`
                  }}
                >
                  <div className="p-6">
                    <div className="flex items-center gap-3 mb-6">
                      <div 
                        className="p-2 rounded-full"
                        style={{ background: `linear-gradient(135deg, ${theme.accent}, ${theme.gradient1})` }}
                      >
                        <Target className="w-6 h-6 text-white" />
                      </div>
                      <div>
                        <h3 
                          className="text-2xl font-bold bg-clip-text text-transparent"
                          style={{
                            background: `linear-gradient(135deg, ${theme.accent}, ${theme.gradient1})`,
                            WebkitBackgroundClip: 'text',
                            WebkitTextFillColor: 'transparent'
                          }}
                        >
                          What I'm Looking For
                        </h3>
                        <p className="text-sm text-muted-foreground">My ideal connection</p>
                      </div>
                    </div>
                  
                  <div className="space-y-4">
                    <div className="bg-white/60 dark:bg-gray-800/60 rounded-xl p-4 border border-rose-200 dark:border-rose-800">
                      <h4 className="font-semibold mb-3 text-rose-800 dark:text-rose-200 flex items-center gap-2">
                        <Heart className="w-4 h-4" />
                        Relationship Type
                      </h4>
                      <Badge className="bg-rose-100 text-rose-800 dark:bg-rose-900/50 dark:text-rose-200 px-3 py-1 font-medium">
                        {userProfile.lookingFor.relationshipType}
                      </Badge>
                    </div>
                    
                    <div className="bg-white/60 dark:bg-gray-800/60 rounded-xl p-4 border border-pink-200 dark:border-pink-800">
                      <h4 className="font-semibold mb-3 text-pink-800 dark:text-pink-200 flex items-center gap-2">
                        <Users className="w-4 h-4" />
                        About My Ideal Match
                      </h4>
                      <p className="text-gray-700 dark:text-gray-300 leading-relaxed">
                        {userProfile.lookingFor.lookingForText}
                      </p>
                    </div>
                    
                    <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                      <div className="bg-white/60 dark:bg-gray-800/60 rounded-xl p-4 border border-purple-200 dark:border-purple-800">
                        <h4 className="font-semibold mb-2 text-purple-800 dark:text-purple-200 flex items-center gap-2">
                          <Calendar className="w-4 h-4" />
                          Age Range
                        </h4>
                        <p className="text-gray-700 dark:text-gray-300 font-medium">
                          {userProfile.lookingFor.ageRange.min} - {userProfile.lookingFor.ageRange.max} years
                        </p>
                      </div>
                      <div className="bg-white/60 dark:bg-gray-800/60 rounded-xl p-4 border border-blue-200 dark:border-blue-800">
                        <h4 className="font-semibold mb-2 text-blue-800 dark:text-blue-200 flex items-center gap-2">
                          <MapPin className="w-4 h-4" />
                          Distance
                        </h4>
                        <p className="text-gray-700 dark:text-gray-300 font-medium">
                          Within {userProfile.lookingFor.maxDistance} miles
                        </p>
                      </div>
                    </div>
                    
                    <div className="bg-white/60 dark:bg-gray-800/60 rounded-xl p-4 border border-green-200 dark:border-green-800">
                      <h4 className="font-semibold mb-3 text-green-800 dark:text-green-200 flex items-center gap-2">
                        <Star className="w-4 h-4" />
                        Important Qualities
                      </h4>
                      <div className="flex flex-wrap gap-2">
                        {userProfile.lookingFor.importantQualities.map((quality, index) => (
                          <Badge key={index} variant="secondary" className="bg-green-100 text-green-800 dark:bg-green-900/50 dark:text-green-200 border border-green-300 dark:border-green-700 px-3 py-1">
                            <span className="mr-1">✓</span>
                            {quality}
                          </Badge>
                        ))}
                      </div>
                    </div>
                    
                    <div className="bg-white/60 dark:bg-gray-800/60 rounded-xl p-4 border border-red-200 dark:border-red-800">
                      <h4 className="font-semibold mb-3 text-red-800 dark:text-red-200 flex items-center gap-2">
                        <Shield className="w-4 h-4" />
                        Deal Breakers
                      </h4>
                      <div className="flex flex-wrap gap-2">
                        {userProfile.lookingFor.dealBreakers.map((dealBreaker, index) => (
                          <Badge key={index} variant="secondary" className="bg-red-100 text-red-800 dark:bg-red-900/50 dark:text-red-200 border border-red-300 dark:border-red-700 px-3 py-1">
                            <span className="mr-1">✗</span>
                            {dealBreaker}
                          </Badge>
                        ))}
                      </div>
                    </div>
                    
                    {userProfile.lookingFor.countriesPreference !== "all" && (
                      <div className="bg-white/60 dark:bg-gray-800/60 rounded-xl p-4 border border-indigo-200 dark:border-indigo-800">
                        <h4 className="font-semibold mb-3 text-indigo-800 dark:text-indigo-200 flex items-center gap-2">
                          <Globe className="w-4 h-4" />
                          Preferred Countries
                        </h4>
                        <div className="flex flex-wrap gap-2">
                          {userProfile.lookingFor.selectedCountries.map((country, index) => (
                            <Badge key={index} variant="outline" className="bg-indigo-50 dark:bg-indigo-900/30 text-indigo-700 dark:text-indigo-300 border-indigo-300 dark:border-indigo-700 flex items-center gap-1 px-3 py-1">
                              <span className="mr-1">🌍</span>
                              {country}
                            </Badge>
                          ))}
                        </div>
                      </div>
                    )}
                  </div>
                  
                  {/* Bottom accent */}
                  <div 
                    className="h-2"
                    style={{
                      background: `linear-gradient(135deg, ${theme.accent}, ${theme.gradient1}, ${theme.gradient2})`
                    }}
                  />
                </div>
              </Card>
              )}


            </div>
          </div>
        </div>
        
        <Footer />
      </div>
    </div>
  );
}

export function UserProfilePage(props: UserProfilePageProps) {
  return (
    <MediaProvider>
      <UserProfileContent {...props} />
    </MediaProvider>
  );
}