import { useState } from "react";
import { Card } from "../ui/card";
import { Button } from "../ui/button";
import { Badge } from "../ui/badge";
import { Avatar, AvatarImage, AvatarFallback } from "../ui/avatar";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription, DialogFooter } from "../ui/dialog";
import { Textarea } from "../ui/textarea";
import { Alert, AlertDescription } from "../ui/alert";
import { 
  Trophy, 
  Users, 
  Play, 
  Flag,
  Crown,
  Zap,
  Target,
  Star,
  MessageCircle,
  AlertCircle,
  ArrowLeft
} from "lucide-react";
import { motion, AnimatePresence } from "motion/react";
import { useUser } from "../UserProvider";
import { usePoints } from "../PointsProvider";
import { toast } from "sonner@2.0.3";

interface JeopardyQuestion {
  id: string;
  category: string;
  points: number;
  question: string;
  answer: string;
  difficulty: "easy" | "medium" | "hard";
  isAnswered: boolean;
}

interface GameRoom {
  id: string;
  players: Array<{
    id: string;
    name: string;
    avatar: string;
    score: number;
    isOnline: boolean;
  }>;
  status: "waiting" | "playing" | "finished";
  currentQuestion: JeopardyQuestion | null;
  startTime: Date;
  maxPlayers: 2;
}

interface ReportModalProps {
  isOpen: boolean;
  onClose: () => void;
  reportedPlayer: string;
  gameId: string;
}

function ReportModal({ isOpen, onClose, reportedPlayer }: ReportModalProps) {
  const [reason, setReason] = useState("");
  const [isSubmitting, setIsSubmitting] = useState(false);

  const handleSubmit = async () => {
    if (!reason.trim()) {
      toast.error("Please provide a reason for reporting");
      return;
    }

    setIsSubmitting(true);
    await new Promise(resolve => setTimeout(resolve, 1000));
    
    toast.success("Report submitted successfully");
    setReason("");
    setIsSubmitting(false);
    onClose();
  };

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-[90vw]">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2 text-base">
            <Flag className="w-4 h-4 text-red-500" />
            Report Player
          </DialogTitle>
          <DialogDescription className="text-sm">
            Reporting: <span className="font-semibold">{reportedPlayer}</span>
          </DialogDescription>
        </DialogHeader>
        
        <div className="space-y-3">
          <Textarea
            value={reason}
            onChange={(e) => setReason(e.target.value)}
            placeholder="Describe the abusive behavior..."
            className="min-h-[100px] text-sm"
          />
          
          <Alert className="py-2">
            <AlertCircle className="h-3 w-3" />
            <AlertDescription className="text-xs">
              False reports may result in account restrictions.
            </AlertDescription>
          </Alert>
        </div>
        
        <DialogFooter className="flex gap-2">
          <Button variant="outline" onClick={onClose} disabled={isSubmitting} size="sm">
            Cancel
          </Button>
          <Button 
            onClick={handleSubmit} 
            disabled={isSubmitting}
            className="bg-red-600 hover:bg-red-700"
            size="sm"
          >
            {isSubmitting ? "Submitting..." : "Submit"}
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}

interface MobileDatingGamesPageProps {
  onNavigate: (view: string) => void;
}

export default function MobileDatingGamesPage({ onNavigate }: MobileDatingGamesPageProps) {
  const { user } = useUser();
  const { addPoints } = usePoints();
  const [gameState, setGameState] = useState<"lobby" | "playing" | "results">("lobby");
  const [currentRoom, setCurrentRoom] = useState<GameRoom | null>(null);
  const [selectedQuestion, setSelectedQuestion] = useState<JeopardyQuestion | null>(null);
  const [userAnswer, setUserAnswer] = useState("");
  const [showAnswer, setShowAnswer] = useState(false);
  const [reportModalOpen, setReportModalOpen] = useState(false);
  const [reportedPlayer, setReportedPlayer] = useState("");

  const categories = ["First Date", "Love Facts", "Languages", "Etiquette", "History"];

  const [board, setBoard] = useState<JeopardyQuestion[][]>([
    [
      { id: "1-1", category: "First Date", points: 100, question: "What % of first dates lead to a second?", answer: "53%", difficulty: "easy", isAnswered: false },
      { id: "1-2", category: "First Date", points: 200, question: "Most popular first date activity?", answer: "Dinner/Coffee", difficulty: "easy", isAnswered: false },
      { id: "1-3", category: "First Date", points: 300, question: "How many minutes late is OK?", answer: "5-10 min", difficulty: "medium", isAnswered: false },
      { id: "1-4", category: "First Date", points: 400, question: "Recommended first date duration?", answer: "1-2 hours", difficulty: "medium", isAnswered: false },
      { id: "1-5", category: "First Date", points: 500, question: "% who check phone on date?", answer: "75%", difficulty: "hard", isAnswered: false },
    ],
    [
      { id: "2-1", category: "Love Facts", points: 100, question: "Average first date length?", answer: "2-3 hours", difficulty: "easy", isAnswered: false },
      { id: "2-2", category: "Love Facts", points: 200, question: "Dates before becoming exclusive?", answer: "5-6", difficulty: "easy", isAnswered: false },
      { id: "2-3", category: "Love Facts", points: 300, question: "Most popular first date day?", answer: "Saturday", difficulty: "medium", isAnswered: false },
      { id: "2-4", category: "Love Facts", points: 400, question: "Time before saying 'I love you'?", answer: "3-4 months", difficulty: "medium", isAnswered: false },
      { id: "2-5", category: "Love Facts", points: 500, question: "% of relationships starting online?", answer: "40%", difficulty: "hard", isAnswered: false },
    ],
    [
      { id: "3-1", category: "Languages", points: 100, question: "Name a love language", answer: "Words/Acts/Gifts/Time/Touch", difficulty: "easy", isAnswered: false },
      { id: "3-2", category: "Languages", points: 200, question: "Who created love languages?", answer: "Gary Chapman", difficulty: "easy", isAnswered: false },
      { id: "3-3", category: "Languages", points: 300, question: "Most common love language?", answer: "Words of Affirmation", difficulty: "medium", isAnswered: false },
      { id: "3-4", category: "Languages", points: 400, question: "Can you have 2 love languages?", answer: "Yes", difficulty: "medium", isAnswered: false },
      { id: "3-5", category: "Languages", points: 500, question: "Year the book was published?", answer: "1992", difficulty: "hard", isAnswered: false },
    ],
    [
      { id: "4-1", category: "Etiquette", points: 100, question: "Split bill on first date?", answer: "Discuss/offer", difficulty: "easy", isAnswered: false },
      { id: "4-2", category: "Etiquette", points: 200, question: "When to text after date?", answer: "Within 24 hours", difficulty: "easy", isAnswered: false },
      { id: "4-3", category: "Etiquette", points: 300, question: "Check phone during date?", answer: "Only emergencies", difficulty: "medium", isAnswered: false },
      { id: "4-4", category: "Etiquette", points: 400, question: "Days to wait for 2nd date?", answer: "3-5 days", difficulty: "medium", isAnswered: false },
      { id: "4-5", category: "Etiquette", points: 500, question: "Topics to avoid on dates?", answer: "Ex/politics/religion/money", difficulty: "hard", isAnswered: false },
    ],
    [
      { id: "5-1", category: "History", points: 100, question: "Valentine's Day date?", answer: "Feb 14", difficulty: "easy", isAnswered: false },
      { id: "5-2", category: "History", points: 200, question: "Most romantic flower?", answer: "Rose", difficulty: "easy", isAnswered: false },
      { id: "5-3", category: "History", points: 300, question: "Where did rings originate?", answer: "Ancient Rome", difficulty: "medium", isAnswered: false },
      { id: "5-4", category: "History", points: 400, question: "Online dating started when?", answer: "1995", difficulty: "medium", isAnswered: false },
      { id: "5-5", category: "History", points: 500, question: "Who invented speed dating?", answer: "Rabbi Yaacov Deyo", difficulty: "hard", isAnswered: false },
    ],
  ]);

  const [leaderboard] = useState([
    { rank: 1, name: "Emma W.", score: 2500, winRate: 75 },
    { rank: 2, name: "James M.", score: 2300, winRate: 67 },
    { rank: 3, name: "Sofia G.", score: 2100, winRate: 80 },
  ]);

  const handleQuickMatch = () => {
    toast.info("Searching for opponent...");
    
    setTimeout(() => {
      const newRoom: GameRoom = {
        id: "new-game",
        players: [
          { id: user?.id || "current", name: user?.name || "You", avatar: user?.profilePicture || "", score: 0, isOnline: true },
          { id: "opp", name: "Jessica T.", avatar: "https://images.unsplash.com/photo-1438761681033-6461ffad8d80?w=100", score: 0, isOnline: true },
        ],
        status: "playing",
        currentQuestion: null,
        startTime: new Date(),
        maxPlayers: 2,
      };
      
      setCurrentRoom(newRoom);
      setGameState("playing");
      toast.success("Match found!");
    }, 2000);
  };

  const handleSelectQuestion = (categoryIndex: number, questionIndex: number) => {
    const question = board[categoryIndex][questionIndex];
    
    if (question.isAnswered) {
      toast.error("Already answered!");
      return;
    }

    setSelectedQuestion(question);
    setShowAnswer(false);
    setUserAnswer("");
  };

  const handleSubmitAnswer = () => {
    if (!selectedQuestion || !currentRoom) return;

    const newBoard = [...board];
    const categoryIndex = categories.indexOf(selectedQuestion.category);
    const questionIndex = newBoard[categoryIndex].findIndex(q => q.id === selectedQuestion.id);
    newBoard[categoryIndex][questionIndex].isAnswered = true;
    setBoard(newBoard);

    const isCorrect = userAnswer.toLowerCase().includes(selectedQuestion.answer.toLowerCase().split(" ")[0]);
    
    if (isCorrect) {
      const newRoom = { ...currentRoom };
      newRoom.players[0].score += selectedQuestion.points;
      setCurrentRoom(newRoom);
      addPoints(selectedQuestion.points);
      toast.success(`Correct! +${selectedQuestion.points} points`);
    } else {
      toast.error("Incorrect!");
    }

    setShowAnswer(true);
    
    setTimeout(() => {
      setSelectedQuestion(null);
      setUserAnswer("");
      setShowAnswer(false);
    }, 3000);
  };

  const handleEndGame = () => {
    if (!currentRoom) return;

    const winner = currentRoom.players.reduce((prev, current) => 
      (prev.score > current.score) ? prev : current
    );

    if (winner.id === user?.id || winner.id === "current") {
      addPoints(500);
      toast.success("🎉 You won! +500 bonus!");
    } else {
      addPoints(100);
      toast.info("Good game! +100 points");
    }

    setGameState("results");
  };

  const handleNewGame = () => {
    setGameState("lobby");
    setCurrentRoom(null);
    const resetBoard = board.map(category => 
      category.map(q => ({ ...q, isAnswered: false }))
    );
    setBoard(resetBoard);
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-pink-50 via-purple-50 to-blue-50 dark:from-gray-900 dark:via-purple-900/20 dark:to-gray-900 pb-20">
      {/* Header */}
      <div className="bg-gradient-to-r from-purple-600 via-purple-700 to-purple-800 dark:from-purple-700 dark:via-purple-800 dark:to-purple-900 text-white sticky top-0 z-20">
        <div className="flex items-center justify-between p-4">
          <Button
            variant="ghost"
            size="sm"
            className="text-white p-2 -ml-2"
            onClick={() => onNavigate('discover')}
          >
            <ArrowLeft className="h-5 w-5" />
          </Button>
          <div className="flex-1 text-center">
            <h1 className="text-lg font-semibold">Dating Jeopardy</h1>
            <p className="text-xs text-purple-100">Test your knowledge</p>
          </div>
          <div className="w-9"></div>
        </div>
      </div>

      <div className="p-4 space-y-4">
        {gameState === "lobby" && (
          <>
            {/* Quick Actions */}
            <div className="grid gap-3">
              <Card className="p-4 border-2 border-purple-200 dark:border-purple-800">
                <div className="flex items-start gap-3">
                  <div className="bg-purple-100 dark:bg-purple-900/30 p-2 rounded-lg flex-shrink-0">
                    <Zap className="w-5 h-5 text-purple-600" />
                  </div>
                  <div className="flex-1 min-w-0">
                    <h3 className="font-semibold mb-1">Quick Match</h3>
                    <p className="text-xs text-muted-foreground mb-3">
                      Find opponent & play now
                    </p>
                    <Button 
                      onClick={handleQuickMatch}
                      size="sm"
                      className="w-full bg-gradient-to-r from-purple-600 to-pink-600"
                    >
                      <Play className="w-3 h-3 mr-1" />
                      Start Match
                    </Button>
                  </div>
                </div>
              </Card>

              <Card className="p-4 border-2 border-blue-200 dark:border-blue-800">
                <div className="flex items-start gap-3">
                  <div className="bg-blue-100 dark:bg-blue-900/30 p-2 rounded-lg flex-shrink-0">
                    <Users className="w-5 h-5 text-blue-600" />
                  </div>
                  <div className="flex-1 min-w-0">
                    <h3 className="font-semibold mb-1">Join Game</h3>
                    <p className="text-xs text-muted-foreground mb-3">
                      Browse active games
                    </p>
                    <Button 
                      variant="outline"
                      size="sm"
                      className="w-full"
                    >
                      <Target className="w-3 h-3 mr-1" />
                      Browse
                    </Button>
                  </div>
                </div>
              </Card>
            </div>

            {/* Leaderboard */}
            <Card className="p-4">
              <div className="flex items-center gap-2 mb-3">
                <Crown className="w-4 h-4 text-yellow-600" />
                <h2 className="font-semibold">Top Players</h2>
              </div>
              
              <div className="space-y-2">
                {leaderboard.map((player) => (
                  <div
                    key={player.rank}
                    className="flex items-center gap-3 p-3 rounded-lg bg-gradient-to-r from-purple-50 to-pink-50 dark:from-purple-900/20 dark:to-pink-900/20"
                  >
                    <div className={`flex items-center justify-center w-6 h-6 rounded-full text-xs font-bold ${
                      player.rank === 1 ? "bg-yellow-400 text-yellow-900" :
                      player.rank === 2 ? "bg-gray-300 text-gray-700" :
                      "bg-orange-400 text-orange-900"
                    }`}>
                      {player.rank}
                    </div>
                    
                    <div className="flex-1 min-w-0">
                      <div className="font-semibold text-sm truncate">{player.name}</div>
                      <div className="text-xs text-muted-foreground">
                        {player.winRate}% win rate
                      </div>
                    </div>
                    
                    <div className="text-right">
                      <div className="font-bold text-purple-600 text-sm">
                        {player.score.toLocaleString()}
                      </div>
                      <div className="text-xs text-muted-foreground">pts</div>
                    </div>
                  </div>
                ))}
              </div>
            </Card>

            {/* How to Play */}
            <Card className="p-4 bg-gradient-to-br from-purple-50 to-pink-50 dark:from-purple-900/20 dark:to-pink-900/20">
              <h2 className="font-semibold mb-3 flex items-center gap-2 text-sm">
                <Star className="w-4 h-4 text-purple-600" />
                How to Play
              </h2>
              <div className="space-y-2">
                {[
                  "Select category & point value",
                  "Read the dating question",
                  "Submit answer before time's up",
                  "Earn points for correct answers",
                  "Most points wins!"
                ].map((step, idx) => (
                  <div key={idx} className="flex gap-2 items-start">
                    <div className="flex-shrink-0 w-5 h-5 rounded-full bg-purple-600 text-white flex items-center justify-center text-xs font-bold mt-0.5">
                      {idx + 1}
                    </div>
                    <p className="text-xs flex-1">{step}</p>
                  </div>
                ))}
              </div>
            </Card>
          </>
        )}

        {gameState === "playing" && currentRoom && (
          <>
            {/* Game Header */}
            <Card className="p-3 bg-gradient-to-r from-purple-600 to-pink-600 text-white">
              <div className="flex items-center justify-between mb-2">
                {currentRoom.players.map((player) => (
                  <div key={player.id} className="flex items-center gap-2">
                    <Avatar className="w-8 h-8 border-2 border-white">
                      <AvatarImage src={player.avatar} />
                      <AvatarFallback>{player.name[0]}</AvatarFallback>
                    </Avatar>
                    <div>
                      <div className="text-sm font-semibold">{player.name}</div>
                      <div className="text-xs text-purple-100">{player.score} pts</div>
                    </div>
                  </div>
                ))}
              </div>
              
              <div className="flex gap-2">
                <Button
                  size="sm"
                  variant="ghost"
                  className="flex-1 text-white hover:bg-white/20 h-8 text-xs"
                  onClick={() => {
                    setReportedPlayer(currentRoom.players[1].name);
                    setReportModalOpen(true);
                  }}
                >
                  <Flag className="w-3 h-3 mr-1" />
                  Report
                </Button>
                <Button
                  size="sm"
                  variant="outline"
                  className="flex-1 bg-white text-purple-600 h-8 text-xs"
                  onClick={handleEndGame}
                >
                  End Game
                </Button>
              </div>
            </Card>

            {/* Jeopardy Board */}
            <Card className="p-3">
              <h2 className="font-bold text-center mb-3 text-sm">Select a Question</h2>
              
              <div className="overflow-x-auto -mx-3 px-3">
                <div className="min-w-[500px]">
                  {/* Categories */}
                  <div className="grid grid-cols-5 gap-1 mb-1">
                    {categories.map((cat, idx) => (
                      <div
                        key={idx}
                        className="bg-gradient-to-br from-purple-600 to-purple-700 text-white p-2 rounded text-center text-xs font-bold"
                      >
                        {cat}
                      </div>
                    ))}
                  </div>

                  {/* Questions */}
                  {[0, 1, 2, 3, 4].map((qIdx) => (
                    <div key={qIdx} className="grid grid-cols-5 gap-1 mb-1">
                      {board.map((category, cIdx) => {
                        const question = category[qIdx];
                        return (
                          <button
                            key={question.id}
                            onClick={() => handleSelectQuestion(cIdx, qIdx)}
                            disabled={question.isAnswered}
                            className={`p-3 rounded font-bold text-sm ${
                              question.isAnswered
                                ? "bg-gray-200 dark:bg-gray-800 text-gray-400"
                                : "bg-gradient-to-br from-blue-500 to-blue-600 text-white active:scale-95"
                            }`}
                          >
                            {question.isAnswered ? "✓" : `$${question.points}`}
                          </button>
                        );
                      })}
                    </div>
                  ))}
                </div>
              </div>
            </Card>

            {/* Question Modal */}
            <AnimatePresence>
              {selectedQuestion && (
                <Dialog open={!!selectedQuestion} onOpenChange={() => setSelectedQuestion(null)}>
                  <DialogContent className="max-w-[90vw]">
                    <DialogHeader>
                      <DialogTitle className="flex items-center justify-between text-sm">
                        <span className="truncate">{selectedQuestion.category}</span>
                        <Badge className="bg-gradient-to-r from-purple-600 to-pink-600 text-white ml-2">
                          ${selectedQuestion.points}
                        </Badge>
                      </DialogTitle>
                    </DialogHeader>
                    
                    <div className="space-y-3">
                      <div className="bg-gradient-to-br from-purple-50 to-pink-50 dark:from-purple-900/20 dark:to-pink-900/20 p-4 rounded-lg">
                        <p className="text-sm font-semibold text-center">
                          {selectedQuestion.question}
                        </p>
                      </div>

                      {!showAnswer ? (
                        <>
                          <Textarea
                            value={userAnswer}
                            onChange={(e) => setUserAnswer(e.target.value)}
                            placeholder="Your answer..."
                            className="min-h-[80px] text-sm"
                          />
                          
                          <Button
                            onClick={handleSubmitAnswer}
                            disabled={!userAnswer.trim()}
                            className="w-full bg-gradient-to-r from-purple-600 to-pink-600"
                            size="sm"
                          >
                            Submit
                          </Button>
                        </>
                      ) : (
                        <Alert className="bg-green-50 dark:bg-green-900/20 border-green-200">
                          <AlertDescription className="text-sm">
                            <div className="font-semibold mb-1">Correct Answer:</div>
                            <div>{selectedQuestion.answer}</div>
                          </AlertDescription>
                        </Alert>
                      )}
                    </div>
                  </DialogContent>
                </Dialog>
              )}
            </AnimatePresence>
          </>
        )}

        {gameState === "results" && currentRoom && (
          <motion.div
            initial={{ opacity: 0, scale: 0.9 }}
            animate={{ opacity: 1, scale: 1 }}
          >
            <Card className="p-6 text-center">
              <div className="inline-flex items-center justify-center w-16 h-16 bg-gradient-to-br from-yellow-400 to-yellow-600 rounded-full mb-4">
                <Trophy className="w-8 h-8 text-white" />
              </div>
              <h2 className="text-2xl font-bold mb-2">Game Over!</h2>
              <p className="text-sm text-muted-foreground mb-4">Great game!</p>

              <div className="space-y-3 mb-6">
                {currentRoom.players
                  .sort((a, b) => b.score - a.score)
                  .map((player, idx) => (
                    <div
                      key={player.id}
                      className={`flex items-center justify-between p-3 rounded-lg ${
                        idx === 0
                          ? "bg-gradient-to-r from-yellow-50 to-yellow-100 dark:from-yellow-900/20 dark:to-yellow-800/20 border-2 border-yellow-400"
                          : "bg-gray-50 dark:bg-gray-800"
                      }`}
                    >
                      <div className="flex items-center gap-2">
                        {idx === 0 && <Crown className="w-5 h-5 text-yellow-600" />}
                        <Avatar className="w-10 h-10">
                          <AvatarImage src={player.avatar} />
                          <AvatarFallback>{player.name[0]}</AvatarFallback>
                        </Avatar>
                        <div className="text-left">
                          <div className="font-semibold text-sm">{player.name}</div>
                          <div className="text-xs text-muted-foreground">
                            {idx === 0 ? "Winner!" : "2nd Place"}
                          </div>
                        </div>
                      </div>
                      <div className="text-xl font-bold text-purple-600">
                        {player.score}
                      </div>
                    </div>
                  ))}
              </div>

              <div className="flex gap-3">
                <Button
                  onClick={handleNewGame}
                  className="flex-1 bg-gradient-to-r from-purple-600 to-pink-600"
                  size="sm"
                >
                  <Play className="w-3 h-3 mr-1" />
                  Play Again
                </Button>
                <Button 
                  variant="outline" 
                  onClick={() => setGameState("lobby")}
                  className="flex-1"
                  size="sm"
                >
                  Lobby
                </Button>
              </div>
            </Card>
          </motion.div>
        )}
      </div>

      <ReportModal
        isOpen={reportModalOpen}
        onClose={() => setReportModalOpen(false)}
        reportedPlayer={reportedPlayer}
        gameId={currentRoom?.id || ""}
      />
    </div>
  );
}
