import React, { useState } from 'react';
import { Plus, Search, Users, Calendar, MapPin, Clock, Star, Heart, MessageCircle } from 'lucide-react';
import { Button } from '../ui/button';
import { Card } from '../ui/card';
import { Avatar, AvatarImage, AvatarFallback } from '../ui/avatar';
import { Badge } from '../ui/badge';
import { Input } from '../ui/input';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '../ui/tabs';
import { useTranslation } from '../TranslationProvider';
import { useUser } from '../UserProvider';

interface MobileGroupsPageProps {
  onNavigate: (view: string) => void;
  onOpenCookieConsent: () => void;
}

export function MobileGroupsPage({ onNavigate, onOpenCookieConsent }: MobileGroupsPageProps) {
  const { t } = useTranslation();
  const { user } = useUser();
  const [searchQuery, setSearchQuery] = useState('');
  const [selectedTab, setSelectedTab] = useState('discover');

  // Mock groups data
  const groups = [
    {
      id: 1,
      name: 'Coffee Lovers NYC',
      description: 'For those who love great coffee and good conversation in the city that never sleeps.',
      category: 'lifestyle',
      members: 1247,
      image: 'https://images.unsplash.com/photo-1495474472287-4d71bcdd2085?w=400',
      isJoined: false,
      isPrivate: false,
      lastActivity: '2 hours ago',
      organizer: {
        name: 'Emma',
        photo: 'https://images.unsplash.com/photo-1494790108755-2616b612b786?w=100'
      }
    },
    {
      id: 2,
      name: 'Hiking Enthusiasts',
      description: 'Join us for weekend hiking adventures and explore beautiful trails together.',
      category: 'outdoor',
      members: 892,
      image: 'https://images.unsplash.com/photo-1551632811-561732d1e306?w=400',
      isJoined: true,
      isPrivate: false,
      lastActivity: '1 day ago',
      organizer: {
        name: 'Isabella',
        photo: 'https://images.unsplash.com/photo-1524504388940-b1c1722653e1?w=100'
      }
    },
    {
      id: 3,
      name: 'Art & Culture Society',
      description: 'Discover art galleries, museums, and cultural events in our city.',
      category: 'culture',
      members: 634,
      image: 'https://images.unsplash.com/photo-1541961017774-22349e4a1262?w=400',
      isJoined: false,
      isPrivate: true,
      lastActivity: '3 days ago',
      organizer: {
        name: 'Sofia',
        photo: 'https://images.unsplash.com/photo-1534528741775-53994a69daeb?w=100'
      }
    }
  ];

  const events = [
    {
      id: 1,
      title: 'Coffee Tasting Evening',
      group: 'Coffee Lovers NYC',
      date: 'Sat, Dec 14',
      time: '7:00 PM',
      location: 'Blue Bottle Coffee, Manhattan',
      attendees: 24,
      maxAttendees: 30,
      image: 'https://images.unsplash.com/photo-1495474472287-4d71bcdd2085?w=400',
      isGoing: false
    },
    {
      id: 2,
      title: 'Central Park Hike',
      group: 'Hiking Enthusiasts',
      date: 'Sun, Dec 15',
      time: '9:00 AM',
      location: 'Central Park, North Meadow',
      attendees: 18,
      maxAttendees: 25,
      image: 'https://images.unsplash.com/photo-1551632811-561732d1e306?w=400',
      isGoing: true
    },
    {
      id: 3,
      title: 'Museum Gallery Tour',
      group: 'Art & Culture Society',
      date: 'Fri, Dec 20',
      time: '6:00 PM',
      location: 'MoMA, Manhattan',
      attendees: 12,
      maxAttendees: 15,
      image: 'https://images.unsplash.com/photo-1541961017774-22349e4a1262?w=400',
      isGoing: false
    }
  ];

  const myGroups = groups.filter(group => group.isJoined);

  return (
    <div className="min-h-screen bg-background">
      {/* Mobile Header */}
      <div className="sticky top-0 z-10 bg-background/95 backdrop-blur-sm border-b border-border">
        <div className="p-4">
          <div className="flex items-center justify-between mb-4">
            <h1 className="text-xl font-bold">Groups & Events</h1>
            <Button size="sm" className="px-4">
              <Plus className="h-4 w-4 mr-1" />
              Create
            </Button>
          </div>

          {/* Search Bar */}
          <div className="relative mb-4">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-muted-foreground" />
            <Input
              placeholder="Search groups and events..."
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              className="pl-10"
            />
          </div>

          {/* Tabs */}
          <Tabs value={selectedTab} onValueChange={setSelectedTab} className="w-full">
            <TabsList className="grid w-full grid-cols-3">
              <TabsTrigger value="discover">Discover</TabsTrigger>
              <TabsTrigger value="my-groups">
                My Groups ({myGroups.length})
              </TabsTrigger>
              <TabsTrigger value="events">Events</TabsTrigger>
            </TabsList>
          </Tabs>
        </div>
      </div>

      {/* Content */}
      <div className="p-4 pb-20">
        <Tabs value={selectedTab} className="w-full">
          {/* Discover Groups Tab */}
          <TabsContent value="discover" className="mt-0">
            <div className="space-y-4">
              {groups.map((group) => (
                <Card key={group.id} className="overflow-hidden">
                  <div className="aspect-[2/1] relative">
                    <img 
                      src={group.image} 
                      alt={group.name}
                      className="w-full h-full object-cover"
                    />
                    <div className="absolute inset-0 bg-gradient-to-t from-black/60 to-transparent" />
                    
                    {group.isPrivate && (
                      <Badge className="absolute top-3 left-3 bg-yellow-500 text-black">
                        Private
                      </Badge>
                    )}

                    <div className="absolute bottom-3 left-3 right-3 text-white">
                      <h3 className="font-bold text-lg mb-1">{group.name}</h3>
                      <div className="flex items-center gap-2 text-sm">
                        <div className="flex items-center gap-1">
                          <Users className="h-3 w-3" />
                          <span>{group.members.toLocaleString()} members</span>
                        </div>
                        <span>•</span>
                        <span className="capitalize">{group.category}</span>
                      </div>
                    </div>
                  </div>

                  <div className="p-4">
                    <p className="text-sm text-muted-foreground mb-3">
                      {group.description}
                    </p>

                    <div className="flex items-center justify-between mb-3">
                      <div className="flex items-center gap-2">
                        <Avatar className="h-6 w-6">
                          <AvatarImage src={group.organizer.photo} />
                          <AvatarFallback>{group.organizer.name[0]}</AvatarFallback>
                        </Avatar>
                        <span className="text-sm text-muted-foreground">
                          Organized by {group.organizer.name}
                        </span>
                      </div>
                      <span className="text-xs text-muted-foreground">
                        {group.lastActivity}
                      </span>
                    </div>

                    <div className="flex gap-2">
                      <Button 
                        size="sm" 
                        className="flex-1"
                        variant={group.isJoined ? "outline" : "default"}
                      >
                        {group.isJoined ? 'Joined' : 'Join Group'}
                      </Button>
                      <Button size="sm" variant="outline">
                        View Details
                      </Button>
                    </div>
                  </div>
                </Card>
              ))}
            </div>
          </TabsContent>

          {/* My Groups Tab */}
          <TabsContent value="my-groups" className="mt-0">
            {myGroups.length === 0 ? (
              <div className="text-center py-12">
                <Users className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
                <h3 className="text-lg font-semibold mb-2">No groups joined yet</h3>
                <p className="text-muted-foreground mb-6">
                  Discover and join groups that match your interests!
                </p>
                <Button onClick={() => setSelectedTab('discover')}>
                  Discover Groups
                </Button>
              </div>
            ) : (
              <div className="space-y-4">
                {myGroups.map((group) => (
                  <Card key={group.id} className="p-4">
                    <div className="flex items-center gap-4">
                      <img 
                        src={group.image} 
                        alt={group.name}
                        className="w-16 h-16 rounded-lg object-cover"
                      />
                      
                      <div className="flex-1 min-w-0">
                        <h3 className="font-semibold truncate">{group.name}</h3>
                        <p className="text-sm text-muted-foreground mb-1 line-clamp-2">
                          {group.description}
                        </p>
                        <div className="flex items-center gap-2 text-xs text-muted-foreground">
                          <Users className="h-3 w-3" />
                          <span>{group.members.toLocaleString()} members</span>
                          <span>•</span>
                          <span>{group.lastActivity}</span>
                        </div>
                      </div>
                    </div>

                    <div className="flex gap-2 mt-3">
                      <Button size="sm" variant="outline" className="flex-1">
                        View Group
                      </Button>
                      <Button size="sm" variant="outline" className="flex-1">
                        Leave Group
                      </Button>
                    </div>
                  </Card>
                ))}
              </div>
            )}
          </TabsContent>

          {/* Events Tab */}
          <TabsContent value="events" className="mt-0">
            <div className="space-y-4">
              {events.map((event) => (
                <Card key={event.id} className="overflow-hidden">
                  <div className="aspect-[3/2] relative">
                    <img 
                      src={event.image} 
                      alt={event.title}
                      className="w-full h-full object-cover"
                    />
                    <div className="absolute inset-0 bg-gradient-to-t from-black/60 to-transparent" />
                    
                    <div className="absolute bottom-3 left-3 right-3 text-white">
                      <Badge variant="secondary" className="mb-2 bg-white/20 text-white">
                        {event.group}
                      </Badge>
                      <h3 className="font-bold text-lg mb-1">{event.title}</h3>
                      <div className="flex items-center gap-2 text-sm">
                        <Calendar className="h-3 w-3" />
                        <span>{event.date} at {event.time}</span>
                      </div>
                    </div>
                  </div>

                  <div className="p-4">
                    <div className="flex items-center gap-1 text-sm text-muted-foreground mb-3">
                      <MapPin className="h-3 w-3" />
                      <span>{event.location}</span>
                    </div>

                    <div className="flex items-center justify-between mb-3">
                      <div className="flex items-center gap-2">
                        <Users className="h-4 w-4 text-muted-foreground" />
                        <span className="text-sm">
                          {event.attendees}/{event.maxAttendees} going
                        </span>
                      </div>
                      
                      <div className="w-20 bg-muted rounded-full h-2">
                        <div 
                          className="bg-primary h-2 rounded-full"
                          style={{ width: `${(event.attendees / event.maxAttendees) * 100}%` }}
                        />
                      </div>
                    </div>

                    <div className="flex gap-2">
                      <Button 
                        size="sm" 
                        className="flex-1"
                        variant={event.isGoing ? "outline" : "default"}
                      >
                        {event.isGoing ? 'Going' : 'Join Event'}
                      </Button>
                      <Button size="sm" variant="outline">
                        Details
                      </Button>
                    </div>
                  </div>
                </Card>
              ))}
            </div>
          </TabsContent>
        </Tabs>
      </div>

      {/* Bottom Navigation */}
      <div className="fixed bottom-0 left-0 right-0 bg-background/95 backdrop-blur-sm border-t border-border">
        <div className="flex items-center justify-around py-2">
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('search')}
          >
            <Search className="h-4 w-4" />
            <span className="text-xs">Search</span>
          </Button>
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('discover')}
          >
            <Heart className="h-4 w-4" />
            <span className="text-xs">Discover</span>
          </Button>
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('matches')}
          >
            <Heart className="h-4 w-4" />
            <span className="text-xs">Matches</span>
          </Button>
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('messages')}
          >
            <MessageCircle className="h-4 w-4" />
            <span className="text-xs">Messages</span>
          </Button>
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2 text-primary"
          >
            <Users className="h-4 w-4" />
            <span className="text-xs">Groups</span>
          </Button>
        </div>
      </div>
    </div>
  );
}