import React, { useState } from 'react';
import { Flame, Heart, X, Star, RotateCcw, Info, ArrowLeft } from 'lucide-react';
import { Button } from '../ui/button';
import { Card } from '../ui/card';
import { Badge } from '../ui/badge';
import { Progress } from '../ui/progress';
import { PhotoCollage } from '../PhotoCollage';
import { useTranslation } from '../TranslationProvider';
import { useUser } from '../UserProvider';
import { motion, AnimatePresence } from 'motion/react';

interface MobileHotOrNotPageProps {
  onNavigate: (view: string) => void;
  onOpenCookieConsent: () => void;
}

export function MobileHotOrNotPage({ onNavigate, onOpenCookieConsent }: MobileHotOrNotPageProps) {
  const { t } = useTranslation();
  const { user } = useUser();
  const [currentIndex, setCurrentIndex] = useState(0);
  const [votes, setVotes] = useState(0);
  const [streak, setStreak] = useState(0);
  const [isVoting, setIsVoting] = useState(false);

  // Mock profiles for Hot or Not
  const profiles = [
    {
      id: 1,
      name: 'Emma',
      age: 28,
      photos: [
        'https://images.unsplash.com/photo-1494790108755-2616b612b786?w=600',
        'https://images.unsplash.com/photo-1517841905240-472988babdf9?w=600'
      ],
      location: 'New York',
      profession: 'Photographer',
      bio: 'Love capturing beautiful moments ✨',
      rating: 0, // Will be set after voting
      isNew: true
    },
    {
      id: 2,
      name: 'Sofia',
      age: 25,
      photos: [
        'https://images.unsplash.com/photo-1534528741775-53994a69daeb?w=600',
        'https://images.unsplash.com/photo-1506863530036-1efeddceb993?w=600'
      ],
      location: 'Los Angeles',
      profession: 'Fitness Trainer',
      bio: 'Health and wellness enthusiast 💪',
      rating: 0,
      isNew: false
    },
    {
      id: 3,
      name: 'Isabella',
      age: 30,
      photos: [
        'https://images.unsplash.com/photo-1524504388940-b1c1722653e1?w=600',
        'https://images.unsplash.com/photo-1539571696357-5a69c17a67c6?w=600'
      ],
      location: 'Miami',
      profession: 'Artist',
      bio: 'Creating beautiful art every day 🎨',
      rating: 0,
      isNew: true
    }
  ];

  const currentProfile = profiles[currentIndex];

  const handleVote = (rating: 'hot' | 'not', value?: number) => {
    if (isVoting) return;
    
    setIsVoting(true);
    setVotes(votes + 1);
    
    if (rating === 'hot') {
      setStreak(streak + 1);
    } else {
      setStreak(0);
    }

    // Update profile rating
    profiles[currentIndex].rating = value || (rating === 'hot' ? 5 : 1);
    
    // Move to next profile after animation
    setTimeout(() => {
      if (currentIndex < profiles.length - 1) {
        setCurrentIndex(currentIndex + 1);
      } else {
        setCurrentIndex(0); // Loop back to beginning
      }
      setIsVoting(false);
    }, 500);
  };

  const handleSkip = () => {
    if (currentIndex < profiles.length - 1) {
      setCurrentIndex(currentIndex + 1);
    } else {
      setCurrentIndex(0);
    }
  };

  return (
    <div className="min-h-screen bg-background relative overflow-hidden">
      {/* Header */}
      <div className="absolute top-0 left-0 right-0 z-10 p-4 bg-gradient-to-b from-black/50 to-transparent">
        <div className="flex items-center justify-between">
          <Button 
            variant="ghost" 
            size="sm" 
            className="text-white p-2"
            onClick={() => onNavigate('search')}
          >
            <ArrowLeft className="h-5 w-5" />
          </Button>
          
          <div className="flex items-center gap-3 text-white">
            <div className="flex items-center gap-1 bg-black/30 px-3 py-1 rounded-full">
              <Flame className="h-4 w-4 text-orange-500" />
              <span className="text-sm font-medium">Hot or Not</span>
            </div>
          </div>
          
          <Button variant="ghost" size="sm" className="text-white p-2">
            <Info className="h-5 w-5" />
          </Button>
        </div>

        {/* Stats Bar */}
        <div className="flex items-center justify-between mt-4 px-2">
          <div className="text-white text-sm">
            <span className="font-medium">{votes}</span> votes
          </div>
          <div className="text-white text-sm">
            <span className="text-orange-500 font-medium">{streak}</span> streak
          </div>
          <div className="text-white text-sm">
            {currentIndex + 1}/{profiles.length}
          </div>
        </div>

        {/* Progress Bar */}
        <div className="mt-2">
          <Progress 
            value={((currentIndex + 1) / profiles.length) * 100} 
            className="h-1 bg-white/20"
          />
        </div>
      </div>

      {/* Profile Card */}
      <div className="h-screen relative">
        <AnimatePresence mode="wait">
          <motion.div
            key={currentProfile.id}
            initial={{ opacity: 0, scale: 0.8, rotateY: -90 }}
            animate={{ opacity: 1, scale: 1, rotateY: 0 }}
            exit={{ 
              opacity: 0, 
              scale: 0.8, 
              rotateY: 90,
              transition: { duration: 0.3 }
            }}
            transition={{ duration: 0.4, ease: "easeOut" }}
            className="absolute inset-0"
          >
            <Card className="h-full w-full overflow-hidden border-0 rounded-none">
              <div className="relative h-full">
                <PhotoCollage
                  photos={currentProfile.photos}
                  className="h-full w-full object-cover"
                  priority={true}
                />
                
                {/* Gradient Overlay */}
                <div className="absolute inset-0 bg-gradient-to-t from-black/80 via-transparent to-transparent" />
                
                {/* New Badge */}
                {currentProfile.isNew && (
                  <div className="absolute top-20 right-4">
                    <Badge className="bg-green-500 text-white">
                      New
                    </Badge>
                  </div>
                )}

                {/* Profile Info */}
                <div className="absolute bottom-32 left-0 right-0 p-6 text-white">
                  <div className="mb-4">
                    <h2 className="text-4xl font-bold mb-2">
                      {currentProfile.name}, {currentProfile.age}
                    </h2>
                    <p className="text-xl mb-2">{currentProfile.profession}</p>
                    <p className="text-lg opacity-90">{currentProfile.location}</p>
                  </div>
                  
                  <p className="text-lg leading-relaxed opacity-95">
                    {currentProfile.bio}
                  </p>
                </div>
              </div>
            </Card>
          </motion.div>
        </AnimatePresence>
      </div>

      {/* Action Buttons */}
      <div className="absolute bottom-8 left-0 right-0 px-8">
        <div className="flex items-center justify-center gap-6">
          {/* Not Hot Button */}
          <motion.div
            whileTap={{ scale: 0.9 }}
            transition={{ duration: 0.1 }}
          >
            <Button
              variant="outline"
              size="lg"
              className="w-16 h-16 rounded-full border-2 border-red-500 bg-white text-red-500 hover:bg-red-50 shadow-lg"
              onClick={() => handleVote('not')}
              disabled={isVoting}
            >
              <X className="h-7 w-7" />
            </Button>
          </motion.div>

          {/* Skip Button */}
          <motion.div
            whileTap={{ scale: 0.9 }}
            transition={{ duration: 0.1 }}
          >
            <Button
              variant="outline"
              size="lg"
              className="w-12 h-12 rounded-full border-2 border-gray-400 bg-white text-gray-500 hover:bg-gray-50 shadow-lg"
              onClick={handleSkip}
              disabled={isVoting}
            >
              <RotateCcw className="h-5 w-5" />
            </Button>
          </motion.div>

          {/* Hot Button */}
          <motion.div
            whileTap={{ scale: 0.9 }}
            transition={{ duration: 0.1 }}
          >
            <Button
              variant="outline"
              size="lg"
              className="w-16 h-16 rounded-full border-2 border-orange-500 bg-white text-orange-500 hover:bg-orange-50 shadow-lg"
              onClick={() => handleVote('hot')}
              disabled={isVoting}
            >
              <Flame className="h-7 w-7" />
            </Button>
          </motion.div>
        </div>

        {/* Rating Scale */}
        <div className="mt-6 px-4">
          <div className="text-center mb-3">
            <p className="text-white text-sm font-medium">Rate 1-10</p>
          </div>
          <div className="flex items-center justify-between">
            {[1, 2, 3, 4, 5, 6, 7, 8, 9, 10].map((rating) => (
              <motion.button
                key={rating}
                whileTap={{ scale: 0.9 }}
                onClick={() => handleVote('hot', rating)}
                disabled={isVoting}
                className={`w-8 h-8 rounded-full text-sm font-medium transition-all ${
                  rating <= 5 
                    ? 'bg-red-500/80 text-white hover:bg-red-500' 
                    : 'bg-orange-500/80 text-white hover:bg-orange-500'
                }`}
              >
                {rating}
              </motion.button>
            ))}
          </div>
        </div>

        {/* Streak Bonus */}
        {streak >= 3 && (
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            className="mt-4 text-center"
          >
            <div className="inline-flex items-center gap-2 bg-orange-500 text-white px-4 py-2 rounded-full">
              <Star className="h-4 w-4" />
              <span className="text-sm font-medium">
                {streak} Hot Streak! 🔥
              </span>
            </div>
          </motion.div>
        )}
      </div>

      {/* Voting Animation Overlay */}
      <AnimatePresence>
        {isVoting && (
          <motion.div
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            exit={{ opacity: 0 }}
            className="absolute inset-0 z-20 flex items-center justify-center bg-black/20"
          >
            <motion.div
              initial={{ scale: 0 }}
              animate={{ scale: 1 }}
              exit={{ scale: 0 }}
              className="bg-white rounded-full p-4 shadow-lg"
            >
              <Heart className="h-8 w-8 text-primary" />
            </motion.div>
          </motion.div>
        )}
      </AnimatePresence>
    </div>
  );
}