import React, { useState, useRef, useEffect } from 'react';
import { Send, Smile, Paperclip, Phone, Video, MoreVertical, ArrowLeft, Heart, Image } from 'lucide-react';
import { Button } from '../ui/button';
import { Input } from '../ui/input';
import { Card } from '../ui/card';
import { Avatar, AvatarImage, AvatarFallback } from '../ui/avatar';
import { Badge } from '../ui/badge';
import { Sheet, SheetContent, SheetTrigger } from '../ui/sheet';
import { useTranslation } from '../TranslationProvider';
import { useUser } from '../UserProvider';
import { motion, AnimatePresence } from 'motion/react';

interface MobileMessagesPageProps {
  onNavigate: (view: string) => void;
  onOpenCookieConsent: () => void;
}

export function MobileMessagesPage({ onNavigate, onOpenCookieConsent }: MobileMessagesPageProps) {
  const { t } = useTranslation();
  const { user } = useUser();
  const [selectedChat, setSelectedChat] = useState<number | null>(null);
  const [newMessage, setNewMessage] = useState('');
  const messagesEndRef = useRef<HTMLDivElement>(null);

  // Mock conversations data
  const conversations = [
    {
      id: 1,
      name: 'Emma',
      photo: 'https://images.unsplash.com/photo-1494790108755-2616b612b786?w=400',
      lastMessage: "Hey! Thanks for the like 😊",
      timestamp: '2m ago',
      unread: 3,
      isOnline: true,
      messages: [
        { id: 1, sender: 'them', text: "Hi there! 👋", timestamp: '10:30 AM' },
        { id: 2, sender: 'me', text: "Hey! How are you doing?", timestamp: '10:32 AM' },
        { id: 3, sender: 'them', text: "I'm doing great! Thanks for the like 😊", timestamp: '10:35 AM' },
        { id: 4, sender: 'them', text: "I saw you're into photography too!", timestamp: '10:35 AM' },
        { id: 5, sender: 'me', text: "Yes! I love taking photos during my travels", timestamp: '10:40 AM' },
        { id: 6, sender: 'them', text: "That's amazing! I'd love to see some of your work", timestamp: '10:42 AM' }
      ]
    },
    {
      id: 2,
      name: 'Sofia',
      photo: 'https://images.unsplash.com/photo-1534528741775-53994a69daeb?w=400',
      lastMessage: "Would love to chat more!",
      timestamp: '1h ago',
      unread: 0,
      isOnline: false,
      messages: [
        { id: 1, sender: 'them', text: "Hi! Nice to match with you", timestamp: '9:15 AM' },
        { id: 2, sender: 'me', text: "Hey Sofia! Nice to meet you", timestamp: '9:20 AM' },
        { id: 3, sender: 'them', text: "Would love to chat more!", timestamp: '9:25 AM' }
      ]
    },
    {
      id: 3,
      name: 'Isabella',
      photo: 'https://images.unsplash.com/photo-1524504388940-b1c1722653e1?w=400',
      lastMessage: "Coffee sounds great!",
      timestamp: '3h ago',
      unread: 1,
      isOnline: true,
      messages: [
        { id: 1, sender: 'me', text: "Would you like to grab coffee sometime?", timestamp: '2:00 PM' },
        { id: 2, sender: 'them', text: "Coffee sounds great! When are you free?", timestamp: '2:15 PM' }
      ]
    }
  ];

  const selectedConversation = conversations.find(c => c.id === selectedChat);

  const scrollToBottom = () => {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
  };

  useEffect(() => {
    scrollToBottom();
  }, [selectedConversation?.messages]);

  const handleSendMessage = () => {
    if (newMessage.trim()) {
      // Handle sending message
      setNewMessage('');
    }
  };

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      handleSendMessage();
    }
  };

  if (selectedChat) {
    return (
      <div className="min-h-screen bg-background flex flex-col">
        {/* Chat Header */}
        <div className="sticky top-0 z-10 bg-background/95 backdrop-blur-sm border-b border-border">
          <div className="flex items-center justify-between p-4">
            <div className="flex items-center gap-3">
              <Button 
                variant="ghost" 
                size="sm" 
                className="p-2"
                onClick={() => setSelectedChat(null)}
              >
                <ArrowLeft className="h-5 w-5" />
              </Button>
              
              <div className="relative">
                <Avatar className="h-10 w-10">
                  <AvatarImage src={selectedConversation?.photo} />
                  <AvatarFallback>{selectedConversation?.name[0]}</AvatarFallback>
                </Avatar>
                {selectedConversation?.isOnline && (
                  <div className="absolute -bottom-1 -right-1 w-3 h-3 bg-green-500 border-2 border-white rounded-full" />
                )}
              </div>
              
              <div>
                <h2 className="font-semibold">{selectedConversation?.name}</h2>
                <p className="text-xs text-muted-foreground">
                  {selectedConversation?.isOnline ? 'Online' : 'Last seen recently'}
                </p>
              </div>
            </div>

            <div className="flex gap-2">
              <Button variant="ghost" size="sm" className="p-2">
                <Phone className="h-5 w-5" />
              </Button>
              <Button variant="ghost" size="sm" className="p-2">
                <Video className="h-5 w-5" />
              </Button>
              <Button variant="ghost" size="sm" className="p-2">
                <MoreVertical className="h-5 w-5" />
              </Button>
            </div>
          </div>
        </div>

        {/* Messages */}
        <div className="flex-1 overflow-y-auto p-4 space-y-4">
          {selectedConversation?.messages.map((message) => (
            <motion.div
              key={message.id}
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.3 }}
              className={`flex ${message.sender === 'me' ? 'justify-end' : 'justify-start'}`}
            >
              <div className={`max-w-[80%] ${message.sender === 'me' ? 'order-2' : ''}`}>
                <div
                  className={`p-3 rounded-2xl ${
                    message.sender === 'me'
                      ? 'bg-primary text-primary-foreground'
                      : 'bg-muted'
                  }`}
                >
                  <p className="text-sm">{message.text}</p>
                </div>
                <p className={`text-xs text-muted-foreground mt-1 ${message.sender === 'me' ? 'text-right' : ''}`}>
                  {message.timestamp}
                </p>
              </div>
            </motion.div>
          ))}
          <div ref={messagesEndRef} />
        </div>

        {/* Message Input */}
        <div className="sticky bottom-0 bg-background/95 backdrop-blur-sm border-t border-border p-4">
          <div className="flex items-end gap-3">
            <Button variant="ghost" size="sm" className="p-2">
              <Paperclip className="h-5 w-5" />
            </Button>
            
            <div className="flex-1 relative">
              <Input
                placeholder="Type a message..."
                value={newMessage}
                onChange={(e) => setNewMessage(e.target.value)}
                onKeyPress={handleKeyPress}
                className="pr-12 min-h-[44px] resize-none"
              />
              <Button 
                variant="ghost" 
                size="sm" 
                className="absolute right-1 top-1/2 transform -translate-y-1/2 p-2"
              >
                <Smile className="h-4 w-4" />
              </Button>
            </div>

            <Button 
              size="sm" 
              className="p-3"
              onClick={handleSendMessage}
              disabled={!newMessage.trim()}
            >
              <Send className="h-4 w-4" />
            </Button>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-background">
      {/* Mobile Header */}
      <div className="sticky top-0 z-10 bg-background/95 backdrop-blur-sm border-b border-border">
        <div className="p-4">
          <div className="flex items-center justify-between">
            <h1 className="text-xl font-bold">{t('messages')}</h1>
            <Button variant="ghost" size="sm" className="p-2">
              <MoreVertical className="h-5 w-5" />
            </Button>
          </div>
        </div>
      </div>

      {/* Conversations List */}
      <div className="p-4 pb-20">
        {conversations.length === 0 ? (
          <div className="text-center py-12">
            <Heart className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
            <h3 className="text-lg font-semibold mb-2">No messages yet</h3>
            <p className="text-muted-foreground mb-6">
              Start a conversation with your matches!
            </p>
            <Button onClick={() => onNavigate('matches')}>
              View Matches
            </Button>
          </div>
        ) : (
          <div className="space-y-3">
            {conversations.map((conversation) => (
              <Card 
                key={conversation.id} 
                className="p-4 cursor-pointer hover:bg-muted/50 transition-colors"
                onClick={() => setSelectedChat(conversation.id)}
              >
                <div className="flex items-center gap-4">
                  <div className="relative">
                    <Avatar className="h-14 w-14">
                      <AvatarImage src={conversation.photo} />
                      <AvatarFallback>{conversation.name[0]}</AvatarFallback>
                    </Avatar>
                    {conversation.isOnline && (
                      <div className="absolute -bottom-1 -right-1 w-4 h-4 bg-green-500 border-2 border-white rounded-full" />
                    )}
                  </div>

                  <div className="flex-1 min-w-0">
                    <div className="flex items-center justify-between mb-1">
                      <h3 className="font-semibold truncate">{conversation.name}</h3>
                      <div className="flex items-center gap-2">
                        <span className="text-xs text-muted-foreground">
                          {conversation.timestamp}
                        </span>
                        {conversation.unread > 0 && (
                          <Badge className="min-w-[20px] h-5 flex items-center justify-center text-xs px-1">
                            {conversation.unread}
                          </Badge>
                        )}
                      </div>
                    </div>
                    
                    <p className={`text-sm truncate ${
                      conversation.unread > 0 ? 'font-medium text-foreground' : 'text-muted-foreground'
                    }`}>
                      {conversation.lastMessage}
                    </p>
                  </div>
                </div>
              </Card>
            ))}
          </div>
        )}
      </div>

      {/* Bottom Navigation */}
      <div className="fixed bottom-0 left-0 right-0 bg-background/95 backdrop-blur-sm border-t border-border">
        <div className="flex items-center justify-around py-2">
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('search')}
          >
            <div className="h-4 w-4" />
            <span className="text-xs">Search</span>
          </Button>
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('discover')}
          >
            <Heart className="h-4 w-4" />
            <span className="text-xs">Discover</span>
          </Button>
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('matches')}
          >
            <Heart className="h-4 w-4" />
            <span className="text-xs">Matches</span>
          </Button>
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2 text-primary"
          >
            <div className="h-4 w-4" />
            <span className="text-xs">Messages</span>
          </Button>
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('profile-settings')}
          >
            <Avatar className="h-4 w-4">
              <AvatarImage src={user?.profilePicture} />
              <AvatarFallback>{user?.firstName?.[0]}</AvatarFallback>
            </Avatar>
            <span className="text-xs">Profile</span>
          </Button>
        </div>
      </div>
    </div>
  );
}