import React, { useState } from 'react';
import { ArrowLeft, Camera, Settings, Shield, Bell, Heart, User, Palette, CreditCard, HelpCircle, LogOut } from 'lucide-react';
import { Button } from '../ui/button';
import { Card } from '../ui/card';
import { Avatar, AvatarImage, AvatarFallback } from '../ui/avatar';
import { Badge } from '../ui/badge';
import { Separator } from '../ui/separator';
import { Switch } from '../ui/switch';
import { useTranslation } from '../TranslationProvider';
import { useUser } from '../UserProvider';

interface MobileProfileSettingsPageProps {
  onNavigate: (view: string) => void;
  onOpenCookieConsent: () => void;
}

export function MobileProfileSettingsPage({ onNavigate, onOpenCookieConsent }: MobileProfileSettingsPageProps) {
  const { t } = useTranslation();
  const { user } = useUser();
  const [notifications, setNotifications] = useState(true);
  const [showOnline, setShowOnline] = useState(true);

  const menuItems = [
    {
      id: 'edit-profile',
      title: 'Edit Profile',
      icon: User,
      action: () => onNavigate('profile-settings'),
      hasArrow: true
    },
    {
      id: 'photos',
      title: 'Photos & Media',
      icon: Camera,
      action: () => onNavigate('profile-settings'),
      hasArrow: true
    },
    {
      id: 'preferences',
      title: 'Discovery Preferences',
      icon: Heart,
      action: () => onNavigate('profile-settings'),
      hasArrow: true
    },
    {
      id: 'subscription',
      title: 'Subscription',
      icon: CreditCard,
      action: () => onNavigate('subscription'),
      hasArrow: true,
      badge: user?.subscriptionTier !== 'free' ? user?.subscriptionTier : undefined
    }
  ];

  const settingsItems = [
    {
      id: 'notifications',
      title: 'Notifications',
      icon: Bell,
      control: 'switch',
      value: notifications,
      onChange: setNotifications
    },
    {
      id: 'online-status',
      title: 'Show Online Status',
      icon: Shield,
      control: 'switch',
      value: showOnline,
      onChange: setShowOnline
    },
    {
      id: 'privacy',
      title: 'Privacy Settings',
      icon: Shield,
      action: () => onNavigate('profile-settings'),
      hasArrow: true
    },
    {
      id: 'theme',
      title: 'Theme & Colors',
      icon: Palette,
      action: () => onNavigate('profile-settings'),
      hasArrow: true
    }
  ];

  const supportItems = [
    {
      id: 'help',
      title: 'Help Center',
      icon: HelpCircle,
      action: () => onNavigate('help-center'),
      hasArrow: true
    },
    {
      id: 'safety',
      title: 'Safety Tips',
      icon: Shield,
      action: () => onNavigate('safety-tips'),
      hasArrow: true
    },
    {
      id: 'guidelines',
      title: 'Community Guidelines',
      icon: Heart,
      action: () => onNavigate('community-guidelines'),
      hasArrow: true
    }
  ];

  return (
    <div className="min-h-screen bg-background">
      {/* Mobile Header */}
      <div className="sticky top-0 z-10 bg-background/95 backdrop-blur-sm border-b border-border">
        <div className="flex items-center justify-between p-4">
          <Button 
            variant="ghost" 
            size="sm" 
            className="p-2"
            onClick={() => onNavigate('search')}
          >
            <ArrowLeft className="h-5 w-5" />
          </Button>
          
          <h1 className="text-lg font-semibold">Profile & Settings</h1>
          
          <Button variant="ghost" size="sm" className="p-2">
            <Settings className="h-5 w-5" />
          </Button>
        </div>
      </div>

      {/* Profile Section */}
      <div className="p-4">
        <Card className="p-6">
          <div className="flex items-center gap-4 mb-4">
            <div className="relative">
              <Avatar className="h-20 w-20">
                <AvatarImage src={user?.profilePicture} />
                <AvatarFallback className="text-xl">{user?.firstName?.[0]}</AvatarFallback>
              </Avatar>
              <Button 
                size="sm" 
                className="absolute -bottom-2 -right-2 rounded-full p-2 h-8 w-8"
              >
                <Camera className="h-3 w-3" />
              </Button>
            </div>
            
            <div className="flex-1">
              <div className="flex items-center gap-2 mb-1">
                <h2 className="text-xl font-bold">{user?.firstName} {user?.lastName}</h2>
                {user?.isVerified && (
                  <Badge className="bg-blue-500 text-white">
                    Verified
                  </Badge>
                )}
              </div>
              <p className="text-muted-foreground">@{user?.username}</p>
              <p className="text-sm text-muted-foreground mt-1">
                {user?.location} • {user?.age} years old
              </p>
            </div>
          </div>

          {/* Profile Stats */}
          <div className="grid grid-cols-3 gap-4 mt-4">
            <div className="text-center">
              <p className="text-2xl font-bold text-primary">85</p>
              <p className="text-xs text-muted-foreground">Profile Views</p>
            </div>
            <div className="text-center">
              <p className="text-2xl font-bold text-primary">23</p>
              <p className="text-xs text-muted-foreground">Matches</p>
            </div>
            <div className="text-center">
              <p className="text-2xl font-bold text-primary">92%</p>
              <p className="text-xs text-muted-foreground">Profile Complete</p>
            </div>
          </div>
        </Card>
      </div>

      {/* Menu Sections */}
      <div className="p-4 space-y-6 pb-20">
        {/* Profile & Account */}
        <div>
          <h3 className="font-semibold mb-3 px-2">Profile & Account</h3>
          <Card className="overflow-hidden">
            {menuItems.map((item, index) => (
              <div key={item.id}>
                <button
                  onClick={item.action}
                  className="w-full flex items-center justify-between p-4 text-left hover:bg-muted/50 transition-colors"
                >
                  <div className="flex items-center gap-3">
                    <item.icon className="h-5 w-5 text-muted-foreground" />
                    <span className="font-medium">{item.title}</span>
                    {item.badge && (
                      <Badge variant="secondary" className="capitalize">
                        {item.badge}
                      </Badge>
                    )}
                  </div>
                  {item.hasArrow && (
                    <div className="h-4 w-4 text-muted-foreground">→</div>
                  )}
                </button>
                {index < menuItems.length - 1 && <Separator />}
              </div>
            ))}
          </Card>
        </div>

        {/* Settings */}
        <div>
          <h3 className="font-semibold mb-3 px-2">Settings</h3>
          <Card className="overflow-hidden">
            {settingsItems.map((item, index) => (
              <div key={item.id}>
                <div className="flex items-center justify-between p-4">
                  <div className="flex items-center gap-3">
                    <item.icon className="h-5 w-5 text-muted-foreground" />
                    <span className="font-medium">{item.title}</span>
                  </div>
                  
                  {item.control === 'switch' ? (
                    <Switch
                      checked={item.value as boolean}
                      onCheckedChange={item.onChange}
                    />
                  ) : item.hasArrow ? (
                    <button onClick={item.action} className="text-muted-foreground">
                      →
                    </button>
                  ) : null}
                </div>
                {index < settingsItems.length - 1 && <Separator />}
              </div>
            ))}
          </Card>
        </div>

        {/* Support */}
        <div>
          <h3 className="font-semibold mb-3 px-2">Support</h3>
          <Card className="overflow-hidden">
            {supportItems.map((item, index) => (
              <div key={item.id}>
                <button
                  onClick={item.action}
                  className="w-full flex items-center justify-between p-4 text-left hover:bg-muted/50 transition-colors"
                >
                  <div className="flex items-center gap-3">
                    <item.icon className="h-5 w-5 text-muted-foreground" />
                    <span className="font-medium">{item.title}</span>
                  </div>
                  {item.hasArrow && (
                    <div className="h-4 w-4 text-muted-foreground">→</div>
                  )}
                </button>
                {index < supportItems.length - 1 && <Separator />}
              </div>
            ))}
          </Card>
        </div>

        {/* Logout */}
        <Card>
          <button
            onClick={() => onNavigate('login')}
            className="w-full flex items-center gap-3 p-4 text-left hover:bg-destructive/10 transition-colors text-destructive"
          >
            <LogOut className="h-5 w-5" />
            <span className="font-medium">Sign Out</span>
          </button>
        </Card>

        {/* Version Info */}
        <div className="text-center text-xs text-muted-foreground">
          <p>SinglesZone v1.0.0</p>
          <p className="mt-1">
            <button onClick={() => onNavigate('privacy')} className="underline">
              Privacy Policy
            </button>
            {' • '}
            <button onClick={() => onNavigate('terms')} className="underline">
              Terms of Service
            </button>
          </p>
        </div>
      </div>

      {/* Bottom Navigation */}
      <div className="fixed bottom-0 left-0 right-0 bg-background/95 backdrop-blur-sm border-t border-border">
        <div className="flex items-center justify-around py-2">
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('search')}
          >
            <div className="h-4 w-4" />
            <span className="text-xs">Search</span>
          </Button>
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('discover')}
          >
            <Heart className="h-4 w-4" />
            <span className="text-xs">Discover</span>
          </Button>
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('matches')}
          >
            <Heart className="h-4 w-4" />
            <span className="text-xs">Matches</span>
          </Button>
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('messages')}
          >
            <div className="h-4 w-4" />
            <span className="text-xs">Messages</span>
          </Button>
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2 text-primary"
          >
            <User className="h-4 w-4" />
            <span className="text-xs">Profile</span>
          </Button>
        </div>
      </div>
    </div>
  );
}