# SinglesZonne Deployment Guide

## Overview
This guide covers deploying your React-based dating platform to your own server with all necessary components.

## Prerequisites
- VPS or dedicated server with root access
- Domain name pointed to your server
- Basic Linux server administration knowledge

## Step 1: Server Setup

### 1.1 Choose Your Server Stack
**Option A: Node.js + Express (Recommended)**
- Node.js 18+ LTS
- Express.js for API endpoints
- PM2 for process management

**Option B: Apache/Nginx + PHP**
- Apache or Nginx web server
- PHP 8.1+ for backend API
- MySQL/PostgreSQL database

**Option C: Static Hosting + External API**
- Nginx for static files
- External backend service (Firebase, Supabase, etc.)

### 1.2 Install Dependencies (Node.js Stack)
```bash
# Update system
sudo apt update && sudo apt upgrade -y

# Install Node.js
curl -fsSL https://deb.nodesource.com/setup_18.x | sudo -E bash -
sudo apt-get install -y nodejs

# Install PM2
sudo npm install -g pm2

# Install Nginx (reverse proxy)
sudo apt install nginx

# Install database
sudo apt install postgresql postgresql-contrib
# OR
sudo apt install mysql-server
```

## Step 2: Build Your React Application

### 2.1 Install Build Tools
```bash
# Install Vite (if not already installed)
npm install -g vite

# Or if using Create React App
npm install -g create-react-app
```

### 2.2 Create Production Build
```bash
# In your project directory
npm install
npm run build
```

This creates a `dist` or `build` folder with optimized static files.

## Step 3: Backend API Setup

### 3.1 Create Express.js Backend
Create `/server/app.js`:

```javascript
const express = require('express');
const cors = require('cors');
const path = require('path');
const app = express();

// Middleware
app.use(cors());
app.use(express.json({ limit: '50mb' }));
app.use(express.static(path.join(__dirname, '../dist')));

// API Routes
app.use('/api/auth', require('./routes/auth'));
app.use('/api/users', require('./routes/users'));
app.use('/api/matches', require('./routes/matches'));
app.use('/api/messages', require('./routes/messages'));
app.use('/api/admin', require('./routes/admin'));

// Serve React app for all other routes
app.get('*', (req, res) => {
  res.sendFile(path.join(__dirname, '../dist/index.html'));
});

const PORT = process.env.PORT || 3000;
app.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`);
});
```

### 3.2 Database Setup
```sql
-- Use your existing schema from database/singleszone_schema.sql
-- Run the SQL file to create all tables
```

## Step 4: Environment Configuration

### 4.1 Create Environment File
Create `/server/.env`:

```env
# Database
DB_HOST=localhost
DB_NAME=singleszone
DB_USER=your_db_user
DB_PASS=your_db_password

# JWT Secret
JWT_SECRET=your_super_secure_jwt_secret_key

# Google Maps API
GOOGLE_MAPS_API_KEY=your_google_maps_api_key

# ReCAPTCHA
RECAPTCHA_SECRET_KEY=your_recaptcha_secret

# Email Service (for verification)
EMAIL_HOST=smtp.your-provider.com
EMAIL_PORT=587
EMAIL_USER=your_email@domain.com
EMAIL_PASS=your_email_password

# File Upload
UPLOAD_PATH=/var/www/uploads
MAX_FILE_SIZE=10485760

# App Settings
NODE_ENV=production
FRONTEND_URL=https://yourdomain.com
```

## Step 5: Server Configuration

### 5.1 Nginx Configuration
Create `/etc/nginx/sites-available/singleszone`:

```nginx
server {
    listen 80;
    server_name yourdomain.com www.yourdomain.com;
    return 301 https://$server_name$request_uri;
}

server {
    listen 443 ssl http2;
    server_name yourdomain.com www.yourdomain.com;

    # SSL Configuration
    ssl_certificate /etc/letsencrypt/live/yourdomain.com/fullchain.pem;
    ssl_certificate_key /etc/letsencrypt/live/yourdomain.com/privkey.pem;
    ssl_protocols TLSv1.2 TLSv1.3;
    ssl_ciphers ECDHE-RSA-AES256-GCM-SHA512:DHE-RSA-AES256-GCM-SHA512;

    # Security Headers
    add_header X-Frame-Options "SAMEORIGIN" always;
    add_header X-XSS-Protection "1; mode=block" always;
    add_header X-Content-Type-Options "nosniff" always;
    add_header Referrer-Policy "no-referrer-when-downgrade" always;
    add_header Content-Security-Policy "default-src 'self' http: https: data: blob: 'unsafe-inline'" always;

    # File Upload Limits
    client_max_body_size 50M;

    # Serve static files
    location / {
        try_files $uri $uri/ @nodejs;
    }

    # Proxy API requests to Node.js
    location @nodejs {
        proxy_pass http://localhost:3000;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
        proxy_cache_bypass $http_upgrade;
    }

    # Handle uploaded files
    location /uploads {
        alias /var/www/uploads;
        expires 1y;
        add_header Cache-Control "public, immutable";
    }

    # Security
    location ~ /\. {
        deny all;
    }

    location ~* \.(js|css|png|jpg|jpeg|gif|ico|svg)$ {
        expires 1y;
        add_header Cache-Control "public, immutable";
    }
}
```

### 5.2 Enable Site
```bash
sudo ln -s /etc/nginx/sites-available/singleszone /etc/nginx/sites-enabled/
sudo nginx -t
sudo systemctl reload nginx
```

## Step 6: SSL Certificate Setup

### 6.1 Install Certbot
```bash
sudo apt install certbot python3-certbot-nginx
```

### 6.2 Obtain Certificate
```bash
sudo certbot --nginx -d yourdomain.com -d www.yourdomain.com
```

## Step 7: Process Management

### 7.1 PM2 Configuration
Create `/server/ecosystem.config.js`:

```javascript
module.exports = {
  apps: [{
    name: 'singleszone',
    script: './app.js',
    instances: 'max',
    exec_mode: 'cluster',
    env: {
      NODE_ENV: 'production',
      PORT: 3000
    },
    error_file: './logs/err.log',
    out_file: './logs/out.log',
    log_file: './logs/combined.log',
    time: true
  }]
};
```

### 7.2 Start Application
```bash
cd /path/to/your/server
pm2 start ecosystem.config.js
pm2 startup
pm2 save
```

## Step 8: File Upload Directory

### 8.1 Create Upload Directories
```bash
sudo mkdir -p /var/www/uploads/{profiles,media,documents}
sudo chown -R www-data:www-data /var/www/uploads
sudo chmod -R 755 /var/www/uploads
```

## Step 9: Database Connection

### 9.1 PostgreSQL Setup
```bash
sudo -u postgres createuser --interactive
sudo -u postgres createdb singleszone
sudo -u postgres psql -d singleszone -f /path/to/database/singleszone_schema.sql
```

### 9.2 MySQL Setup
```bash
mysql -u root -p
CREATE DATABASE singleszone;
CREATE USER 'singleszone_user'@'localhost' IDENTIFIED BY 'secure_password';
GRANT ALL PRIVILEGES ON singleszone.* TO 'singleszone_user'@'localhost';
source /path/to/database/singleszone_schema.sql;
```

## Step 10: Final Deployment Steps

### 10.1 Upload Files
```bash
# Copy your built React app
sudo cp -r /local/path/to/dist/* /var/www/html/singleszone/

# Copy server files
sudo cp -r /local/path/to/server /var/www/singleszone-api/
```

### 10.2 Install Dependencies
```bash
cd /var/www/singleszone-api
sudo npm install --production
```

### 10.3 Start Services
```bash
sudo systemctl start nginx
sudo systemctl enable nginx
pm2 start ecosystem.config.js
```

## Step 11: Monitoring and Maintenance

### 11.1 Log Monitoring
```bash
# PM2 logs
pm2 logs

# Nginx logs
sudo tail -f /var/log/nginx/access.log
sudo tail -f /var/log/nginx/error.log
```

### 11.2 Backup Script
Create `/scripts/backup.sh`:

```bash
#!/bin/bash
DATE=$(date +%Y%m%d_%H%M%S)
BACKUP_DIR="/backups"

# Database backup
pg_dump singleszone > "$BACKUP_DIR/db_backup_$DATE.sql"

# Files backup
tar -czf "$BACKUP_DIR/files_backup_$DATE.tar.gz" /var/www/uploads

# Keep only last 7 days
find $BACKUP_DIR -name "*.sql" -mtime +7 -delete
find $BACKUP_DIR -name "*.tar.gz" -mtime +7 -delete
```

## Step 12: Security Considerations

### 12.1 Firewall Setup
```bash
sudo ufw allow ssh
sudo ufw allow 'Nginx Full'
sudo ufw enable
```

### 12.2 Fail2Ban
```bash
sudo apt install fail2ban
sudo systemctl enable fail2ban
```

## Step 13: Performance Optimization

### 13.1 Nginx Caching
Add to your Nginx config:

```nginx
# Enable gzip
gzip on;
gzip_vary on;
gzip_min_length 1024;
gzip_types text/plain text/css text/xml text/javascript application/javascript application/xml+rss application/json;

# Browser caching
location ~* \.(jpg|jpeg|png|gif|ico|css|js)$ {
    expires 1y;
    add_header Cache-Control "public, immutable";
}
```

## Troubleshooting

### Common Issues:
1. **File permissions**: Ensure www-data owns web files
2. **Database connection**: Check credentials in .env
3. **SSL issues**: Verify certificate installation
4. **API errors**: Check PM2 logs for Node.js errors
5. **Upload failures**: Verify upload directory permissions

### Health Checks:
```bash
# Check services
sudo systemctl status nginx
pm2 status

# Check logs
pm2 logs singleszone
sudo tail -f /var/log/nginx/error.log
```

## Post-Deployment Checklist

- [ ] SSL certificate installed and working
- [ ] Database connected and populated
- [ ] File uploads working
- [ ] API endpoints responding
- [ ] Mobile responsiveness verified
- [ ] Google Maps integration configured
- [ ] Email verification working
- [ ] Payment processing configured (if applicable)
- [ ] Backup system in place
- [ ] Monitoring setup
- [ ] DNS properly configured

Your SinglesZonne dating platform should now be live and accessible at your domain!